# ------------------------------------------------------------------------
# BEAUTY DETR
# Copyright (c) 2022 Ayush Jain & Nikolaos Gkanatsios
# Licensed under CC-BY-NC [see LICENSE for details]
# All Rights Reserved
# ------------------------------------------------------------------------
"""A class to collect and evaluate language grounding results."""

import torch

from models.losses import _iou3d_par, box_cxcyczwhd_to_xyzxyz
from utils.eval_det import iou3d_rotated_vs_aligned
import utils.misc as misc
from collections import defaultdict
import ipdb

st = ipdb.set_trace


class GroundingEvaluator:
    """
    Evaluate language grounding.

    Args:
        only_root (bool): detect only the root noun
        thresholds (list): IoU thresholds to check
        topks (list): k to evaluate top--k accuracy
        prefixes (list): names of layers to evaluate
    """

    def __init__(self, only_root=False, thresholds=[0.25, 0.5], topks=[1, 5, 10], prefixes=[]):
        """Initialize accumulators."""
        self.only_root = only_root
        self.thresholds = thresholds
        self.topks = topks
        self.prefixes = prefixes

        self.reset()

    def reset(self):
        """Reset accumulators to empty."""
        # self.dets = {
        #     (prefix, t, k, mode): 0 for prefix in self.prefixes for t in self.thresholds for k in self.topks for mode in ["bbs", "bbf"]
        # }  # Number of hit GT boxes, e.g. accuracy at IoU 0.5 for top-1
        # self.gts = dict(self.dets)  # Total number of GT boxes

        self.dets = defaultdict(int)
        self.gts = defaultdict(int)

        self.dets.update({"vd": 0, "vid": 0})
        self.dets.update({"hard": 0, "easy": 0})
        self.dets.update({"multi": 0, "unique": 0})
        self.gts.update({"vd": 1e-14, "vid": 1e-14})
        self.gts.update({"hard": 1e-14, "easy": 1e-14})
        self.gts.update({"multi": 1e-14, "unique": 1e-14})

        # Additional total_acc statistics
        self.dets.update({("total_acc", t, "bbf"): 0 for t in self.thresholds})
        self.gts.update({("total_acc", t, "bbf"): 1e-14 for t in self.thresholds})  # Prevent division by zero

        self.prediction_records = []

    def print_stats(self):
        """Print accumulated accuracies."""
        return_str = "\n"
        mode_str = {"bbs": "Box given span (soft-token)", "bbf": "Box given span (contrastive)"}
        for prefix in ["last_", "proposal_"]:  # self.prefixes: # 我改了，但是如果报错了，可以改回来
            for mode in ["bbs", "bbf"]:
                for t in self.thresholds:
                    line = f"{prefix} {mode_str[mode]} Acc{t:.2f}: " + ", ".join(
                        [f"Top-{k}: {self.dets[(prefix, t, k, mode)] / max(self.gts[(prefix, t, k, mode)], 1):.3f}" for k in self.topks]
                    )
                    # print(line)
                    return_str += line + "\n"

        return_str += "\n==Analysis==\n"

        for t in self.thresholds:
            acc = self.dets[("total_acc", t, "bbf")] / self.gts[("total_acc", t, "bbf")]
            return_str += f"Acc@{t} = {acc:.4f}  "

        return_str += "\n\n"

        # for field in ["easy", "hard", "vd", "vid", "unique", "multi"]:
        #     # print(field, self.dets[field] / self.gts[field])
        #     value = self.dets[field] / self.gts[field]
        #     return_str += f"{field}: {value:.3f}\n"

        return return_str

    def synchronize_between_processes(self):
        all_dets = misc.all_gather(self.dets)
        all_gts = misc.all_gather(self.gts)

        if misc.is_main_process():
            merged_predictions = {}
            for key in all_dets[0].keys():
                merged_predictions[key] = 0
                for p in all_dets:
                    # 确保所有值都在 CPU 上
                    if isinstance(p[key], torch.Tensor):
                        p[key] = p[key].cpu()
                    merged_predictions[key] += p[key]
            self.dets = merged_predictions

            merged_predictions = {}
            for key in all_gts[0].keys():
                merged_predictions[key] = 0
                for p in all_gts:
                    # 确保所有值都在 CPU 上
                    if isinstance(p[key], torch.Tensor):
                        p[key] = p[key].cpu()
                    merged_predictions[key] += p[key]
            self.gts = merged_predictions

    def evaluate(self, batch_data, end_points, prefix):
        """
        Evaluate all accuracies.

        Args:
            batch_data (dict): contains original data (utterances, meta_path, etc.)
            end_points (dict): contains predictions and gt
            prefix (str): layer name
        """
        self.evaluate_bbox_by_span(batch_data, end_points, prefix)
        self.evaluate_bbox_by_contrast(batch_data, end_points, prefix)

    def evaluate_bbox_by_span(self, batch_data, end_points, prefix):
        """
        Evaluate bounding box IoU for top gt span detections.

        Args:
            batch_data (dict): contains original data (utterances, meta_path, etc.)
            end_points (dict): contains predictions and gt
            prefix (str): layer name
        """
        # Parse gt - NOW USING ROTATED GT FOR FAIR COMPARISON
        positive_map = torch.clone(end_points["positive_map"])
        positive_map[positive_map > 0] = 1
        gt_bboxes_rotated = batch_data["gt_bboxes"]  # (B, 132, 7) or (B, 132, 9) with rotation
        
        # Parse predictions
        sem_scores = end_points[f"{prefix}sem_cls_scores"].softmax(-1)  # B, num_query=256, len_token=256

        if sem_scores.shape[-1] != positive_map.shape[-1]:
            sem_scores_ = torch.zeros(sem_scores.shape[0], sem_scores.shape[1], positive_map.shape[-1]).to(sem_scores.device)
            sem_scores_[:, :, : sem_scores.shape[-1]] = sem_scores
            sem_scores = sem_scores_

        # Parse predictions
        pred_center = end_points[f"{prefix}center"]  # B, Q, 3
        pred_size = end_points[f"{prefix}pred_size"]  # (B,Q,3) (l,w,h)
        assert (pred_size < 0).sum() == 0
        pred_bbox = torch.cat([pred_center, pred_size], dim=-1)  # B, Q=256, 6, each query corresponds to a box

        # Highest scoring box -> iou
        for bid in range(len(positive_map)):
            # Keep scores for annotated objects only
            num_obj = int(end_points["box_label_mask"][bid].sum())  # 1
            pmap = positive_map[bid, :num_obj]
            scores = (sem_scores[bid].unsqueeze(0) * pmap.unsqueeze(1)).sum(-1)  # (1, Q, 256)  # (obj, 1, 256)  # (obj, Q) # Score of each query for target token

            # 10 predictions per gt box
            top = scores.argsort(1, True)[:, :10]  # (obj, 10) # Sort each GT (only 1 here) and get top 10 queries
            pbox = pred_bbox[bid, top.reshape(-1)]  #  # Query indices, sorted by score from high to low

            # IoU - NOW USING ROTATED GT FOR FAIR COMPARISON
            gt_boxes = gt_bboxes_rotated[bid][:num_obj]  # (1, 7) or (1, 9) - with rotation
            ious, _ = iou3d_rotated_vs_aligned(
                gt_boxes,  # (1, 7/9) - rotated GT bbox
                pbox       # (10, 6) - axis-aligned pred bboxes
            )  # returns (1, 10) - IoU between 1 gt and 10 predictions

            # Measure IoU>threshold, ious are (obj, 10)
            topks = self.topks  # [1, 5, 10]
            for t in self.thresholds:  # 0.25, 0.5
                thresholded = ious > t
                for k in topks:
                    found = thresholded[:, :k].any(1)  # Top-1: Check if any of first 1 has IoU > 0.5 # ious[:, :1] = [0.55] > 0.5
                    # NOTE bbs is "Box given span (soft-token)"
                    self.dets[(prefix, t, k, "bbs")] += found.sum().item()  # Number of hit GT boxes
                    self.gts[(prefix, t, k, "bbs")] += len(thresholded)  # Total number of GT boxes

    def evaluate_bbox_by_contrast(self, batch_data, end_points, prefix):
        """
        使用对比学习方法评估 bounding box IoU（通过 query 特征和 token 特征的相似度）
        
        核心思想：
        1. DETR 模型预测 256 个候选框（set prediction）
        2. 通过对比学习计算每个候选框与语言描述的匹配分数
        3. 选择分数最高的 top-k 个候选框
        4. 计算这些候选框与 gt 的 IoU，评估准确率

        Args:
            batch_data (dict): 包含原始数据（utterances, meta_path 等）
            end_points (dict): 包含模型预测结果和 gt
            prefix (str): 层名称，如 "last_" 或 "proposal_"
        """
        # ============ 1. 解析 Ground Truth ============
        # ipdb.set_trace() # TODO 如果是 batch_data["gt_bboxes"]
        # positive_map, gt_bboxes = self._parse_gt(end_points) 
        # ! 从 batch_data 获取原始 GT bboxes（保留旋转信息）
        positive_map = torch.clone(end_points["positive_map"])
        positive_map[positive_map > 0] = 1
        gt_bboxes_rotated = batch_data["gt_bboxes"]  # (B, 132, 7) 或 (B, 132, 9)
        # Waymo: [x,y,z,l,w,h,yaw]  Quad/Drone: [x,y,z,l,w,h,yaw,pitch,roll]
        
        # positive_map: (B=8, 132, 256) - 每个 gt 对象对应的 token 位置（one-hot）
        # gt_bboxes_rotated: (B=8, 132, 7/9) - gt bbox with rotation，前 num_obj 个有效
        
        # ============ 2. 解析模型预测 ============
        pred_center = end_points[f"{prefix}center"]  # (B=8, Q=256, 3) - 预测的中心点坐标
        pred_size = end_points[f"{prefix}pred_size"]  # (B=8, Q=256, 3) - 预测的尺寸 (l,w,h)
        assert (pred_size < 0).sum() == 0  # 确保尺寸为正
        pred_bbox = torch.cat([pred_center, pred_size], dim=-1)  # (B=8, Q=256, 6)
        # DETR 架构：每个样本预测 256 个候选框（queries）

        # ============ 3. 计算对比学习分数 ============
        proj_tokens = end_points["proj_tokens"]  # (B=8, tokens, 64)
        # 文本特征：语言描述的 tokens 投影到对比学习空间（64维）
        
        proj_queries = end_points[f"{prefix}proj_queries"]  # (B=8, Q=256, 64)
        # Query 特征：256 个候选框的特征投影到对比学习空间（64维）
        
        sem_scores = torch.matmul(
            proj_queries, proj_tokens.transpose(-1, -2)
        )  # (B=8, Q=256, tokens) - 每个 query 与每个 token 的相似度（点积）
        
        sem_scores_ = (sem_scores / 0.07).softmax(-1)  # (B=8, Q=256, tokens)
        # 温度系数 0.07：对比学习中的常用参数，用于调整分布的平滑度
        # softmax：归一化为概率分布
        
        # Padding 到固定维度 256
        sem_scores = torch.zeros(sem_scores_.size(0), sem_scores_.size(1), 256)  # (B=8, Q=256, 256)
        sem_scores = sem_scores.to(sem_scores_.device)
        sem_scores[:, : sem_scores_.size(1), : sem_scores_.size(2)] = sem_scores_

        # ============ 4. 逐样本评估 ============
        for bid in range(len(positive_map)):  # 遍历 batch 中的每个样本
            # 4.1 获取有效的 gt 对象数量
            num_obj = int(end_points["box_label_mask"][bid].sum())  # 当前样本有多少个 gt
            assert num_obj == 1, f"num_obj: {num_obj}. only support obj number is 1."
            # 目前只支持单目标场景（quad/drone/waymo 单目标）
            
            # 4.2 计算每个 query 与目标描述的匹配分数
            pmap = positive_map[bid, :num_obj]  # (1, 256) - 目标对应的 token 位置
            # 例如：描述 "the red car"，pmap 标记了 "red" 和 "car" 对应的 token 位置
            
            scores = (sem_scores[bid].unsqueeze(0) * pmap.unsqueeze(1)).sum(-1)  # (1, 256)
            # sem_scores[bid]: (256, 256) - 当前样本的 256 个 queries 对所有 tokens 的分数
            # .unsqueeze(0): (1, 256, 256)
            # pmap.unsqueeze(1): (1, 1, 256) - 目标 token 的 mask
            # 相乘后 sum(-1)：只保留目标 tokens 的分数，得到每个 query 的总分
            # 结果：(1, 256) - 256 个 queries 对目标描述的匹配分数

            # 4.3 选择 top-10 最高分的候选框
            top = scores.argsort(1, True)[:, :10]  # (1, 10)
            # argsort(1, True)：按分数降序排序，返回索引
            # [:, :10]：取前 10 个（分数最高的 10 个 query 索引）
            
            pbox = pred_bbox[bid, top.reshape(-1)]  # (10, 6) - [cx,cy,cz,w,h,d] axis-aligned
            # 从 256 个候选框中取出这 10 个最匹配的预测框

            # ipdb.set_trace() # TODO print gt_bboxes_rotated[bid][:num_obj]
            # 4.4 计算 IoU
            # ious, _ = _iou3d_par(
            #     box_cxcyczwhd_to_xyzxyz(gt_bboxes[bid][:num_obj]),  # (1, 6) - gt bbox
            #     box_cxcyczwhd_to_xyzxyz(pbox)  # (10, 6) - top-10 预测框
            # )  # 返回 (1, 10) - 1 个 gt 与 10 个预测框的 IoU
            
            # ! 4.4 计算 IoU（rotated GT vs axis-aligned pred）
            gt_boxes = gt_bboxes_rotated[bid][:num_obj]  # (1, 7) 或 (1, 9) - with rotation
            ious, _ = iou3d_rotated_vs_aligned(
                gt_boxes,  # (1, 7/9) - rotated GT bbox
                pbox       # (10, 6) - axis-aligned pred bboxes
            )  # 返回 (1, 10) - 1 个 gt 与 10 个预测框的 IoU
            # 因为 num_obj==1（单目标），所以 ious 已经是正确形状，无需 reshape

            # 4.5 记录预测结果（用于后续分析）
            meta_path = batch_data["meta_path"][bid]
            dataset = meta_path.split("/")[-4]
            sequence = meta_path.split("/")[-3]
            frame = meta_path.split("/")[-2]

            record = {
                "id": f"{dataset}/{sequence}/{frame}",
                "utterance": batch_data["utterances"][bid],
                "gt_box": batch_data["gt_bboxes"][bid][:num_obj].cpu().numpy().tolist(), # (1, 7/9) - 保留完整的旋转信息
                "pred_box": pbox[0].cpu().numpy().tolist(),  # top-1 预测框
                "ious": ious[:, 0].cpu().numpy().tolist(),  # top-1 的 IoU
            }
            self.prediction_records.append(record)

            # 累计平均 IoU（用于 mIoU 计算）
            self.dets["iou"] += ious[:, 0].cpu().numpy().sum()
            self.dets["num_iou"] += num_obj

            # ============ 5. 计算准确率指标 ============
            # 遍历不同的 IoU 阈值（0.25, 0.5）
            for t in self.thresholds:
                thresholded = ious > t  # (1, 10) - 哪些预测框的 IoU 超过阈值
                # 例如 t=0.25 时，thresholded = [False, True, True, False, ...]

                # 遍历不同的 top-k（1, 5, 10）
                for k in self.topks:
                    # 检查 top-k 个预测中是否有任意一个 IoU > threshold
                    found = thresholded[:, :k].any(1)  # (1,) - bool tensor
                    # .any(1)：对于每个 gt，检查 top-k 中是否有匹配的
                    # 例如 k=1: 只看 top-1 是否匹配
                    # 例如 k=5: 看 top-5 中是否有任意一个匹配
                    
                    all_found = found.all().item()  # bool (0 或 1)
                    # .all()：要求所有 gt 都被匹配（单 gt 时等价于 found[0]）
                    # .item()：转为 Python bool/int

                    # 更新统计量
                    # NOTE: bbf = "Box given span (contrastive)" 对比学习方法
                    self.dets[(prefix, t, k, "bbf")] += all_found  # 成功样本数 +1 或 +0
                    self.gts[(prefix, t, k, "bbf")] += 1  # 总样本数 +1

                    # 只对最后一层的 top-1 计算 total_acc（用于打印 Acc@0.25）
                    if prefix == "last_" and k == 1:
                        self.dets[("total_acc", t, "bbf")] += all_found
                        self.gts[("total_acc", t, "bbf")] += 1

    def _parse_gt(self, end_points):
        positive_map = torch.clone(end_points["positive_map"])  # (B, K, 256)
        positive_map[positive_map > 0] = 1
        gt_center = end_points["center_label"][:, :, 0:3]  # (B, K, 3)
        gt_size = end_points["size_gts"]  # (B, K2,3)
        gt_bboxes = torch.cat([gt_center, gt_size], dim=-1)  # cxcyczwhd
        if self.only_root:  # MARK ony first object if true
            positive_map = positive_map[:, :1]  # (B, 1, 256)
            gt_bboxes = gt_bboxes[:, :1]  # (B, 1, 6)
        return positive_map, gt_bboxes
