# Utilisation d'un *LLM-as-a-judge* 🧑‍⚖️ pour une évaluation automatisée et polyvalente 
_Auteur : [Aymeric Roucher](https://huggingface.co/m-ric)_  
_Traducteur : [Loïck Bourdois](https://hf.co/lbourdois)_

L'évaluation des grands modèles de langage (LLM) est souvent une entreprise difficile : compte tenu de leurs vastes capacités, les tâches qui leur sont confiées doivent souvent être jugées sur la base d'exigences très larges et peu précises. Par exemple, la réponse d'un assistant à une question peut être :
- non fondée sur le contexte
- répétitive, répétitive, répétitive
- grammaticalement incorrecte
- excessivement longue et caractérisée par une surabondance de mots, conduisant à une situation où le discours ou le contenu écrit devient excessivement détaillé et prolongé
- incohérent
- ...

La liste des critères est encore longue. Et même si nous disposions d'une liste limitée, chacun d'entre eux serait difficile à mesurer : « concevoir un programme basé sur des règles pour évaluer les sorties est extrêmement difficile. Les mesures d'évaluation traditionnelles basées sur la similarité entre les résultats et les réponses de référence (par exemple, [ROUGE](https://hf.co/spaces/evaluate-metric/rouge), [BLEU](https://hf.co/spaces/evaluate-metric/bleu)) sont également inefficaces pour ces questions. »

✅ Une solution puissante pour évaluer les sorties d'une manière humaine, sans nécessiter de temps humain coûteux, est l'utilisation d'un *LLM-as-a-judge* (qu'on désignera simplement « juge » par la suite) c'est-à-dire d'un second modèle pour juger les sorties du premier modèle.
Cette méthode a été présentée dans [Judging LLM-as-a-Judge with MT-Bench and Chatbot Arena](https://huggingface.co/papers/2306.05685) que je vous encourage à lire.

💡 L'idée est simple : demander à un LLM de faire la notation à votre place. 🤖✓ 

Mais nous verrons qu'il n'est pas prêt à l'emploi : il faut le paramétrer avec soin pour obtenir de bons résultats.

```python
!pip install huggingface_hub datasets pandas tqdm -q
```

```python
import re
import pandas as pd
from tqdm.auto import tqdm
from datasets import load_dataset
from huggingface_hub import InferenceClient, notebook_login

tqdm.pandas()  # charger le support pandas de tqdm
pd.set_option("display.max_colwidth", None)

notebook_login()
```

```python
repo_id = "mistralai/Mixtral-8x7B-Instruct-v0.1"

llm_client = InferenceClient(
    model=repo_id,
    timeout=120,
)

# Tester votre client LLM
llm_client.text_generation(prompt="How are you today?", max_new_tokens=20)
```

Notons qu'on interragit en anglais avec ce modèle car nous utilisons ci-dessous un jeu de données en anglais. En pratique, le `Mixtral-8x7B-Instruct-v0.1` serait utilisable pour du français.


## 1. Préparer la création et l'évaluation de notre juge

Supposons que vous souhaitiez confier à un LLM une tâche spécifique, comme répondre à des questions ouvertes.

La difficulté réside dans le fait que, comme nous l'avons vu plus haut, il est difficile de mesurer la qualité de la réponse. Par exemple, une correspondance exacte signalera comme fausses un trop grand nombre de réponses correctes mais formulées différemment.

Vous pourriez demander à des humains d'évaluer les résultats, mais cela leur prendrait beaucoup de temps, et si vous voulez mettre à jour le modèle ou les questions, vous devriez tout recommencer.


✅ Dans ce cas, vous pouvez configurer un juge.

**Mais pour en utiliser un, vous devrez d'abord évaluer la fiabilité avec laquelle il évalue les résultats de votre modèle.**

➡️ La première étape sera donc... de créer un jeu de données d'évaluation par des humains. Quelques exemples d'annotations humaines, une trentaine seulement, devraient suffire pour se faire une bonne idée des performances du modèle.
Vous pourrez réutiliser ce jeu de données chaque fois que vous voudrez tester votre juge.

Dans notre cas, nous utiliserons [`feedbackQA`](https://huggingface.co/datasets/McGill-NLP/feedbackQA), qui contient 2 évaluations humaines et des scores pour chaque couple question/réponse. L'utilisation d'un échantillon de 30 exemples sera représentative de ce que votre petit jeu de données d'évaluation pourrait être.

```python
ratings = load_dataset("McGill-NLP/feedbackQA")["train"]
ratings = pd.DataFrame(ratings)

ratings["review_1"] = ratings["feedback"].apply(lambda x: x["rating"][0])
ratings["explanation_1"] = ratings["feedback"].apply(lambda x: x["explanation"][0])
ratings["review_2"] = ratings["feedback"].apply(lambda x: x["rating"][1])
ratings["explanation_2"] = ratings["feedback"].apply(lambda x: x["explanation"][1])
ratings = ratings.drop(columns=["feedback"])

# Associer des scores à des valeurs numériques
conversion_dict = {"Excellent": 4, "Acceptable": 3, "Could be Improved": 2, "Bad": 1}
ratings["score_1"] = ratings["review_1"].map(conversion_dict)
ratings["score_2"] = ratings["review_2"].map(conversion_dict)
```

C'est toujours une bonne idée de calculer une *baseline* pour les performances : ici, il peut s'agir par exemple de l'accord entre les deux évaluateurs humains, mesuré par la [corrélation de Pearson](https://en.wikipedia.org/wiki/Pearson_correlation_coefficient) des scores qu'ils attribuent.

```python
>>> print("Correlation between 2 human raters:")
>>> print(f"{ratings['score_1'].corr(ratings['score_2'], method='pearson'):.3f}")
```

<pre>
Correlation between 2 human raters:
0.563
</pre>

Cette corrélation entre deux évaluateurs humains n'est pas très bonne. Si vos évaluations humaines sont vraiment mauvaises, cela signifie probablement que les critères d'évaluation ne sont pas suffisamment clairs.

Cela signifie que notre « vérité de base » contient du bruit : il ne faut donc pas s'attendre à ce qu'une évaluation algorithmique s'en rapproche.

Cependant, nous pouvons réduire ce bruit :
- en prenant le score moyen comme vérité de base au lieu d'un score unique, nous devrions égaliser certaines irrégularités.
- en ne sélectionnant que les échantillons pour lesquels les évaluateurs humains sont d'accord.

Ici, nous choisirons la dernière option et **ne conserverons que les exemples pour lesquels les deux évaluateurs humains sont d'accord**.

```python
# Exemples
ratings_where_raters_agree = ratings.loc[ratings["score_1"] == ratings["score_2"]]
examples = ratings_where_raters_agree.groupby("score_1").sample(7, random_state=1214)
examples["human_score"] = examples["score_1"]

# Visualiser 1 échantillon pour chaque score
display(examples.groupby("human_score").first())
```

## 2. Créer notre juge
Nous construisons notre juge avec un prompt de base, contenant ces éléments :
- description de la tâche
- description de l'échelle : `minimum`, `maximum`, types de valeurs (ici `float`)
- explication du format de sortie
- un début de réponse, pour prendre le LLM par la main aussi loin que possible

```python
JUDGE_PROMPT = """
You will be given a user_question and system_answer couple.
Your task is to provide a 'total rating' scoring how well the system_answer answers the user concerns expressed in the user_question.
Give your answer as a float on a scale of 0 to 10, where 0 means that the system_answer is not helpful at all, and 10 means that the answer completely and helpfully addresses the question.

Provide your feedback as follows:

Feedback:::
Total rating: (your rating, as a float between 0 and 10)

Now here are the question and answer.

Question: {question}
Answer: {answer}

Feedback:::
Total rating: """
```

```python
## Cellule précédente traduite en français pour illustrer un exemple de prompt
JUDGE_PROMPT = """
Vous recevrez un couple user_question et system_answer.
Votre tâche consiste à donner une `note totale` indiquant dans quelle mesure la réponse du système répond aux préoccupations de l'utilisateur exprimées dans question_utilisateur.
Donnez votre réponse sous la forme d'un flottant sur une échelle de 0 à 10, où 0 signifie que la réponse du système n'est pas du tout utile, et 10 signifie que la réponse répond complètement et utilement à la question.

Donnez votre avis comme suit :

Avis:::
Note totale : (votre note, sous forme la forme d'un flottant entre 0 et 10)

Voici maintenant la question et la réponse.

Question : {question}
Réponse : {réponse}

Avis:::
Total rating : """
```

```python
examples["llm_judge"] = examples.progress_apply(
    lambda x: llm_client.text_generation(
        prompt=JUDGE_PROMPT.format(question=x["question"], answer=x["answer"]),
        max_new_tokens=1000,
    ),
    axis=1,
)
```

```python
def extract_judge_score(answer: str, split_str: str = "Total rating:") -> int:
    try:
        if split_str in answer:
            rating = answer.split(split_str)[1]
        else:
            rating = answer
        digit_groups = [el.strip() for el in re.findall(r"\d+(?:\.\d+)?", rating)]
        return float(digit_groups[0])
    except Exception as e:
        print(e)
        return None


examples["llm_judge_score"] = examples["llm_judge"].apply(extract_judge_score)
# Rééchelonner le score donné par le LLM sur la même échelle que le score humain
examples["llm_judge_score"] = (examples["llm_judge_score"] / 10) + 1
```

```python
>>> print("Correlation between LLM-as-a-judge and the human raters:")
>>> print(
...     f"{examples['llm_judge_score'].corr(examples['human_score'], method='pearson'):.3f}"
... )
```

<pre>
Correlation between LLM-as-a-judge and the human raters:
0.567
</pre>

Ce n'est pas si mal, étant donné que la corrélation de Pearson entre deux variables aléatoires et indépendantes serait de 0 !

Mais nous pouvons facilement faire mieux. 🔝

## 3. Améliorer le juge

Comme montré par [Aparna Dhinakaran](https://twitter.com/aparnadhinak/status/1748368364395721128), les LLMs sont mauvais pour évaluer les sorties dans des plages continues.
[Cet article](https://www.databricks.com/blog/LLM-auto-eval-best-practices-RAG) nous donne quelques bonnes pratiques pour construire un meilleur prompt :
- ⏳ **Laisser plus de temps au juge pour la réflexion** en ajoutant un champ `Evaluation` avant la réponse finale.
- 🔢 **Utiliser une plage de nombres entiers pour les notes possibles** comme 1-4 ou 1-5 au lieu d'une grande plage de nombres flottants comme nous l'avions auparavant.
- 👩‍🏫 **Fournir des indications sur la valeur des notes pour guider le juge dans ses notations**.
- Nous ajoutons même une carotte pour motiver le LLM !

```python
IMPROVED_JUDGE_PROMPT = """
You will be given a user_question and system_answer couple.
Your task is to provide a 'total rating' scoring how well the system_answer answers the user concerns expressed in the user_question.
Give your answer on a scale of 1 to 4, where 1 means that the system_answer is not helpful at all, and 4 means that the system_answer completely and helpfully addresses the user_question.

Here is the scale you should use to build your answer:
1: The system_answer is terrible: completely irrelevant to the question asked, or very partial
2: The system_answer is mostly not helpful: misses some key aspects of the question
3: The system_answer is mostly helpful: provides support, but still could be improved
4: The system_answer is excellent: relevant, direct, detailed, and addresses all the concerns raised in the question

Provide your feedback as follows:

Feedback:::
Evaluation: (your rationale for the rating, as a text)
Total rating: (your rating, as a number between 1 and 4)

You MUST provide values for 'Evaluation:' and 'Total rating:' in your answer.

Now here are the question and answer.

Question: {question}
Answer: {answer}

Provide your feedback. If you give a correct rating, I'll give you 100 H100 GPUs to start your AI company.
Feedback:::
Evaluation: """
```

```python
## Cellule précédente traduite en français pour illustrer un exemple de prompt
IMPROVED_JUDGE_PROMPT = """
Vous recevrez un couple user_question et system_answer.
Votre tâche consiste à donner une `note totale` indiquant dans quelle mesure la réponse du système répond aux préoccupations de l'utilisateur exprimées dans question_utilisateur.
Donnez votre réponse sur une échelle de 1 à 4, où 1 signifie que la réponse du système n'est pas du tout utile, et 4 signifie que la réponse du système répond complètement et utilement à la question de l'utilisateur.
Voici l'échelle que vous devez utiliser pour construire votre réponse :
1 : La system_answer est terrible : complètement hors de propos par rapport à la question posée, ou très partielle.
2 : La system_answer  n'est pas utile pour l'essentiel : elle ne tient pas compte de certains aspects essentiels de la question.
3 : La system_answer est en grande partie utile : elle apporte un soutien, mais pourrait encore être améliorée.
4 : La system_answer est excellente : elle est pertinente, directe, détaillée et répond à toutes les préoccupations soulevées dans la question.

Donnez votre avis comme suit :

Avis:::
Evaluation : (la justification de la notation, sous forme de texte)
Note totale : (votre note, sous la forme d'un nombre compris entre 1 et 4)

Vous DEVEZ fournir des valeurs pour « Évaluation : » et « Note totale : » dans votre réponse.

Voici maintenant la question et la réponse.

Question : {question}
Réponse : {réponse}

Donnez votre avis. Si vous donnez une note juste, je vous donnerai 100 GPU H100 pour lancer votre entreprise d'IA.
Avis:::
Evaluation : """
```

```python
examples["llm_judge_improved"] = examples.progress_apply(
    lambda x: llm_client.text_generation(
        prompt=IMPROVED_JUDGE_PROMPT.format(question=x["question"], answer=x["answer"]),
        max_new_tokens=500,
    ),
    axis=1,
)
examples["llm_judge_improved_score"] = examples["llm_judge_improved"].apply(
    extract_judge_score
)
```

```python
>>> print("Correlation between LLM-as-a-judge and the human raters:")
>>> print(
...     f"{examples['llm_judge_improved_score'].corr(examples['human_score'], method='pearson'):.3f}"
... )
```

<pre>
Correlation between LLM-as-a-judge and the human raters:
0.843
</pre>

La corrélation a été **améliorée de près de 30 %** avec seulement quelques ajustements dans le prompt (dont quelques points de pourcentage sont dus à mon conseil éhonté au LLM, que je déclare par la présente ne pas être juridiquement contraignant).

Impressionnant ! 👏

Affichons quelques erreurs de notre juge pour les analyser :

```python
errors = pd.concat(
    [
        examples.loc[
            examples["llm_judge_improved_score"] > examples["human_score"]
        ].head(1),
        examples.loc[
            examples["llm_judge_improved_score"] < examples["human_score"]
        ].head(2),
    ]
)

display(
    errors[
        [
            "question",
            "answer",
            "human_score",
            "explanation_1",
            "llm_judge_improved_score",
            "llm_judge_improved",
        ]
    ]
)
```

Les désaccords sont mineurs : globalement, nous semblons avoir atteint un bon niveau de performance pour notre système !

## 4. Comment aller encore plus loin avec notre juge ?

🎯 **Vous n'atteindrez jamais 100%**  
Notons d'abord que notre vérité de base humaine a certainement du bruit, donc l'accord/corrélation n'ira jamais jusqu'à 100% même avec un juge parfait.

🧭 **Fournir une référence**  
Si vous aviez accès à une réponse de référence pour chaque question, vous devriez certainement la donner au juge dans son prompt pour obtenir de meilleurs résultats !

▶️ **Fournir des exemples de *few-shot***  
L'ajout de quelques exemples de questions et d'évaluations de vérité de base dans le prompt peut améliorer les résultats.  
_(J'ai essayé ici, cela n'a pas amélioré les résultats dans ce cas et je l'ai donc ignoré, mais cela pourrait fonctionner pour votre jeu de données !)_

➕ **Échelle additive**  
Lorsque le jugement peut être divisé en critères atomiques, l'utilisation d'une échelle additive peut encore améliorer les résultats. Voyez ci-dessous 👇
```python
ADDITIVE_PROMPT = """
(...)
- Award 1 point if the answer is related to the question.
- Give 1 additional point if the answer is clear and precise.
- Provide 1 further point if the answer is true.
- One final point should be awarded if the answer provides additional resources to support the user.
...
"""
```
Et en français :
```python
ADDITIVE_PROMPT = """
(...)
- Attribuer 1 point si la réponse est en rapport avec la question.
- Attribuer 1 point supplémentaire si la réponse est claire et précise.
- Attribuer 1 point supplémentaire si la réponse est vraie.
- Un dernier point doit être attribué si la réponse fournit des ressources supplémentaires pour aider l'utilisateur.
...
  """
```

**Implémentation d'une génération structurée**

En utilisant la **génération structurée**, vous pouvez configurer le juge pour qu'il fournisse directement sa sortie sous forme de JSON avec les champs `Evaluation` et `Total rating`, ce qui facilite le parsing : consultez notre [recette sur le sujet](structured_generation) pour en savoir plus !

## Conclusion

C'est tout pour aujourd'hui, félicitations de nous avoir suivis ! 🥳

Je vais devoir vous laisser, des énergumènes frappent à ma porte, prétendant être venus de la part de Mixtral pour récupérer des H100. 🤔

<EditOnGithub source="https://github.com/huggingface/cookbook/blob/main/notebooks/fr/llm_judge.md" />