# 使用 LangChain 在 HuggingFace 文档上构建高级 RAG
_作者: [Aymeric Roucher](https://huggingface.co/m-ric)_

这个 notebook 主要讲述了你怎么构建一个高级的 RAG，用于回答一个关于特定知识库的问题（这里，是 HuggingFace 文档），使用 LangChain。

对于 RAG 的介绍，你可以查看[这个教程](rag_zephyr_langch)

RAG 系统是复杂的，它有许多组块:这里画一个简单的 RAG 图表，其中用蓝色标注了所有系统增强的可能性。

<img src="https://huggingface.co/datasets/huggingface/cookbook-images/resolve/main/RAG_workflow.png" height="700">

> 💡 可以看到，这个架构中有许多步骤可以调整：正确调整系统将带来显著的性能提升。

在这个 notebook 中，我们将研究许多这些蓝色标注的部分，看看如何调整你的 RAG 系统以获得最佳性能。

__让我们深入研究模型架构吧！__ 首先，安装所需的模型依赖项。

```python
!pip install -q torch transformers transformers accelerate bitsandbytes langchain sentence-transformers faiss-gpu openpyxl pacmap datasets langchain-community ragatouille
```

```python
%reload_ext dotenv
%dotenv
```

```python
from tqdm.notebook import tqdm
import pandas as pd
from typing import Optional, List, Tuple
from datasets import Dataset
import matplotlib.pyplot as plt

pd.set_option(
    "display.max_colwidth", None
)  # this will be helpful when visualizing retriever outputs
```

### 加载你的知识基础

```python
import datasets

ds = datasets.load_dataset("m-ric/huggingface_doc", split="train")
```

```python
from langchain.docstore.document import Document as LangchainDocument

RAW_KNOWLEDGE_BASE = [
    LangchainDocument(page_content=doc["text"], metadata={"source": doc["source"]})
    for doc in tqdm(ds)
]
```

# 1. 检索器- 嵌入 🗂️
__检索器的作用类似于内部搜索引擎__：给定用户查询，它从你的知识库中返回几个相关的片段。

这些片段随后将被输入到阅读器模型中，以帮助其生成答案。

所以 __我们的目标在这里是，给定一个用户问题，从我们的知识库中找到最多的片段来回答这个问题。__

这是一个宽泛的目标，它留下了一些问题。我们应该检索多少片段？这个参数将被命名为`top_k`。

这些片段应该有多长？这被称为 `chunk size` （片段大小）。没有一刀切的答案，但这里有一些要点：
- 🔀 你的 `chunk size` 允许从一段片段到另一段片段有所不同。
- 由于你的检索中总会存在一些噪音，增加 `top_k` 可以提高你检索到的片段中包含相关元素的概率。🎯 射更多的箭增加了你命中目标的概率。
- 同时，你检索到的文档的总长度不应过高：例如，对于大多数当前模型来说，16k 个 token 可能会因为[中间丢失现象](https://huggingface.co/papers/2307.03172)而在信息中淹没你的阅读器模型。🎯 只给你的阅读器模型提供最相关的见解，而不是一堆书！


> 在这个 notebook 中，我们使用 Langchain 库，因为 __它为向量数据库提供了大量的选项，并允许我们在整个处理过程中保留文档的元数据__。


### 1.1 将文档拆分为片段(chuncks)

- 在这一部分，__我们将知识库中的文档拆分成更小的片段__，这些片段将是喂给阅读器 LLM 生成答案的片段。
- 目标是准备一组**语义上相关的片段**。因此，它们的大小应该适配确切的想法：太小会截断想法，太大则会稀释它们。

💡 _对于文本拆分存在许多选项：按单词拆分，按句子边界拆分，递归拆分以树状方式处理文档以保留结构信息... 要了解更多关于拆分的信息，我建议你阅读[这个很棒的 notebook](https://github.com/FullStackRetrieval-com/RetrievalTutorials/blob/main/5_Levels_Of_Text_Splitting.ipynb)，这是由 Greg Kamradt 编写的。_


- **递归拆分**使用给定的一组分隔符逐步将文本分解为更小的部分，这些分隔符按从最重要到最不重要的顺序排序。如果第一次拆分没有给出正确大小或形状的片段，该方法会使用不同的分隔符在新的片段上重复自身。例如，使用分隔符列表`["\n\n", "\n", ".", ""]`：
    - 该方法首先在出现双行中断`"\n\n"`的任何地方拆分文档。
    - 结果文档将在简单的行中断`"\n"`处再次拆分，然后在句子结尾`"."`处拆分。
    - 最后，如果有些片段仍然太大，它们将在超过最大大小时拆分。

- 使用这种方法，整体结构得到了很好的保留，代价是片段大小会有轻微的变化。

> [这个空间](https://huggingface.co/spaces/A-Roucher/chunk_visualizer)让你可视化不同的拆分选项如何影响你得到的片段。

🔬 让我们用片段大小做一些实验，从任意大小开始，看看拆分是如何工作的。我们使用 Langchain 的 `RecursiveCharacterTextSplitter` 实现递归拆分。
- 参数 `chunk_size` 控制单个片段的长度：这个长度默认计算为片段中的字符数。
- 参数 `chunk_overlap` 允许相邻片段彼此有一些重叠。这减少了想法被两个相邻片段之间的拆分切割成两半的概率。我们武断地将这个设置为片段大小的1/10，你可以尝试不同的值！

```python
from langchain.text_splitter import RecursiveCharacterTextSplitter

# We use a hierarchical list of separators specifically tailored for splitting Markdown documents
# This list is taken from LangChain's MarkdownTextSplitter class.
MARKDOWN_SEPARATORS = [
    "\n#{1,6} ",
    "```\n",
    "\n\\*\\*\\*+\n",
    "\n---+\n",
    "\n___+\n",
    "\n\n",
    "\n",
    " ",
    "",
]

text_splitter = RecursiveCharacterTextSplitter(
    chunk_size=1000,  # the maximum number of characters in a chunk: we selected this value arbitrarily
    chunk_overlap=100,  # the number of characters to overlap between chunks
    add_start_index=True,  # If `True`, includes chunk's start index in metadata
    strip_whitespace=True,  # If `True`, strips whitespace from the start and end of every document
    separators=MARKDOWN_SEPARATORS,
)

docs_processed = []
for doc in RAW_KNOWLEDGE_BASE:
    docs_processed += text_splitter.split_documents([doc])
```

我们还必须记住，当我们嵌入文档时，我们将使用一个接受特定最大序列长度 `max_seq_length` 的嵌入模型。

因此，我们应该确保我们的片段大小低于这个限制，因为任何更长的片段在处理之前都会被截断，从而失去相关性。


```python
>>> from sentence_transformers import SentenceTransformer

>>> # To get the value of the max sequence_length, we will query the underlying `SentenceTransformer` object used in the RecursiveCharacterTextSplitter.
>>> print(
...     f"Model's maximum sequence length: {SentenceTransformer('thenlper/gte-small').max_seq_length}"
... )

>>> from transformers import AutoTokenizer

>>> tokenizer = AutoTokenizer.from_pretrained("thenlper/gte-small")
>>> lengths = [len(tokenizer.encode(doc.page_content)) for doc in tqdm(docs_processed)]

>>> # Plot the distribution of document lengths, counted as the number of tokens
>>> fig = pd.Series(lengths).hist()
>>> plt.title("Distribution of document lengths in the knowledge base (in count of tokens)")
>>> plt.show()
```

<pre>
Model's maximum sequence length: 512
</pre>

👀 可以看到，__片段长度与我们的 512 个 token 的限制不匹配__，并且有些文档超出了限制，因此它们的一部分将在截断中丢失！
 - 因此，我们应该更改 `RecursiveCharacterTextSplitter` 类，以计算 token 数量而不是字符数量。
 - 然后，我们可以选择一个特定的片段大小，这里我们会选择低于 512 的阈值：
    - 较小的文档可能允许拆分更专注于特定想法的内容。
    - 但太小的片段会拆分句子，从而再次失去意义：适当的调整是一个平衡的问题。

```python
>>> from langchain.text_splitter import RecursiveCharacterTextSplitter
>>> from transformers import AutoTokenizer

>>> EMBEDDING_MODEL_NAME = "thenlper/gte-small"


>>> def split_documents(
...     chunk_size: int,
...     knowledge_base: List[LangchainDocument],
...     tokenizer_name: Optional[str] = EMBEDDING_MODEL_NAME,
... ) -> List[LangchainDocument]:
...     """
...     Split documents into chunks of maximum size `chunk_size` tokens and return a list of documents.
...     """
...     text_splitter = RecursiveCharacterTextSplitter.from_huggingface_tokenizer(
...         AutoTokenizer.from_pretrained(tokenizer_name),
...         chunk_size=chunk_size,
...         chunk_overlap=int(chunk_size / 10),
...         add_start_index=True,
...         strip_whitespace=True,
...         separators=MARKDOWN_SEPARATORS,
...     )

...     docs_processed = []
...     for doc in knowledge_base:
...         docs_processed += text_splitter.split_documents([doc])

...     # Remove duplicates
...     unique_texts = {}
...     docs_processed_unique = []
...     for doc in docs_processed:
...         if doc.page_content not in unique_texts:
...             unique_texts[doc.page_content] = True
...             docs_processed_unique.append(doc)

...     return docs_processed_unique


>>> docs_processed = split_documents(
...     512,  # We choose a chunk size adapted to our model
...     RAW_KNOWLEDGE_BASE,
...     tokenizer_name=EMBEDDING_MODEL_NAME,
... )

>>> # Let's visualize the chunk sizes we would have in tokens from a common model
>>> from transformers import AutoTokenizer

>>> tokenizer = AutoTokenizer.from_pretrained(EMBEDDING_MODEL_NAME)
>>> lengths = [len(tokenizer.encode(doc.page_content)) for doc in tqdm(docs_processed)]
>>> fig = pd.Series(lengths).hist()
>>> plt.title("Distribution of document lengths in the knowledge base (in count of tokens)")
>>> plt.show()
```

<img src="data:image/jpeg;base64,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">


➡️ 现在分块长度分布看起来好多了!

### 1.2 构建向量数据库

我们希望为我们知识库的所有片段计算嵌入向量：要了解更多关于句子嵌入的信息，我们建议阅读[这个指南](https://osanseviero.github.io/hackerllama/blog/posts/sentence_embeddings/)。

#### 检索的工作原理

一旦所有片段都被嵌入，我们就将它们存储到一个向量数据库中。当用户输入一个查询时，它会被之前使用的同一模型嵌入，并且相似性搜索会返回向量数据库中最接近的文档。

因此，技术挑战在于，给定一个查询向量，快速找到向量数据库中这个向量的最近邻。为此，我们需要选择两件事：一个距离度量，以及一个搜索算法，以便在成千上万的记录数据库中快速找到最近邻。

##### 最近邻搜索算法

最近邻搜索算法有很多选择：我们选择 Facebook 的 [FAISS](https://github.com/facebookresearch/faiss)，因为 FAISS 对于大多数用例来说性能足够好，而且它广为人知，因此被广泛实现。

##### 距离度量

关于距离度量，你可以在[这里](https://osanseviero.github.io/hackerllama/blog/posts/sentence_embeddings/#distance-between-embeddings)找到一个很好的指南。简而言之：
- **余弦相似度**计算两个向量之间的相似性，作为它们相对角度的余弦值：它允许我们比较向量的方向，而不考虑它们的幅度。使用它需要对所有向量进行归一化，将它们重新缩放到单位范数。
- **点积**考虑幅度，有时会有不希望的效果，即增加向量的长度会使它与所有其他向量更相似。
- **欧氏距离**是向量末端之间的距离。

你可以尝试[这个小测](https://developers.google.com/machine-learning/clustering/similarity/check-your-understanding)来检查你对这些概念的理解。但是一旦向量被归一化，[选择特定的距离度量并不重要](https://platform.openai.com/docs/guides/embeddings/which-distance-function-should-i-use)。

我们的特定模型与余弦相似度配合得很好，所以我们选择这个距离度量，并在嵌入模型中以及 FAISS 索引的 `distance_strategy` 参数中设置它。使用余弦相似度，我们需要归一化我们的嵌入向量。

🚨👇 下面的单元格需要在 A10G 上运行几分钟！



```python
from langchain.vectorstores import FAISS
from langchain_community.embeddings import HuggingFaceEmbeddings
from langchain_community.vectorstores.utils import DistanceStrategy

embedding_model = HuggingFaceEmbeddings(
    model_name=EMBEDDING_MODEL_NAME,
    multi_process=True,
    model_kwargs={"device": "cuda"},
    encode_kwargs={"normalize_embeddings": True},  # set True for cosine similarity
)

KNOWLEDGE_VECTOR_DATABASE = FAISS.from_documents(
    docs_processed, embedding_model, distance_strategy=DistanceStrategy.COSINE
)
```

👀 为了可视化搜索最接近的文档，我们使用 PaCMAP 将我们的嵌入向量从 384 维降至 2 维。

💡 _我们选择 PaCMAP 而不是其他技术，如 t-SNE 或 UMAP，因为[它效率高（保留局部和全局结构），对初始化参数鲁棒且速度快](https://www.nature.com/articles/s42003-022-03628-x#Abs1)。_


```python
# embed a user query in the same space
user_query = "How to create a pipeline object?"
query_vector = embedding_model.embed_query(user_query)
```

```python
import pacmap
import numpy as np
import plotly.express as px

embedding_projector = pacmap.PaCMAP(
    n_components=2, n_neighbors=None, MN_ratio=0.5, FP_ratio=2.0, random_state=1
)

embeddings_2d = [
    list(KNOWLEDGE_VECTOR_DATABASE.index.reconstruct_n(idx, 1)[0])
    for idx in range(len(docs_processed))
] + [query_vector]

# fit the data (The index of transformed data corresponds to the index of the original data)
documents_projected = embedding_projector.fit_transform(np.array(embeddings_2d), init="pca")
```

```python
df = pd.DataFrame.from_dict(
    [
        {
            "x": documents_projected[i, 0],
            "y": documents_projected[i, 1],
            "source": docs_processed[i].metadata["source"].split("/")[1],
            "extract": docs_processed[i].page_content[:100] + "...",
            "symbol": "circle",
            "size_col": 4,
        }
        for i in range(len(docs_processed))
    ]
    + [
        {
            "x": documents_projected[-1, 0],
            "y": documents_projected[-1, 1],
            "source": "User query",
            "extract": user_query,
            "size_col": 100,
            "symbol": "star",
        }
    ]
)

# visualize the embedding
fig = px.scatter(
    df,
    x="x",
    y="y",
    color="source",
    hover_data="extract",
    size="size_col",
    symbol="symbol",
    color_discrete_map={"User query": "black"},
    width=1000,
    height=700,
)
fig.update_traces(
    marker=dict(opacity=1, line=dict(width=0, color="DarkSlateGrey")), selector=dict(mode="markers")
)
fig.update_layout(
    legend_title_text="<b>Chunk source</b>",
    title="<b>2D Projection of Chunk Embeddings via PaCMAP</b>",
)
fig.show()
```

<img src="https://huggingface.co/datasets/huggingface/cookbook-images/resolve/main/PaCMAP_embeddings.png" height="700">

➡️ 在上面的图表中，你可以看到知识库文档的空间表示。由于向量嵌入代表了文档的含义，它们在意义上的接近应该在它们的嵌入的接近程度上反映出来。

用户查询的嵌入也被显示出来：我们想要找到意义最接近的 `k` 个文档，因此我们选择最接近的 `k` 个向量。

在 LangChain 向量数据库实现中，这个搜索操作是由方法 `vector_database.similarity_search(query)` 执行的。

这里是结果：


```python
>>> print(f"\nStarting retrieval for {user_query=}...")
>>> retrieved_docs = KNOWLEDGE_VECTOR_DATABASE.similarity_search(query=user_query, k=5)
>>> print("\n==================================Top document==================================")
>>> print(retrieved_docs[0].page_content)
>>> print("==================================Metadata==================================")
>>> print(retrieved_docs[0].metadata)
```

<pre>
Starting retrieval for user_query='How to create a pipeline object?'...

==================================Top document==================================
```

## Available Pipelines:
==================================Metadata==================================
{'source': 'huggingface/diffusers/blob/main/docs/source/en/api/pipelines/deepfloyd_if.md', 'start_index': 16887}
</pre>

# 2. 阅读器- LLM 💬

在这一部分，__LLM 阅读器读取检索到的上下文以形成其答案。__

实际上有多个可以调整的子步骤：
1. 检索到的文档内容被聚合并放入“上下文”中，这其中有许多处理选项，如_提示压缩_。
2. 上下文和用户查询被聚合并形成一个提示(prompt)，然后交给 LLM 生成其答案。


### 2.1. 阅读器模型

在选择阅读器模型时，有几个方面很重要：
- 阅读器模型的 `max_seq_length` 必须适应我们的提示(prompt)，其中包括检索器调用输出的上下文：上下文包括 5 个每份 512 个 token 的文档，所以我们至少需要 4k 个 token 的上下文长度。
- 阅读器模型

在这个例子中，我们选择了 [`HuggingFaceH4/zephyr-7b-beta`](https://huggingface.co/HuggingFaceH4/zephyr-7b-beta)，这是一个小而强大的模型。

由于每周都会发布许多模型，你可能想要用最新最好的模型替换这个模型。跟踪开源 LLM 的最佳方式是查看 [Open-source LLM leaderboard](https://huggingface.co/spaces/HuggingFaceH4/open_llm_leaderboard)。

为了加速推理，我们将加载模型的量化版本：


```python
from transformers import pipeline
import torch
from transformers import AutoTokenizer, AutoModelForCausalLM, BitsAndBytesConfig

READER_MODEL_NAME = "HuggingFaceH4/zephyr-7b-beta"

bnb_config = BitsAndBytesConfig(
    load_in_4bit=True,
    bnb_4bit_use_double_quant=True,
    bnb_4bit_quant_type="nf4",
    bnb_4bit_compute_dtype=torch.bfloat16,
)
model = AutoModelForCausalLM.from_pretrained(READER_MODEL_NAME, quantization_config=bnb_config)
tokenizer = AutoTokenizer.from_pretrained(READER_MODEL_NAME)

READER_LLM = pipeline(
    model=model,
    tokenizer=tokenizer,
    task="text-generation",
    do_sample=True,
    temperature=0.2,
    repetition_penalty=1.1,
    return_full_text=False,
    max_new_tokens=500,
)
```

```python
READER_LLM("What is 4+4? Answer:")
```

### 2.2. 提示(Prompt)

下面的 RAG 提示模板是我们将要提供给阅读器 LLM 的内容：需要将其格式化为阅读器 LLM 的聊天模板,这点非常重要。

我们向其提供我们的上下文和用户的问题。

```python
>>> prompt_in_chat_format = [
...     {
...         "role": "system",
...         "content": """Using the information contained in the context,
... give a comprehensive answer to the question.
... Respond only to the question asked, response should be concise and relevant to the question.
... Provide the number of the source document when relevant.
... If the answer cannot be deduced from the context, do not give an answer.""",
...     },
...     {
...         "role": "user",
...         "content": """Context:
... {context}
... ---
... Now here is the question you need to answer.

... Question: {question}""",
...     },
... ]
>>> RAG_PROMPT_TEMPLATE = tokenizer.apply_chat_template(
...     prompt_in_chat_format, tokenize=False, add_generation_prompt=True
... )
>>> print(RAG_PROMPT_TEMPLATE)
```

<pre>
<|system|>
Using the information contained in the context, 
give a comprehensive answer to the question.
Respond only to the question asked, response should be concise and relevant to the question.
Provide the number of the source document when relevant.
If the answer cannot be deduced from the context, do not give an answer.</s>
<|user|>
Context:
{context}
---
Now here is the question you need to answer.

Question: {question}</s>
<|assistant|>
</pre>

让我们在之前检索的文档上测试我们的阅读器!

```python
>>> retrieved_docs_text = [
...     doc.page_content for doc in retrieved_docs
... ]  # we only need the text of the documents
>>> context = "\nExtracted documents:\n"
>>> context += "".join([f"Document {str(i)}:::\n" + doc for i, doc in enumerate(retrieved_docs_text)])

>>> final_prompt = RAG_PROMPT_TEMPLATE.format(
...     question="How to create a pipeline object?", context=context
... )

>>> # Redact an answer
>>> answer = READER_LLM(final_prompt)[0]["generated_text"]
>>> print(answer)
```

<pre>
To create a pipeline object, follow these steps:

1. Define the inputs and outputs of your pipeline. These could be strings, dictionaries, or any other format that best suits your use case.

2. Inherit the `Pipeline` class from the `transformers` module and implement the following methods:

   - `preprocess`: This method takes the raw inputs and returns a preprocessed dictionary that can be passed to the model.

   - `_forward`: This method performs the actual inference using the model and returns the output tensor.

   - `postprocess`: This method takes the output tensor and returns the final output in the desired format.

   - `_sanitize_parameters`: This method is used to sanitize the input parameters before passing them to the model.

3. Load the necessary components, such as the model and scheduler, into the pipeline object.

4. Instantiate the pipeline object and return it.

Here's an example implementation based on the given context:

```python
from transformers import Pipeline
import torch
from diffusers import StableDiffusionPipeline

class MyPipeline(Pipeline):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.pipe = StableDiffusionPipeline.from_pretrained("my_model")

    def preprocess(self, inputs):
        # Preprocess the inputs as needed
        return {"input_ids":...}

    def _forward(self, inputs):
        # Run the forward pass of the model
        return self.pipe(**inputs).images[0]

    def postprocess(self, outputs):
        # Postprocess the outputs as needed
        return outputs["sample"]

    def _sanitize_parameters(self, params):
        # Sanitize the input parameters
        return params

my_pipeline = MyPipeline()
result = my_pipeline("My input string")
print(result)
```

Note that this implementation assumes that the model and scheduler are already loaded into memory. If they need to be loaded dynamically, you can modify the `__init__` method accordingly.
</pre>

### 2.3. 重排序(rerank)

对于 RAG 来说，通常更好的选择会最终检索出比你想要的更多的文档，然后在保留 `top_k` 之前，使用更强大的检索模型对结果进行重新排序。

为此，[Colbertv2](https://arxiv.org/abs/2112.01488)是一个很好的选择：它不是像我们传统的嵌入模型那样的双向编码器，而是一个交叉编码器，它计算查询 token 与每个文档 token 之间更细致的交互。

由于有了 [RAGatouille 库](https://github.com/bclavie/RAGatouille)，它的使用变得非常简单。


```python
from ragatouille import RAGPretrainedModel

RERANKER = RAGPretrainedModel.from_pretrained("colbert-ir/colbertv2.0")
```

# 3. 集成所有组件

```python
from transformers import Pipeline


def answer_with_rag(
    question: str,
    llm: Pipeline,
    knowledge_index: FAISS,
    reranker: Optional[RAGPretrainedModel] = None,
    num_retrieved_docs: int = 30,
    num_docs_final: int = 5,
) -> Tuple[str, List[LangchainDocument]]:
    # Gather documents with retriever
    print("=> Retrieving documents...")
    relevant_docs = knowledge_index.similarity_search(query=question, k=num_retrieved_docs)
    relevant_docs = [doc.page_content for doc in relevant_docs]  # keep only the text

    # Optionally rerank results
    if reranker:
        print("=> Reranking documents...")
        relevant_docs = reranker.rerank(question, relevant_docs, k=num_docs_final)
        relevant_docs = [doc["content"] for doc in relevant_docs]

    relevant_docs = relevant_docs[:num_docs_final]

    # Build the final prompt
    context = "\nExtracted documents:\n"
    context += "".join([f"Document {str(i)}:::\n" + doc for i, doc in enumerate(relevant_docs)])

    final_prompt = RAG_PROMPT_TEMPLATE.format(question=question, context=context)

    # Redact an answer
    print("=> Generating answer...")
    answer = llm(final_prompt)[0]["generated_text"]

    return answer, relevant_docs
```

让我们看看我们的 RAG 流水线是怎么回答用户的询问的。

```python
>>> question = "how to create a pipeline object?"

>>> answer, relevant_docs = answer_with_rag(
...     question, READER_LLM, KNOWLEDGE_VECTOR_DATABASE, reranker=RERANKER
... )
```

<pre>
=> Retrieving documents...
</pre>

```python
>>> print("==================================Answer==================================")
>>> print(f"{answer}")
>>> print("==================================Source docs==================================")
>>> for i, doc in enumerate(relevant_docs):
...     print(f"Document {i}------------------------------------------------------------")
...     print(doc)
```

<pre>
==================================Answer==================================
To create a pipeline object, follow these steps:

1. Import the `pipeline` function from the `transformers` module:

   ```python
   from transformers import pipeline
   ```

2. Choose the task you want to perform, such as object detection, sentiment analysis, or image generation, and pass it as an argument to the `pipeline` function:

   - For object detection:

     ```python
     >>> object_detector = pipeline('object-detection')
     >>> object_detector(image)
     [{'score': 0.9982201457023621,
       'label':'remote',
       'box': {'xmin': 40, 'ymin': 70, 'xmax': 175, 'ymax': 117}},
     ...]
     ```

   - For sentiment analysis:

     ```python
     >>> classifier = pipeline("sentiment-analysis")
     >>> classifier("This is a great product!")
     {'labels': ['POSITIVE'],'scores': tensor([0.9999], device='cpu', dtype=torch.float32)}
     ```

   - For image generation:

     ```python
     >>> image = pipeline(
    ... "stained glass of darth vader, backlight, centered composition, masterpiece, photorealistic, 8k"
    ... ).images[0]
     >>> image
     PILImage mode RGB size 7680x4320 at 0 DPI
     ```

Note that the exact syntax may vary depending on the specific pipeline being used. Refer to the documentation for more details on how to use each pipeline.

In general, the process involves importing the necessary modules, selecting the desired pipeline task, and passing it to the `pipeline` function along with any required arguments. The resulting pipeline object can then be used to perform the selected task on input data.
==================================Source docs==================================
Document 0------------------------------------------------------------
# Allocate a pipeline for object detection
>>> object_detector = pipeline('object-detection')
>>> object_detector(image)
[{'score': 0.9982201457023621,
  'label': 'remote',
  'box': {'xmin': 40, 'ymin': 70, 'xmax': 175, 'ymax': 117}},
 {'score': 0.9960021376609802,
  'label': 'remote',
  'box': {'xmin': 333, 'ymin': 72, 'xmax': 368, 'ymax': 187}},
 {'score': 0.9954745173454285,
  'label': 'couch',
  'box': {'xmin': 0, 'ymin': 1, 'xmax': 639, 'ymax': 473}},
 {'score': 0.9988006353378296,
  'label': 'cat',
  'box': {'xmin': 13, 'ymin': 52, 'xmax': 314, 'ymax': 470}},
 {'score': 0.9986783862113953,
  'label': 'cat',
  'box': {'xmin': 345, 'ymin': 23, 'xmax': 640, 'ymax': 368}}]
Document 1------------------------------------------------------------
# Allocate a pipeline for object detection
>>> object_detector = pipeline('object_detection')
>>> object_detector(image)
[{'score': 0.9982201457023621,
  'label': 'remote',
  'box': {'xmin': 40, 'ymin': 70, 'xmax': 175, 'ymax': 117}},
 {'score': 0.9960021376609802,
  'label': 'remote',
  'box': {'xmin': 333, 'ymin': 72, 'xmax': 368, 'ymax': 187}},
 {'score': 0.9954745173454285,
  'label': 'couch',
  'box': {'xmin': 0, 'ymin': 1, 'xmax': 639, 'ymax': 473}},
 {'score': 0.9988006353378296,
  'label': 'cat',
  'box': {'xmin': 13, 'ymin': 52, 'xmax': 314, 'ymax': 470}},
 {'score': 0.9986783862113953,
  'label': 'cat',
  'box': {'xmin': 345, 'ymin': 23, 'xmax': 640, 'ymax': 368}}]
Document 2------------------------------------------------------------
Start by creating an instance of [`pipeline`] and specifying a task you want to use it for. In this guide, you'll use the [`pipeline`] for sentiment analysis as an example:

```py
>>> from transformers import pipeline

>>> classifier = pipeline("sentiment-analysis")
Document 3------------------------------------------------------------
```

## Add the pipeline to 🤗 Transformers

If you want to contribute your pipeline to 🤗 Transformers, you will need to add a new module in the `pipelines` submodule
with the code of your pipeline, then add it to the list of tasks defined in `pipelines/__init__.py`.

Then you will need to add tests. Create a new file `tests/test_pipelines_MY_PIPELINE.py` with examples of the other tests.

The `run_pipeline_test` function will be very generic and run on small random models on every possible
architecture as defined by `model_mapping` and `tf_model_mapping`.

This is very important to test future compatibility, meaning if someone adds a new model for
`XXXForQuestionAnswering` then the pipeline test will attempt to run on it. Because the models are random it's
impossible to check for actual values, that's why there is a helper `ANY` that will simply attempt to match the
output of the pipeline TYPE.

You also *need* to implement 2 (ideally 4) tests.

- `test_small_model_pt` : Define 1 small model for this pipeline (doesn't matter if the results don't make sense)
  and test the pipeline outputs. The results should be the same as `test_small_model_tf`.
- `test_small_model_tf` : Define 1 small model for this pipeline (doesn't matter if the results don't make sense)
  and test the pipeline outputs. The results should be the same as `test_small_model_pt`.
- `test_large_model_pt` (`optional`): Tests the pipeline on a real pipeline where the results are supposed to
  make sense. These tests are slow and should be marked as such. Here the goal is to showcase the pipeline and to make
  sure there is no drift in future releases.
- `test_large_model_tf` (`optional`): Tests the pipeline on a real pipeline where the results are supposed to
  make sense. These tests are slow and should be marked as such. Here the goal is to showcase the pipeline and to make
  sure there is no drift in future releases.
Document 4------------------------------------------------------------
```

2. Pass a prompt to the pipeline to generate an image:

```py
image = pipeline(
	"stained glass of darth vader, backlight, centered composition, masterpiece, photorealistic, 8k"
).images[0]
image
</pre>

✅ 现在我们已经拥有了一个完整且性能出色的 RAG 系统。今天的教程就到这里！恭喜你坚持到了最后 🥳

# 进一步探索 🗺️

这并不是旅程的终点！你可以尝试许多步骤来改进你的 RAG 系统。我们建议以迭代的方式进行：对系统进行小的更改，看看哪些可以提升性能。

### 设置评估流水线

- 💬 “你不能改进你没有衡量的模型性能”，甘地说过... 或者至少 Llama2 告诉我他这么说过。无论如何，你绝对应该从衡量性能开始：这意味着构建一个小的评估数据集，然后在评估数据集上监控你的 RAG 系统的性能。

### 改进检索器

🛠️ __你可以使用这些选项来调整结果：__

- 调整分块方法：
    - 片段的大小
    - 方法：使用不同的分隔符进行拆分，使用[语义分块](https://python.langchain.com/docs/modules/data_connection/document_transformers/semantic-chunker)...

- 更改嵌入模型

👷‍♀️ __还可以考虑以下事项：__

- 尝试另一种分块方法，如语义分块
- 更改使用的索引（这里使用的是 FAISS）
- 查询扩展：以略微不同的方式重新构建用户查询以检索更多文档。

### 改进阅读器
🛠️ __这里你可以尝试以下选项来改善结果：__

- 调整提示
- 开启/关闭重排序
- 选择一个更强大的阅读器模型

💡 __这里有许多选项可以考虑以进一步改善结果：__
- 压缩检索到的上下文，只保留与回答查询最相关的部分。
- 扩展 RAG 系统，使其更加用户友好：
    - 引用来源
    - 使其能够进行对话

<EditOnGithub source="https://github.com/huggingface/cookbook/blob/main/notebooks/zh-CN/advanced_rag.md" />