## 使用大型语言模型作为评审者清理现有的偏好数据集

_作者：[David Berenstein](https://huggingface.co/davidberenstein1957) 和 [Sara Han Díaz](https://huggingface.co/sdiazlor)_

- **库**: [argilla](https://github.com/argilla-io/argilla), [hf-inference-endpoints](https://github.com/huggingface/huggingface_hub)
- **组件**: [LoadDataFromDicts](https://distilabel.argilla.io/dev/components-gallery/steps/loaddatafromdicts/), [UltraFeedback](https://distilabel.argilla.io/latest/components-gallery/tasks/ultrafeedback/), [KeepColumns](https://distilabel.argilla.io/latest/components-gallery/steps/groupcolumns/), [PreferenceToArgilla](https://distilabel.argilla.io/latest/components-gallery/steps/textgenerationtoargilla/), [InferenceEndpointsLLM](https://distilabel.argilla.io/latest/components-gallery/llms/inferenceendpointsllm/), [GlobalStep](https://distilabel.argilla.io/latest/sections/how_to_guides/basic/step/global_step/)

在本教程中，我们将使用 **distilabel** 清理数据集，利用大型语言模型（LLMs）作为评审者，通过提供 AI 反馈来评估数据的质量。[distilabel](https://github.com/argilla-io/distilabel) 是一个用于工程师的合成数据和 AI 反馈框架，帮助快速、可靠且可扩展地构建基于经过验证的研究论文的管道。查看文档 [这里](https://distilabel.argilla.io/latest/)。

为了评估响应，我们将使用与 distilabel 集成的 [无服务器 HF 推理 API](https://huggingface.co/docs/api-inference/index)。该服务免费但有请求限制，允许你通过简单的 HTTP 请求测试和评估超过 150,000 个公开模型，或者使用你自己的私有模型，推理任务在 Hugging Face 共享基础设施上进行。如果需要更多计算能力，您可以使用 [Hugging Face 推理端点](https://huggingface.co/docs/inference-endpoints/guides/create_endpoint) 部署自己的推理端点。

最后，为了进一步整理数据，我们将使用 [Argilla](https://github.com/argilla-io/argilla)，它允许我们对数据质量提供人工反馈。Argilla 是一个为 AI 工程师和领域专家提供的协作工具，帮助他们为项目构建高质量的数据集。查看文档 [这里](https://docs.argilla.io/latest/)。

## 开始

### 安装依赖

为了完成本教程，你需要通过 pip 安装 distilabel SDK 和一些第三方库。

```python
!pip install "distilabel[hf-inference-endpoints]"
```

```python
!pip install "transformers~=4.0" "torch~=2.0"
```

让我们进行必要的导入：

```python
import random

from datasets import load_dataset

from distilabel.llms import InferenceEndpointsLLM
from distilabel.pipeline import Pipeline
from distilabel.steps import (
    KeepColumns,
    LoadDataFromDicts,
    PreferenceToArgilla,
)
from distilabel.steps.tasks import UltraFeedback
```

你需要一个 `HF_TOKEN` 才能使用 HF 推理端点。在此 Notebook 中直接登录以使用它。

```python
import os
from huggingface_hub import login

login(token=os.getenv("HF_TOKEN"), add_to_git_credential=True)
```

### （可选）部署 Argilla

你可以跳过此步骤，或者将其替换为任何其他数据评估工具，但如果缺乏数据质量，模型的性能将受到影响，因此我们建议你查看你的数据。如果你已经部署了 Argilla，可以跳过此步骤。否则，你可以按照 [此指南](https://docs.argilla.io/latest/getting_started/quickstart/) 快速部署 Argilla。

同时，你需要将 Argilla 作为 distilabel 的附加组件安装。

```python
!pip install "distilabel[argilla, hf-inference-endpoints]"
```

## 数据集

在这种情况下，我们将清理一个偏好数据集，因此我们将使用 Hugging Face Hub 上的 [`Intel/orca_dpo_pairs`](https://huggingface.co/datasets/Intel/orca_dpo_pairs) 数据集。

<iframe
  src="https://huggingface.co/datasets/Intel/orca_dpo_pairs/embed/viewer/default/train"
  frameborder="0"
  width="100%"
  height="560px"
></iframe>

```python
dataset = load_dataset("Intel/orca_dpo_pairs", split="train[:20]")
```

接下来，我们将打乱 `chosen` 和 `rejected` 列，以避免数据集中的任何偏差。

```python
def shuffle_and_track(chosen, rejected):
    pair = [chosen, rejected]
    random.shuffle(pair)
    order = ["chosen" if x == chosen else "rejected" for x in pair]
    return {"generations": pair, "order": order}

dataset = dataset.map(lambda x: shuffle_and_track(x["chosen"], x["rejected"]))
```

```python
dataset = dataset.to_list()
```

### （可选）创建自定义步骤

步骤是 distilabel 管道中的一个模块，用于操作、生成或评估数据等任务。提供了一组预定义的步骤，但你也可以创建 [自定义步骤](https://distilabel.argilla.io/latest/sections/how_to_guides/basic/step/#defining-custom-steps)。与之前章节中的数据预处理不同，你可以使用自定义步骤来打乱列。这个步骤应该放在一个单独的模块中，以便导入并在管道中使用。在这种情况下，管道将首先使用 `LoadDataFromHub` 步骤加载 `orca_dpo_pairs` 数据集，然后应用 `ShuffleStep` 步骤。

```python
<CopyLLMTxtMenu containerStyle="float: right; margin-left: 10px; display: inline-flex; position: relative; z-index: 10;"></CopyLLMTxtMenu>

# "shuffle_step.py"
from typing import TYPE_CHECKING, List
from distilabel.steps import GlobalStep, StepInput

if TYPE_CHECKING:
    from distilabel.steps.typing import StepOutput
    
import random

class ShuffleStep(GlobalStep):
    @property
    def inputs(self) -> List[str]:
        return ["instruction", "chosen", "rejected"]

    @property
    def outputs(self) -> List[str]:
        return ["instruction", "generations", "order"]

    def process(self, inputs: StepInput) -> "StepOutput":
        outputs = []

        for input in inputs:
            chosen = input["chosen"]
            rejected = input["rejected"]
            pair = [chosen, rejected]
            random.shuffle(pair)
            order = ["chosen" if x == chosen else "rejected" for x in pair]
            
            outputs.append({"instruction": input["instruction"], "generations": pair, "order": order})

        yield outputs
```

```python
from shuffle_step import ShuffleStep
```

## 定义管道

为了清理一个现有的偏好数据集，我们需要定义一个包含所有必要步骤的 `Pipeline`。类似的工作流也可以用于清理 SFT（监督微调）数据集。接下来，我们将详细讲解每个步骤。

### 加载数据集
我们将使用刚才打乱的数据集作为源数据。

- 组件：`LoadDataFromDicts`
- 输入列：`system`、`question`、`chosen`、`rejected`、`generations` 和 `order`，与加载的字典列表中的键相同。
- 输出列：`system`、`instruction`、`chosen`、`rejected`、`generations` 和 `order`。我们将使用 `output_mappings` 来重命名列。

```python
load_dataset = LoadDataFromDicts(
    data=dataset[:1],
    output_mappings={"question": "instruction"},
    pipeline=Pipeline(name="showcase-pipeline"),
)
load_dataset.load()
next(load_dataset.process())
```

### 评估响应

为了评估响应的质量，我们将使用 [`meta-llama/Meta-Llama-3.1-70B-Instruct`](https://huggingface.co/meta-llama/Meta-Llama-3.1-70B-Instruct)，并应用 `UltraFeedback` 任务，根据不同维度（如有用性、诚实性、遵循指令的能力、真实性）来评判响应。对于 SFT 数据集，您可以改用 [`PrometheusEval`](../papers/prometheus.md)。

- 组件：使用 `InferenceEndpointsLLM` 的 `UltraFeedback` 任务
- 输入列：`instruction`、`generations`
- 输出列：`ratings`、`rationales`、`distilabel_metadata`、`model_name`

根据你的使用场景并为了提高结果，你可以使用任何 [你选择的其他 LLM](https://distilabel.argilla.io/latest/components-gallery/llms/)。

```python
evaluate_responses = UltraFeedback(
    aspect="overall-rating",
    llm=InferenceEndpointsLLM(
        model_id="meta-llama/Meta-Llama-3.1-70B-Instruct",
        tokenizer_id="meta-llama/Meta-Llama-3.1-70B-Instruct",
        generation_kwargs={"max_new_tokens": 512, "temperature": 0.7},
    ),
    pipeline=Pipeline(name="showcase-pipeline"),
)
evaluate_responses.load()
next(
    evaluate_responses.process(
        [
            {
                "instruction": "What's the capital of Spain?",
                "generations": ["Madrid", "Barcelona"],
            }
        ]
    )
)
```

### 仅保留必要的列

我们将去除不需要的列。

- 组件：`KeepColumns`
- 输入列：`system`、`instruction`、`chosen`、`rejected`、`generations`、`ratings`、`rationales`、`distilabel_metadata` 和 `model_name`
- 输出列：`instruction`、`chosen`、`rejected`、`generations` 和 `order`

```python
keep_columns = KeepColumns(
    columns=[
        "instruction",
        "generations",
        "order",
        "ratings",
        "rationales",
        "model_name",
    ],
    pipeline=Pipeline(name="showcase-pipeline"),
)
keep_columns.load()
next(
    keep_columns.process(
        [
            {
                "system": "",
                "instruction": "What's the capital of Spain?",
                "chosen": "Madrid",
                "rejected": "Barcelona",
                "generations": ["Madrid", "Barcelona"],
                "order": ["chosen", "rejected"],
                "ratings": [5, 1],
                "rationales": ["", ""],
                "model_name": "meta-llama/Meta-Llama-3.1-70B-Instruct",
            }
        ]
    )
)
```

### （可选）进一步的数据整理

你可以使用 Argilla 进一步整理您的数据。

- 组件：`PreferenceToArgilla` 步骤
- 输入列：`instruction`、`generations`、`generation_models`、`ratings`
- 输出列：`instruction`、`generations`、`generation_models`、`ratings`

```python
to_argilla = PreferenceToArgilla(
    dataset_name="cleaned-dataset",
    dataset_workspace="argilla",
    api_url="https://[your-owner-name]-[your-space-name].hf.space",
    api_key="[your-api-key]",
    num_generations=2
)
```

## 运行管道

下面，你可以看到完整管道定义:

```python
with Pipeline(name="clean-dataset") as pipeline:

    load_dataset = LoadDataFromDicts(
        data=dataset, output_mappings={"question": "instruction"}
    )

    evaluate_responses = UltraFeedback(
        aspect="overall-rating",
        llm=InferenceEndpointsLLM(
            model_id="meta-llama/Meta-Llama-3.1-70B-Instruct",
            tokenizer_id="meta-llama/Meta-Llama-3.1-70B-Instruct",
            generation_kwargs={"max_new_tokens": 512, "temperature": 0.7},
        ),
    )

    keep_columns = KeepColumns(
        columns=[
            "instruction",
            "generations",
            "order",
            "ratings",
            "rationales",
            "model_name",
        ]
    )

    to_argilla = PreferenceToArgilla(
        dataset_name="cleaned-dataset",
        dataset_workspace="argilla",
        api_url="https://[your-owner-name]-[your-space-name].hf.space",
        api_key="[your-api-key]",
        num_generations=2,
    )

    load_dataset.connect(evaluate_responses)
    evaluate_responses.connect(keep_columns)
    keep_columns.connect(to_argilla)
```

现在我们来运行管道，清理我们的偏好数据集。

```python
distiset = pipeline.run()
```

让我们检查一下！如果你已经将数据加载到 Argilla 中，你可以在 [Argilla UI 中开始标注](https://docs.argilla.io/latest/how_to_guides/annotate/)。

你可以将数据集推送到 Hub 以便与社区共享，并 [嵌入它以探索数据](https://huggingface.co/docs/hub/datasets-viewer-embed)。

```python
distiset.push_to_hub("[your-owner-name]/example-cleaned-preference-dataset")
```

<iframe
  src="https://huggingface.co/datasets/distilabel-internal-testing/example-cleaned-preference-dataset/embed/viewer/default/train"
  frameborder="0"
  width="100%"
  height="560px"
></iframe>

## 总结

在本教程中，我们展示了使用 distilabel 构建清理偏好数据集管道的详细步骤。然而，你可以根据自己的使用场景自定义此管道，例如清理 SFT 数据集或添加自定义步骤。

我们以一个偏好数据集作为起点，并通过打乱数据来避免任何偏差。接下来，我们使用一个模型通过无服务器的 Hugging Face 推理 API 评估了响应，遵循了 UltraFeedback 标准。最后，我们保留了必要的列，并使用 Argilla 进行了进一步的数据整理。

<EditOnGithub source="https://github.com/huggingface/cookbook/blob/main/notebooks/zh-CN/clean_dataset_judges_distilabel.md" />