# 高級接口功能

<DocNotebookDropdown
  classNames="absolute z-10 right-0 top-0"
  options={[
    {label: "Google Colab", value: "https://colab.research.google.com/github/huggingface/notebooks/blob/master/course/chapter9/section6.ipynb"},
    {label: "Aws Studio", value: "https://studiolab.sagemaker.aws/import/github/huggingface/notebooks/blob/master/course/chapter9/section6.ipynb"},
]} />

現在我們可以構建和共享一個基本接口, 讓我們來探索一些更高級的特性, 如狀態和解釋。

### 使用狀態保存數據

Gradio 支持 *會話狀態*, 其中數據在頁面加載中的多個提交中持續存在。會話狀態對於構建演示很有用, 例如, 你希望在用戶與模型交互時保留數據的聊天機器人。請注意, 會話狀態不會在模型的不同用戶之間共享數據。

要將數據存儲在會話狀態中, 你需要做三件事:

1. 向函數中傳遞一個 *額外的參數* , 該參數表示接口的狀態。
1. 在函數結束時, 將狀態的更新值作為 *額外的返回值* 返回。
1. 在創建`接口`時添加 'state' 輸入和 'state' 輸出組件。

請參閱下面的聊天機器人示例:

```py
import random

import gradio as gr


def chat(message, history):
    history = history or []
    if message.startswith("How many"):
        response = random.randint(1, 10)
    elif message.startswith("How"):
        response = random.choice(["Great", "Good", "Okay", "Bad"])
    elif message.startswith("Where"):
        response = random.choice(["Here", "There", "Somewhere"])
    else:
        response = "I don't know"
    history.append((message, response))
    return history, history


iface = gr.Interface(
    chat,
    ["text", "state"],
    ["chatbot", "state"],
    allow_screenshot=False,
    allow_flagging="never",
)
iface.launch()
```

<iframe src="https://course-demos-Chatbot-Demo.hf.space" frameBorder="0" height="350" title="Gradio app" class="container p-0 flex-grow space-iframe" allow="accelerometer; ambient-light-sensor; autoplay; battery; camera; document-domain; encrypted-media; fullscreen; geolocation; gyroscope; layout-animations; legacy-image-formats; magnetometer; microphone; midi; oversized-images; payment; picture-in-picture; publickey-credentials-get; sync-xhr; usb; vr ; wake-lock; xr-spatial-tracking" sandbox="allow-forms allow-modals allow-popups allow-popups-to-escape-sandbox allow-same-origin allow-scripts allow-downloads"></iframe>

請注意輸出組件的狀態如何在提交之間保持不變。注意: 可以給 state 參數傳入一個默認值, 作為 state 的初始值。

### 通過解釋來理解預測

大多數機器學習模型都是黑盒子, 函數的內部邏輯對終端用戶是隱藏的。為了提高透明度, 我們通過簡單地將 Interface 類中的解釋關鍵字設置為默認值, 使向模型添加解釋變得非常容易。這允許你的用戶理解輸入的哪些部分負責輸出。看看下面這個簡單的接口, 它顯示了一個還包括解釋的圖像分類器:

```py
import requests
import tensorflow as tf

import gradio as gr

inception_net = tf.keras.applications.MobileNetV2()  # load the model

# Download human-readable labels for ImageNet.
response = requests.get("https://git.io/JJkYN")
labels = response.text.split("\n")


def classify_image(inp):
    inp = inp.reshape((-1, 224, 224, 3))
    inp = tf.keras.applications.mobilenet_v2.preprocess_input(inp)
    prediction = inception_net.predict(inp).flatten()
    return {labels[i]: float(prediction[i]) for i in range(1000)}


image = gr.Image(shape=(224, 224))
label = gr.Label(num_top_classes=3)

title = "Gradio Image Classifiction + Interpretation Example"
gr.Interface(
    fn=classify_image, inputs=image, outputs=label, interpretation="default", title=title
).launch()
```

通過提交一個輸入, 然後單擊輸出組件下的Interpret來測試解釋功能。

<iframe src="https://course-demos-gradio-image-interpretation.hf.space" frameBorder="0" height="570" title="Gradio app" class="container p-0 flex-grow space-iframe" allow="accelerometer; ambient-light-sensor; autoplay; battery; camera; document-domain; encrypted-media; fullscreen; geolocation; gyroscope; layout-animations; legacy-image-formats; magnetometer; microphone; midi; oversized-images; payment; picture-in-picture; publickey-credentials-get; sync-xhr; usb; vr ; wake-lock; xr-spatial-tracking" sandbox="allow-forms allow-modals allow-popups allow-popups-to-escape-sandbox allow-same-origin allow-scripts allow-downloads"></iframe>

除了Gradio提供的默認解釋方法之外, 你還可以為 `interpretation` 參數指定 `shap`, 並設置 `num_shap` 參數。這使用基於 Shapley 的解釋, 你可以在 [here](https://christophm.github.io/interpretable-ml-book/shap.html) 閱讀更多信息。最後, 還可以將自己的解釋函數傳入 `interpretation` 參數。在Gradio的入門頁面 [here](https://gradio.app/getting_started/) 中可以看到一個例子。

這結束了我們對Gradio的`Interface`類的深入研究。正如我們所看到的, 這個類使用幾行Python代碼創建機器學習演示變得簡單。然而, 有時你會想通過改變佈局或鏈接多個預測函數來定製你的demo。如果我們能以某種方式將 `接口` 分成可定製的 "塊", 那不是很好嗎? 幸運的是, 有! 這是最後一部分的主題。

<EditOnGithub source="https://github.com/huggingface/course/blob/main/chapters/zh-TW/chapter9/6.mdx" />