# Xử lý dữ liệu

{#if fw === 'pt'}

<CourseFloatingBanner chapter={3}
  classNames="absolute z-10 right-0 top-0"
  notebooks={[
    {label: "Google Colab", value: "https://colab.research.google.com/github/huggingface/notebooks/blob/master/course/vi/chapter3/section2_pt.ipynb"},
    {label: "Aws Studio", value: "https://studiolab.sagemaker.aws/import/github/huggingface/notebooks/blob/master/course/vi/chapter3/section2_pt.ipynb"},
]} />

{:else}

<CourseFloatingBanner chapter={3}
  classNames="absolute z-10 right-0 top-0"
  notebooks={[
    {label: "Google Colab", value: "https://colab.research.google.com/github/huggingface/notebooks/blob/master/course/vi/chapter3/section2_tf.ipynb"},
    {label: "Aws Studio", value: "https://studiolab.sagemaker.aws/import/github/huggingface/notebooks/blob/master/course/vi/chapter3/section2_tf.ipynb"},
]} />

{/if}

{#if fw === 'pt'}
Tiếp tục với ví dụ từ [chương trước](/course/chapter2), đây là cách chúng ta sẽ huấn luyện một bộ phân loại chuỗi trên một lô trong PyTorch:

```python
import torch
from torch.optim import AdamW
from transformers import AutoTokenizer, AutoModelForSequenceClassification

# Tương tự như ví dụ trước
checkpoint = "bert-base-uncased"
tokenizer = AutoTokenizer.from_pretrained(checkpoint)
model = AutoModelForSequenceClassification.from_pretrained(checkpoint)
sequences = [
    "I've been waiting for a HuggingFace course my whole life.",
    "This course is amazing!",
]
batch = tokenizer(sequences, padding=True, truncation=True, return_tensors="pt")

# Đây là phần mới
batch["labels"] = torch.tensor([1, 1])

optimizer = AdamW(model.parameters())
loss = model(**batch).loss
loss.backward()
optimizer.step()
```
{:else}
Tiếp tục với ví dụ từ [chương trước](/course/chapter2), đây là cách chúng ta sẽ huấn luyện một bộ phân loại chuỗi trên một lô trong TensorFlow:

```python
import tensorflow as tf
import numpy as np
from transformers import AutoTokenizer, TFAutoModelForSequenceClassification

# Tương tự như ví dụ trước
checkpoint = "bert-base-uncased"
tokenizer = AutoTokenizer.from_pretrained(checkpoint)
model = TFAutoModelForSequenceClassification.from_pretrained(checkpoint)
sequences = [
    "I've been waiting for a HuggingFace course my whole life.",
    "This course is amazing!",
]
batch = dict(tokenizer(sequences, padding=True, truncation=True, return_tensors="tf"))

# Đây là phần mới
model.compile(optimizer="adam", loss="sparse_categorical_crossentropy")
labels = tf.convert_to_tensor([1, 1])
model.train_on_batch(batch, labels)
```
{/if}

Tất nhiên, chỉ huấn luyện mô hình trên hai câu sẽ không mang lại kết quả tốt. Để có được kết quả tốt hơn, bạn sẽ cần chuẩn bị một bộ dữ liệu lớn hơn.

Trong phần này, chúng tôi sẽ sử dụng tập dữ liệu MRPC (Microsoft Research Paraphrase Corpus) làm ví dụ, được giới thiệu trong [bài báo](https://www.aclweb.org/anthology/I05-5002.pdf) của William B. Dolan và Chris Brockett. Tập dữ liệu bao gồm 5,801 cặp câu, với nhãn cho biết chúng có phải là câu diễn giải hay không (tức là nếu cả hai câu đều có nghĩa giống nhau). Chúng tôi đã chọn nó cho chương này vì nó là một tập dữ liệu nhỏ, vì vậy thật dễ dàng để thử nghiệm với việc huấn luyện về nó.

### Tải bộ dữ liệu từ Hub

{#if fw === 'pt'}
<Youtube id="_BZearw7f0w"/>
{:else}
<Youtube id="W_gMJF0xomE"/>
{/if}

Hub không chỉ chứa các mô hình; nó cũng có nhiều bộ dữ liệu nhiều ngôn ngữ khác nhau. Bạn có thể xem qua tập dữ liệu [tại đây](https://huggingface.co/datasets) và chúng tôi khuyên bạn nên thử tải và xử lý bộ dữ liệu mới khi bạn đã xem qua phần này (xem tài liệu chung [tại đây](https://huggingface.co/docs/datasets/loading)). Nhưng hiện tại, hãy tập trung vào bộ dữ liệu MRPC! Đây là một trong 10 bộ dữ liệu tạo nên [bộ chuẩn GLUE](https://gluebenchmark.com/), là một điểm chuẩn học thuật được sử dụng để đo hiệu suất của các mô hình ML trên 10 tác vụ phân loại văn bản khác nhau.

Thư viện 🤗 Datasets cung cấp một lệnh rất đơn giản để tải xuống và lưu vào bộ nhớ cache một tập dữ liệu trên Hub. Chúng ta có thể tải xuống bộ dữ liệu MRPC như sau:

```py
from datasets import load_dataset

raw_datasets = load_dataset("glue", "mrpc")
raw_datasets
```

```python out
DatasetDict({
    train: Dataset({
        features: ['sentence1', 'sentence2', 'label', 'idx'],
        num_rows: 3668
    })
    validation: Dataset({
        features: ['sentence1', 'sentence2', 'label', 'idx'],
        num_rows: 408
    })
    test: Dataset({
        features: ['sentence1', 'sentence2', 'label', 'idx'],
        num_rows: 1725
    })
})
```

Như bạn có thể thấy, chúng ta nhận được một đối tượng `DatasetDict` chứa tập huấn luyện, tập kiểm định và tập kiểm thử. Mỗi tập chứa một số cột (`sentence1`, `sentence2`, `label`, và `idx`) và một số hàng thay đổi, là số phần tử trong mỗi tập (vì vậy, có 3,668 cặp câu trong tập huấn luyện, 408 trong tập kiểm chứng và 1,725 trong tập kiểm định).

Lệnh này tải xuống và lưu vào bộ nhớ cache các tập dữ liệu, mặc định lưu trong *~/.cache/huggingface/datasets*. Nhớ lại từ Chương 2 rằng bạn có thể tùy chỉnh thư mục bộ nhớ cache của mình bằng cách đặt biến môi trường `HF_HOME`.

Chúng ta có thể truy cập từng cặp câu trong đối tượng `raw_datasets` của mình bằng cách lập chỉ mục, giống như với từ điển:

```py
raw_train_dataset = raw_datasets["train"]
raw_train_dataset[0]
```

```python out
{'idx': 0,
 'label': 1,
 'sentence1': 'Amrozi accused his brother , whom he called " the witness " , of deliberately distorting his evidence .',
 'sentence2': 'Referring to him as only " the witness " , Amrozi accused his brother of deliberately distorting his evidence .'}
```

Chúng ta có thể thấy các nhãn vốn là số nguyên, vì vậy chúng ta không phải thực hiện bất kỳ bước xử lý trước nào ở đó. Để biết số nguyên nào tương ứng với nhãn nào, chúng ta có thể kiểm tra `features` của `raw_train_dataset`. Điều này sẽ cho chúng tôi biết loại của mỗi cột:

```py
raw_train_dataset.features
```

```python out
{'sentence1': Value(dtype='string', id=None),
 'sentence2': Value(dtype='string', id=None),
 'label': ClassLabel(num_classes=2, names=['not_equivalent', 'equivalent'], names_file=None, id=None),
 'idx': Value(dtype='int32', id=None)}
```

Phía sau, `label` thuộc loại `ClassLabel` và ánh xạ các số nguyên thành tên nhãn được lưu trữ trong thư mục *names*. `0` tương ứng với `không tương đương`, và `1` tương ứng với `tương đương`.

<Tip>

✏️ **Thử nghiệm thôi!** Nhìn vào phần tử thứ 15 của tập huấn luyện và phần tử 87 của tập kiểm định. Nhãn của chúng là gì?

</Tip>

### Tiền xử lý một bộ dữ liệu

{#if fw === 'pt'}
<Youtube id="0u3ioSwev3s"/>
{:else}
<Youtube id="P-rZWqcB6CE"/>
{/if}

Để tiền xử lý bộ dữ liệu, chúng ta cần chuyển văn bản thành các số mà mô hình có thể hiểu được. Như bạn đã thấy trong [chương trước](/course/chapter2), điều này được thực hiện với một tokenizer. Chúng ta có thể cung cấp cho tokenizer một câu hoặc một danh sách các câu, vì vậy chúng ta có thể tokenizer trực tiếp tất cả các câu đầu tiên và tất cả các câu thứ hai của mỗi cặp như sau:

```py
from transformers import AutoTokenizer

checkpoint = "bert-base-uncased"
tokenizer = AutoTokenizer.from_pretrained(checkpoint)
tokenized_sentences_1 = tokenizer(raw_datasets["train"]["sentence1"])
tokenized_sentences_2 = tokenizer(raw_datasets["train"]["sentence2"])
```

Tuy nhiên, chúng ta không thể chỉ chuyển hai chuỗi vào mô hình và nhận được dự đoán liệu hai câu có phải là diễn giải hay không. Chúng ta cần xử lý hai chuỗi như một cặp và áp dụng tiền xử lý thích hợp. May mắn thay, tokenizer cũng có thể nhận một cặp chuỗi và chuẩn bị nó theo cách mà mô hình BERT của ta mong đợi:

```py
inputs = tokenizer("This is the first sentence.", "This is the second one.")
inputs
```

```python out
{ 
  'input_ids': [101, 2023, 2003, 1996, 2034, 6251, 1012, 102, 2023, 2003, 1996, 2117, 2028, 1012, 102],
  'token_type_ids': [0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1],
  'attention_mask': [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]
}
```

Chúng ta đã thảo luận về `input_ids` và `attention_mask` trong [Chương 2](/course/chapter2), nhưng chúng ta tạm dừng để nói về `token_type_ids`. Trong ví dụ này, đây là phần cho mô hình biết phần nào của đầu vào là câu đầu tiên và phần nào là câu thứ hai.

<Tip>

✏️ **Thử nghiệm thôi!** Lấy phần tử 15 của tập huấn luyện và tokenize hai câu riêng biệt và như một cặp. Sự khác biệt giữa hai kết quả là gì?

</Tip>

Nếu chúng ta giải mã các ID bên trong `input_ids` trở lại các từ:

```py
tokenizer.convert_ids_to_tokens(inputs["input_ids"])
```

ta sẽ nhận được:

```python out
['[CLS]', 'this', 'is', 'the', 'first', 'sentence', '.', '[SEP]', 'this', 'is', 'the', 'second', 'one', '.', '[SEP]']
```

Có thể thấy mô hình kì vọng các đầu vào có dạng `[CLS] câu1 [SEP] câu2 [SEP]` khi có hai câu. Căn chỉnh điều này với `token_type_ids` cho ta kết quả:

```python out
['[CLS]', 'this', 'is', 'the', 'first', 'sentence', '.', '[SEP]', 'this', 'is', 'the', 'second', 'one', '.', '[SEP]']
[      0,      0,    0,     0,       0,          0,   0,       0,      1,    1,     1,        1,     1,   1,       1]
```

Như bạn có thể thấy, các phần của đầu vào tương ứng với `[CLS] câu1 [SEP]` đều có loại token ID là `0`, trong khi các phần khác, tương ứng với `câu2 [SEP]`, tất cả đều có loại token ID là `1`.

Lưu ý rằng nếu bạn chọn một checkpoint khác, bạn sẽ không nhất thiết phải có `token_type_ids` trong đầu vào được tokenize của mình (ví dụ: chúng sẽ không được trả lại nếu bạn sử dụng mô hình DistilBERT). Chúng chỉ được trả lại khi mô hình biết phải làm gì với chúng, bởi vì nó đã nhìn thấy chúng trong quá trình huấn luyện trước.

Ở đây, BERT được huấn luyện trước với các token ID và trên đầu mục tiêu mô hình ngôn ngữ được che mà chúng ta đã đề cập trong [Chương 1](/course/chapter1), nó có một mục tiêu bổ sung được gọi là _dự đoán câu tiếp theo_. Mục tiêu của tác vụ này là mô hình hóa mối quan hệ giữa các cặp câu.

Với dự đoán câu tiếp theo, mô hình được cung cấp các cặp câu (với các token được che ngẫu nhiên) và được yêu cầu dự đoán liệu câu thứ hai có theo sau câu đầu tiên hay không. Để làm cho tác vụ trở nên không tầm thường, một nửa là các câu tiếp nối nhau trong tài liệu gốc mà chúng được trích xuất, và nửa còn lại là hai câu đến từ hai tài liệu khác nhau.

Nói chung, bạn không cần phải lo lắng về việc có hay không có `token_type_ids` trong đầu vào được tokenize của mình: miễn là bạn sử dụng cùng một checkpoint cho trình tokenize và mô hình, mọi thứ sẽ ổn vì trình tokenize nhận biết cần cung cấp những gì với mô hình của nó.

Bây giờ chúng ta đã thấy cách trình tokenize của chúng ta có thể xử lý một cặp câu, chúng ta có thể sử dụng nó để mã hóa toàn bộ tập dữ liệu của mình: giống như trong [chương trước](/course/chapter2), chúng ta có thể cung cấp cho trình tokenize danh sách các cặp bằng cách đưa cho nó danh sách các câu đầu tiên, sau đó là danh sách các câu thứ hai. Điều này cũng tương thích với các tùy chọn đệm và cắt bớt mà chúng ta đã thấy trong [Chương 2](/course/chapter2). Vì vậy, một cách để tiền xử lý trước tập dữ liệu huấn luyện là:

```py
tokenized_dataset = tokenizer(
    raw_datasets["train"]["sentence1"],
    raw_datasets["train"]["sentence2"],
    padding=True,
    truncation=True,
)
```

Điều này hoạt động tốt, nhưng nó có nhược điểm là trả về từ điển (với các khóa của chúng tôi, `input_ids`, `attention_mask` và `token_type_ids`, và các giá trị là danh sách các danh sách). Nó cũng sẽ chỉ hoạt động nếu bạn có đủ RAM để lưu trữ toàn bộ tập dữ liệu của mình trong quá trình tokenize (trong khi các tập dữ liệu từ thư viện 🤗 Datasets là các tệp [Apache Arrow](https://arrow.apache.org/) được lưu trữ trên đĩa, vì vậy bạn chỉ giữ các mẫu bạn yêu cầu đã tải trong bộ nhớ).

Để giữ dữ liệu dưới dạng tập dữ liệu, chúng ta sẽ sử dụng phương thức [`Dataset.map()`](https://huggingface.co/docs/datasets/package_reference/main_classes#datasets.Dataset.map). Điều này cũng cho phép chúng ta linh hoạt hơn, nếu chúng ta cần thực hiện nhiều tiền xử lý hơn là chỉ tokenize. Phương thức `map()` hoạt động bằng cách áp dụng một hàm trên mỗi phần tử của tập dữ liệu, vì vậy hãy xác định một hàm tokenize các đầu vào của chúng ta:

```py
def tokenize_function(example):
    return tokenizer(example["sentence1"], example["sentence2"], truncation=True)
```

Hàm này lấy một từ điển (giống như các mục trong tập dữ liệu của chúng ta) và trả về một từ điển mới với các khóa `input_ids`, `attention_mask` và `token_type_ids`. Lưu ý rằng nó cũng hoạt động nếu từ điển `example` chứa một số mẫu (mỗi khóa là một danh sách các câu) vì `tokenizer` hoạt động trên danh sách các cặp câu, như đã thấy trước đây. Điều này sẽ cho phép chúng ta sử dụng tùy chọn `batch = True` trong lệnh gọi `map()`, từ đó sẽ tăng tốc đáng kể quá trình tokenize. `Tokenizer` được hỗ trợ bởi một tokenizer được viết bằng Rust từ thư viện [🤗 Tokenizer](https://github.com/huggingface/tokenizers). Tokenizer này có thể rất nhanh, nhưng chỉ khi chúng ta cung cấp nhiều đầu vào cùng một lúc.

Lưu ý rằng chúng ta đã để tạm bỏ qua tham số `padding` trong hàm tokenize của ta. Điều này là do việc đệm tất cả các mẫu đến chiều dài tối đa không hiệu quả: tốt hơn nên đệm các mẫu khi chúng ta đang tạo một lô, vì khi đó chúng ta chỉ cần đệm đến chiều dài tối đa trong lô đó chứ không phải chiều dài tối đa trong toàn bộ tập dữ liệu. Điều này có thể tiết kiệm rất nhiều thời gian và công suất xử lý khi các đầu vào có độ dài rất thay đổi!

Đây là cách chúng ta áp dụng chức năng mã hóa trên tất cả các tập dữ liệu của ta cùng một lúc. Chúng ta đang sử dụng `batch = True` trong lệnh gọi tới `map`, vì vậy, hàm được áp dụng cho nhiều phần tử của tập dữ liệu cùng một lúc, chứ không phải trên từng phần tử riêng biệt. Điều này cho phép việc tiền xử lý nhanh hơn.

```py
tokenized_datasets = raw_datasets.map(tokenize_function, batched=True)
tokenized_datasets
```

Cách thư viện 🤗 Datasets áp dụng bước xử lý này là thêm các trường mới vào bộ dữ liệu, mỗi khóa trong từ điển được trả về bởi hàm tiền xử lý một trường:

```python out
DatasetDict({
    train: Dataset({
        features: ['attention_mask', 'idx', 'input_ids', 'label', 'sentence1', 'sentence2', 'token_type_ids'],
        num_rows: 3668
    })
    validation: Dataset({
        features: ['attention_mask', 'idx', 'input_ids', 'label', 'sentence1', 'sentence2', 'token_type_ids'],
        num_rows: 408
    })
    test: Dataset({
        features: ['attention_mask', 'idx', 'input_ids', 'label', 'sentence1', 'sentence2', 'token_type_ids'],
        num_rows: 1725
    })
})
```

Bạn thậm chí có thể sử dụng đa xử lý khi áp dụng chức năng tiền xử lý của mình với `map()` bằng cách truyền tham số `num_proc`. Chúng ta không làm điều này ở đây vì thư viện 🤗 Tokenizers đã sử dụng nhiều chuỗi để tokenize ác mẫu của nhanh hơn, nhưng nếu bạn không sử dụng trình tokenize nhanh được thư viện này hỗ trợ, bước trên có thể tăng tốc quá trình xử lý trước của bạn.

`Tokenize_function` của chúng ta trả về một từ điển với các khóa `input_ids`, `attention_mask` và `token_type_ids`, vì vậy ba trường đó được thêm vào tất cả các phần bộ dữ liệu của chúng ta. Lưu ý rằng ta cũng có thể đã thay đổi các trường hiện có nếu hàm tiền xử lý trả về một giá trị mới cho một khóa hiện có trong tập dữ liệu mà ta đã áp dụng `map()`.

Điều cuối cùng chúng ta sẽ cần làm là đệm tất cả các ví dụ để có độ dài của phần tử dài nhất khi chúng tôi gộp các phần tử lại với nhau - một kỹ thuật mà chúng tôi gọi là *đệm động*.

### Phần đệm động

<Youtube id="7q5NyFT8REg"/>

{#if fw === 'pt'}
Hàm chịu trách nhiệm tập hợp các mẫu lại với nhau trong một lô được gọi là *collate function* hay *hàm đối chiếu*. Đó là một tham số bạn có thể đưa vào khi xây dựng một `DataLoader`, mặc định đây là một hàm sẽ chỉ chuyển đổi các mẫu của bạn thành các tensors PyTorch và nối chúng (đệ quy nếu các phần tử của bạn là list, tuple hoặc dict). Điều này sẽ không thể xảy ra trong trường hợp của chúng ta vì tất cả các đầu vào ta có sẽ không có cùng kích thước. Chúng ta đã cố tình hoãn việc bổ sung đệm, để chỉ áp dụng nó khi cần thiết trên mỗi lô và tránh để các đầu vào quá dài với nhiều đệm. Điều này sẽ đẩy nhanh quá trình huấn luyện lên một chút, nhưng lưu ý rằng nếu bạn đang huấn luyện trên TPU thì nó có thể gây ra vấn đề - TPU thích các hình dạng cố định, ngay cả khi điều đó yêu cầu thêm đệm.

{:else}

Hàm chịu trách nhiệm tập hợp các mẫu lại với nhau trong một lô được gọi là *collate function* hay *hàm đối chiếu*. Đó là một tham số bạn có thể đưa vào khi xây dựng một `DataLoader`, mặc định đây là một hàm sẽ chỉ chuyển đổi các mẫu của bạn thành các tensors PyTorch và nối chúng (đệ quy nếu các phần tử của bạn là list, tuple hoặc dict). Điều này sẽ không thể xảy ra trong trường hợp của chúng ta vì tất cả các đầu vào ta có sẽ không có cùng kích thước. Chúng ta đã cố tình hoãn việc bổ sung đệm, để chỉ áp dụng nó khi cần thiết trên mỗi lô và tránh để các đầu vào quá dài với nhiều đệm. Điều này sẽ đẩy nhanh quá trình huấn luyện lên một chút, nhưng lưu ý rằng nếu bạn đang huấn luyện trên TPU thì nó có thể gây ra vấn đề - TPU thích các hình dạng cố định, ngay cả khi điều đó yêu cầu thêm đệm.

{/if}

Để thực hiện điều này trong thực tế, chúng ta phải định nghĩa một hàm đối chiếu sẽ áp dụng đúng số lượng đệm cho các mục của tập dữ liệu mà chúng ta muốn gộp hàng loạt lại với nhau. May mắn thay, thư viện 🤗 Transformers cung cấp cho chúng ta một chức năng như vậy thông qua `DataCollatorWithPadding`. Cần có trình tokenize khi bạn khởi tạo nó (để biết cần sử dụng token đệm nào và liệu mô hình mong đợi đệm ở bên trái hay bên phải của các đầu vào) và sẽ thực hiện mọi thứ bạn cần:

{#if fw === 'pt'}
```py
from transformers import DataCollatorWithPadding

data_collator = DataCollatorWithPadding(tokenizer=tokenizer)
```
{:else}
```py
from transformers import DataCollatorWithPadding

data_collator = DataCollatorWithPadding(tokenizer=tokenizer, return_tensors="tf")
```
{/if}

Để kiểm tra món mới này, chúng ta hãy lấy một vài mẫu từ tập huấn luyện mà chúng ta muốn ghép lại với nhau. Ở đây, chúng ta xóa các cột `idx`, `sentence1`, và `sentence2` vì chúng không cần thiết và chứa các chuỗi (và chúng ta không thể tạo tensor bằng chuỗi) và xem độ dài của mỗi mục trong lô:

```py
samples = tokenized_datasets["train"][:8]
samples = {k: v for k, v in samples.items() if k not in ["idx", "sentence1", "sentence2"]}
[len(x) for x in samples["input_ids"]]
```

```python out
[50, 59, 47, 67, 59, 50, 62, 32]
```

Không có gì ngạc nhiên, ta nhận được các mẫu có độ dài khác nhau, từ 32 đến 67. Đệm động có nghĩa là tất cả các mẫu trong lô này phải được đệm đến chiều dài 67, chiều dài tối đa bên trong lô. Nếu không có đệm động, tất cả các mẫu sẽ phải được đệm đến độ dài tối đa trong toàn bộ tập dữ liệu hoặc độ dài tối đa mà mô hình có thể chấp nhận. Hãy kiểm tra kỹ xem `data_collator` của chúng ta có tự động đệm lô đúng cách hay không:

```py
batch = data_collator(samples)
{k: v.shape for k, v in batch.items()}
```

{#if fw === 'tf'}

```python out
{'attention_mask': TensorShape([8, 67]),
 'input_ids': TensorShape([8, 67]),
 'token_type_ids': TensorShape([8, 67]),
 'labels': TensorShape([8])}
```

{:else}

```python out
{'attention_mask': torch.Size([8, 67]),
 'input_ids': torch.Size([8, 67]),
 'token_type_ids': torch.Size([8, 67]),
 'labels': torch.Size([8])}
```

Trông khá ổn! Giờ ta đã chuyển từ văn bản thô sang các lô mà mô hình có thể xử lý, và ta đã sẵn sàng tinh chỉnh nó!

{/if}

<Tip>

✏️ **Thử nghiệm thôi!** Sao chép tiền xử lý trên tập dữ liệu GLUE SST-2. Nó hơi khác một chút vì nó bao gồm các câu đơn thay vì các cặp, nhưng phần còn lại của những gì ta đã làm sẽ tương tự nhau. Với một thử thách khó hơn, hãy cố gắng viết một hàm tiền xử lý hoạt động trên bất kỳ tác vụ GLUE nào.

</Tip>

{#if fw === 'tf'}

Bây giờ chúng ta đã có bộ dữ liệu và bộ đối chiếu dữ liệu, ta cần phải kết hợp chúng lại với nhau. Chúng ta có thể tải các lô và đối chiếu theo cách thủ công, nhưng cách này rất tốn công sức và có lẽ cũng không hiệu quả lắm. Thay vào đó, có một phương pháp đơn giản cung cấp giải pháp hiệu quả cho vấn đề này: `to_tf_dataset()`. Nso được bao một `tf.data.Dataset` xung quanh tập dữ liệu của bạn, với một chức năng đối chiếu tùy chọn. `tf.data.Dataset` là một định dạng TensorFlow gốc mà Keras có thể sử dụng cho `model.fit()`, vì vậy phương pháp này ngay lập tức chuyển đổi một 🤗 Dataset sang một định dạng sẵn sàng để huấn luyện. Hãy xem nó hoạt động với tập dữ liệu của chúng tôi!

```py
tf_train_dataset = tokenized_datasets["train"].to_tf_dataset(
    columns=["attention_mask", "input_ids", "token_type_ids"],
    label_cols=["labels"],
    shuffle=True,
    collate_fn=data_collator,
    batch_size=8,
)

tf_validation_dataset = tokenized_datasets["validation"].to_tf_dataset(
    columns=["attention_mask", "input_ids", "token_type_ids"],
    label_cols=["labels"],
    shuffle=False,
    collate_fn=data_collator,
    batch_size=8,
)
```

Và nó đó! Chúng ta có thể chuyển những bộ dữ liệu đó sang bài giảng tiếp theo, nơi việc huấn luyện sẽ trở nên đơn giản một cách dễ chịu sau tất cả những công việc khó khăn của việc xử lý trước dữ liệu.

{/if}


<EditOnGithub source="https://github.com/huggingface/course/blob/main/chapters/vi/chapter3/2.mdx" />