# Классификация звука с помощью конвейера

Классификация аудиозаписей предполагает присвоение одной или нескольких меток аудиозаписи в зависимости от ее содержания. 
Эти метки могут соответствовать различным категориям звуков, таким как музыка, речь, шум, или более специфическим категориям,
таким как пение птиц или звук автомобильного двигателя.

Прежде чем перейти к подробному описанию работы наиболее популярных аудиотрансформеров, а также к тонкой настройке пользовательской
модели, давайте посмотрим, как с помощью 🤗 Transformers можно использовать готовую предварительно обученную модель для классификации
аудиозаписи, набрав всего несколько строк кода.

Воспользуемся тем же набором данных [MINDS-14](https://huggingface.co/datasets/PolyAI/minds14), который вы исследовали в предыдущем разделе.
Если вы помните, MINDS-14 содержит записи людей, задающих вопросы системе дистанционного банковского обслуживания на нескольких языках и
диалектах, и для каждой записи имеет значение `intent_class`. Мы можем классифицировать записи по намерению звонящего.

Как и прежде, загрузим подмножество данных `en-AU`, чтобы опробовать конвейер, и повысим частоту дискретизации до 16 кГц, что и требуется
для большинства речевых моделей.

```py
from datasets import load_dataset
from datasets import Audio

minds = load_dataset("PolyAI/minds14", name="en-AU", split="train")
minds = minds.cast_column("audio", Audio(sampling_rate=16_000))
```

Для классификации аудиозаписи на множество классов мы можем использовать конвейер `audio-classification` из 🤗 Transformers.
В нашем случае нам нужна модель, настроенная на классификацию намерений, причем именно на наборе данных MINDS-14.
К счастью для нас, в Хабе есть модель, которая как раз для этого и предназначена! Давайте загрузим ее с помощью функции `pipeline()`:

```py
from transformers import pipeline

classifier = pipeline(
    "audio-classification",
    model="anton-l/xtreme_s_xlsr_300m_minds14",
)
```

Этот конвейер ожидает на вход аудиоданные в виде массива NumPy. Вся предварительная обработка исходных аудиоданных будет
удобно выполнена конвейером. Давайте рассмотрим пример, чтобы опробовать его в действии:

```py
example = minds[0]
```
Если вспомнить структуру набора данных, то исходные аудиоданные хранятся в массиве NumPy под именем `["audio"]["array"]`,
передадим их непосредственно в `classifier`.

```py
classifier(example["audio"]["array"])
```

**Output:**
```out
[
    {"score": 0.9631525278091431, "label": "pay_bill"},
    {"score": 0.02819698303937912, "label": "freeze"},
    {"score": 0.0032787492964416742, "label": "card_issues"},
    {"score": 0.0019414445850998163, "label": "abroad"},
    {"score": 0.0008378693601116538, "label": "high_value_payment"},
]
```

Модель очень уверена, что звонивший хотел узнать об оплате счета. Посмотрим, какова реальная метка для этого примера:

```py
id2label = minds.features["intent_class"].int2str
id2label(example["intent_class"])
```

Ура! Предсказанная метка оказалась верной! Здесь нам повезло найти модель, которая может классифицировать именно те метки,
которые нам нужны. В большинстве случаев при решении задачи классификации набор классов в предварительно обученной модели не совпадает
с теми классами, которые необходимо выделить с помощью модели. В этом случае можно произвести дообучение предварительно обученной
модели, чтобы "откалибровать" ее по точному набору меток классов. Как это сделать, мы узнаем в следующих разделах. Теперь рассмотрим
другую очень распространенную задачу обработки речи - _автоматическое распознавание речи_.

<EditOnGithub source="https://github.com/huggingface/audio-transformers-course/blob/main/chapters/ru/chapter2/audio_classification_pipeline.mdx" />