# HfApi Client

Below is the documentation for the `HfApi` class, which serves as a Python wrapper for the Hugging Face Hub's API.

All methods from the `HfApi` are also accessible from the package's root directly. Both approaches are detailed below.

Using the root method is more straightforward but the [HfApi](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi) class gives you more flexibility.
In particular, you can pass a token that will be reused in all HTTP calls. This is different
from `hf auth login` or [login()](/docs/huggingface_hub/main/en/package_reference/authentication#huggingface_hub.login) as the token is not persisted on the machine.
It is also possible to provide a different endpoint or configure a custom user-agent.

```python
from huggingface_hub import HfApi, list_models

# Use root method
models = list_models()

# Or configure a HfApi client
hf_api = HfApi(
    endpoint="https://huggingface.co", # Can be a Private Hub endpoint.
    token="hf_xxx", # Token is not persisted on the machine.
)
models = hf_api.list_models()
```

## HfApi[[huggingface_hub.HfApi]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.HfApi</name><anchor>huggingface_hub.HfApi</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L1649</source><parameters>[{"name": "endpoint", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[str, bool, None] = None"}, {"name": "library_name", "val": ": Optional[str] = None"}, {"name": "library_version", "val": ": Optional[str] = None"}, {"name": "user_agent", "val": ": Union[dict, str, None] = None"}, {"name": "headers", "val": ": Optional[dict[str, str]] = None"}]</parameters><paramsdesc>- **endpoint** (`str`, *optional*) --
  Endpoint of the Hub. Defaults to <https://huggingface.co>.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.
- **library_name** (`str`, *optional*) --
  The name of the library that is making the HTTP request. Will be added to
  the user-agent header. Example: `"transformers"`.
- **library_version** (`str`, *optional*) --
  The version of the library that is making the HTTP request. Will be added
  to the user-agent header. Example: `"4.24.0"`.
- **user_agent** (`str`, `dict`, *optional*) --
  The user agent info in the form of a dictionary or a single string. It will
  be completed with information about the installed packages.
- **headers** (`dict`, *optional*) --
  Additional headers to be sent with each request. Example: `{"X-My-Header": "value"}`.
  Headers passed here are taking precedence over the default headers.</paramsdesc><paramgroups>0</paramgroups></docstring>

Client to interact with the Hugging Face Hub via HTTP.

The client is initialized with some high-level settings used in all requests
made to the Hub (HF endpoint, authentication, user agents...). Using the `HfApi`
client is preferred but not mandatory as all of its public methods are exposed
directly at the root of `huggingface_hub`.





<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>accept_access_request</name><anchor>huggingface_hub.HfApi.accept_access_request</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L8750</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "user", "val": ": str"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  The id of the repo to accept access request for.
- **user** (`str`) --
  The username of the user which access request should be accepted.
- **repo_type** (`str`, *optional*) --
  The type of the repo to accept access request for. Must be one of `model`, `dataset` or `space`.
  Defaults to `model`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><raises>- `HfHubHTTPError` -- 
  HTTP 400 if the repo is not gated.
- `HfHubHTTPError` -- 
  HTTP 403 if you only have read-only access to the repo. This can be the case if you don't have `write`
  or `admin` role in the organization the repo belongs to or if you passed a `read` token.
- `HfHubHTTPError` -- 
  HTTP 404 if the user does not exist on the Hub.
- `HfHubHTTPError` -- 
  HTTP 404 if the user access request cannot be found.
- `HfHubHTTPError` -- 
  HTTP 404 if the user access request is already in the accepted list.</raises><raisederrors>`HfHubHTTPError`</raisederrors></docstring>

Accept an access request from a user for a given gated repo.

Once the request is accepted, the user will be able to download any file of the repo and access the community
tab. If the approval mode is automatic, you don't have to accept requests manually. An accepted request can be
cancelled or rejected at any time using [cancel_access_request()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.cancel_access_request) and [reject_access_request()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.reject_access_request).

For more info about gated repos, see https://huggingface.co/docs/hub/models-gated.








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>add_collection_item</name><anchor>huggingface_hub.HfApi.add_collection_item</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L8302</source><parameters>[{"name": "collection_slug", "val": ": str"}, {"name": "item_id", "val": ": str"}, {"name": "item_type", "val": ": CollectionItemType_T"}, {"name": "note", "val": ": Optional[str] = None"}, {"name": "exists_ok", "val": ": bool = False"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **collection_slug** (`str`) --
  Slug of the collection to update. Example: `"TheBloke/recent-models-64f9a55bb3115b4f513ec026"`.
- **item_id** (`str`) --
  ID of the item to add to the collection. It can be the ID of a repo on the Hub (e.g. `"facebook/bart-large-mnli"`)
  or a paper id (e.g. `"2307.09288"`).
- **item_type** (`str`) --
  Type of the item to add. Can be one of `"model"`, `"dataset"`, `"space"` or `"paper"`.
- **note** (`str`, *optional*) --
  A note to attach to the item in the collection. The maximum size for a note is 500 characters.
- **exists_ok** (`bool`, *optional*) --
  If `True`, do not raise an error if item already exists.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><raises>- `HfHubHTTPError` -- 
  HTTP 403 if you only have read-only access to the repo. This can be the case if you don't have `write`
  or `admin` role in the organization the repo belongs to or if you passed a `read` token.
- `HfHubHTTPError` -- 
  HTTP 404 if the item you try to add to the collection does not exist on the Hub.
- `HfHubHTTPError` -- 
  HTTP 409 if the item you try to add to the collection is already in the collection (and exists_ok=False)</raises><raisederrors>`HfHubHTTPError`</raisederrors></docstring>
Add an item to a collection on the Hub.



Returns: [Collection](/docs/huggingface_hub/main/en/package_reference/collections#huggingface_hub.Collection)





<ExampleCodeBlock anchor="huggingface_hub.HfApi.add_collection_item.example">

Example:

```py
>>> from huggingface_hub import add_collection_item
>>> collection = add_collection_item(
...     collection_slug="davanstrien/climate-64f99dc2a5067f6b65531bab",
...     item_id="pierre-loic/climate-news-articles",
...     item_type="dataset"
... )
>>> collection.items[-1].item_id
"pierre-loic/climate-news-articles"
# ^item got added to the collection on last position

# Add item with a note
>>> add_collection_item(
...     collection_slug="davanstrien/climate-64f99dc2a5067f6b65531bab",
...     item_id="datasets/climate_fever",
...     item_type="dataset"
...     note="This dataset adopts the FEVER methodology that consists of 1,535 real-world claims regarding climate-change collected on the internet."
... )
(...)
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>add_space_secret</name><anchor>huggingface_hub.HfApi.add_space_secret</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L6770</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "key", "val": ": str"}, {"name": "value", "val": ": str"}, {"name": "description", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  ID of the repo to update. Example: `"bigcode/in-the-stack"`.
- **key** (`str`) --
  Secret key. Example: `"GITHUB_API_KEY"`
- **value** (`str`) --
  Secret value. Example: `"your_github_api_key"`.
- **description** (`str`, *optional*) --
  Secret description. Example: `"Github API key to access the Github API"`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups></docstring>
Adds or updates a secret in a Space.

Secrets allow to set secret keys or tokens to a Space without hardcoding them.
For more details, see https://huggingface.co/docs/hub/spaces-overview#managing-secrets.




</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>add_space_variable</name><anchor>huggingface_hub.HfApi.add_space_variable</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L6859</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "key", "val": ": str"}, {"name": "value", "val": ": str"}, {"name": "description", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  ID of the repo to update. Example: `"bigcode/in-the-stack"`.
- **key** (`str`) --
  Variable key. Example: `"MODEL_REPO_ID"`
- **value** (`str`) --
  Variable value. Example: `"the_model_repo_id"`.
- **description** (`str`) --
  Description of the variable. Example: `"Model Repo ID of the implemented model"`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups></docstring>
Adds or updates a variable in a Space.

Variables allow to set environment variables to a Space without hardcoding them.
For more details, see https://huggingface.co/docs/hub/spaces-overview#managing-secrets-and-environment-variables




</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>auth_check</name><anchor>huggingface_hub.HfApi.auth_check</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L9780</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  The repository to check for access. Format should be `"user/repo_name"`.
  Example: `"user/my-cool-model"`.

- **repo_type** (`str`, *optional*) --
  The type of the repository. Should be one of `"model"`, `"dataset"`, or `"space"`.
  If not specified, the default is `"model"`.

- **token** `(Union[bool, str, None]`, *optional*) --
  A valid user access token. If not provided, the locally saved token will be used, which is the
  recommended authentication method. Set to `False` to disable authentication.
  Refer to: https://huggingface.co/docs/huggingface_hub/quick-start#authentication.</paramsdesc><paramgroups>0</paramgroups><raises>- [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) -- 
  Raised if the repository does not exist, is private, or the user does not have access. This can
  occur if the `repo_id` or `repo_type` is incorrect or if the repository is private but the user
  is not authenticated.

- [GatedRepoError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.GatedRepoError) -- 
  Raised if the repository exists but is gated and the user is not authorized to access it.</raises><raisederrors>[RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) or [GatedRepoError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.GatedRepoError)</raisederrors></docstring>

Check if the provided user token has access to a specific repository on the Hugging Face Hub.

This method verifies whether the user, authenticated via the provided token, has access to the specified
repository. If the repository is not found or if the user lacks the required permissions to access it,
the method raises an appropriate exception.







Example:
<ExampleCodeBlock anchor="huggingface_hub.HfApi.auth_check.example">

Check if the user has access to a repository:

```python
>>> from huggingface_hub import auth_check
>>> from huggingface_hub.utils import GatedRepoError, RepositoryNotFoundError

try:
    auth_check("user/my-cool-model")
except GatedRepoError:
    # Handle gated repository error
    print("You do not have permission to access this gated repository.")
except RepositoryNotFoundError:
    # Handle repository not found error
    print("The repository was not found or you do not have access.")
```

</ExampleCodeBlock>

In this example:
- If the user has access, the method completes successfully.
- If the repository is gated or does not exist, appropriate exceptions are raised, allowing the user
to handle them accordingly.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>cancel_access_request</name><anchor>huggingface_hub.HfApi.cancel_access_request</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L8710</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "user", "val": ": str"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  The id of the repo to cancel access request for.
- **user** (`str`) --
  The username of the user which access request should be cancelled.
- **repo_type** (`str`, *optional*) --
  The type of the repo to cancel access request for. Must be one of `model`, `dataset` or `space`.
  Defaults to `model`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><raises>- `HfHubHTTPError` -- 
  HTTP 400 if the repo is not gated.
- `HfHubHTTPError` -- 
  HTTP 403 if you only have read-only access to the repo. This can be the case if you don't have `write`
  or `admin` role in the organization the repo belongs to or if you passed a `read` token.
- `HfHubHTTPError` -- 
  HTTP 404 if the user does not exist on the Hub.
- `HfHubHTTPError` -- 
  HTTP 404 if the user access request cannot be found.
- `HfHubHTTPError` -- 
  HTTP 404 if the user access request is already in the pending list.</raises><raisederrors>`HfHubHTTPError`</raisederrors></docstring>

Cancel an access request from a user for a given gated repo.

A cancelled request will go back to the pending list and the user will lose access to the repo.

For more info about gated repos, see https://huggingface.co/docs/hub/models-gated.








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>cancel_job</name><anchor>huggingface_hub.HfApi.cancel_job</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L10101</source><parameters>[{"name": "job_id", "val": ": str"}, {"name": "namespace", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **job_id** (`str`) --
  ID of the Job.

- **namespace** (`str`, *optional*) --
  The namespace where the Job is running. Defaults to the current user's namespace.

- **token** `(Union[bool, str, None]`, *optional*) --
  A valid user access token. If not provided, the locally saved token will be used, which is the
  recommended authentication method. Set to `False` to disable authentication.
  Refer to: https://huggingface.co/docs/huggingface_hub/quick-start#authentication.</paramsdesc><paramgroups>0</paramgroups></docstring>

Cancel a compute Job on Hugging Face infrastructure.




</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>change_discussion_status</name><anchor>huggingface_hub.HfApi.change_discussion_status</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L6525</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "discussion_num", "val": ": int"}, {"name": "new_status", "val": ": Literal['open', 'closed']"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "comment", "val": ": Optional[str] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated
  by a `/`.
- **discussion_num** (`int`) --
  The number of the Discussion or Pull Request . Must be a strictly positive integer.
- **new_status** (`str`) --
  The new status for the discussion, either `"open"` or `"closed"`.
- **comment** (`str`, *optional*) --
  An optional comment to post with the status change.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if uploading to a dataset or
  space, `None` or `"model"` if uploading to a model. Default is
  `None`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[DiscussionStatusChange](/docs/huggingface_hub/main/en/package_reference/community#huggingface_hub.DiscussionStatusChange)</rettype><retdesc>the status change event</retdesc></docstring>
Closes or re-opens a Discussion or Pull Request.







<ExampleCodeBlock anchor="huggingface_hub.HfApi.change_discussion_status.example">

Examples:
```python
>>> new_title = "New title, fixing a typo"
>>> HfApi().rename_discussion(
...     repo_id="username/repo_name",
...     discussion_num=34
...     new_title=new_title
... )
# DiscussionStatusChange(id='deadbeef0000000', type='status-change', ...)

```

</ExampleCodeBlock>

> [!TIP]
> Raises the following errors:
>
>     - [`HTTPError`](https://requests.readthedocs.io/en/latest/api/#requests.HTTPError)
>       if the HuggingFace API returned an error
>     - [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError)
>       if some parameter value is invalid
>     - [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError)
>       If the repository to download from cannot be found. This may be because it doesn't exist,
>       or because it is set to `private` and you do not have access.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>comment_discussion</name><anchor>huggingface_hub.HfApi.comment_discussion</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L6382</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "discussion_num", "val": ": int"}, {"name": "comment", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated
  by a `/`.
- **discussion_num** (`int`) --
  The number of the Discussion or Pull Request . Must be a strictly positive integer.
- **comment** (`str`) --
  The content of the comment to create. Comments support markdown formatting.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if uploading to a dataset or
  space, `None` or `"model"` if uploading to a model. Default is
  `None`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[DiscussionComment](/docs/huggingface_hub/main/en/package_reference/community#huggingface_hub.DiscussionComment)</rettype><retdesc>the newly created comment</retdesc></docstring>
Creates a new comment on the given Discussion.







<ExampleCodeBlock anchor="huggingface_hub.HfApi.comment_discussion.example">

Examples:
```python

>>> comment = """
... Hello @otheruser!
...
... # This is a title
...
... **This is bold**, *this is italic* and ~this is strikethrough~
... And [this](http://url) is a link
... """

>>> HfApi().comment_discussion(
...     repo_id="username/repo_name",
...     discussion_num=34
...     comment=comment
... )
# DiscussionComment(id='deadbeef0000000', type='comment', ...)

```

</ExampleCodeBlock>

> [!TIP]
> Raises the following errors:
>
>     - [`HTTPError`](https://requests.readthedocs.io/en/latest/api/#requests.HTTPError)
>       if the HuggingFace API returned an error
>     - [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError)
>       if some parameter value is invalid
>     - [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError)
>       If the repository to download from cannot be found. This may be because it doesn't exist,
>       or because it is set to `private` and you do not have access.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>create_branch</name><anchor>huggingface_hub.HfApi.create_branch</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L5732</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "branch", "val": ": str"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "exist_ok", "val": ": bool = False"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  The repository in which the branch will be created.
  Example: `"user/my-cool-model"`.

- **branch** (`str`) --
  The name of the branch to create.

- **revision** (`str`, *optional*) --
  The git revision to create the branch from. It can be a branch name or
  the OID/SHA of a commit, as a hexadecimal string. Defaults to the head
  of the `"main"` branch.

- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.

- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if creating a branch on a dataset or
  space, `None` or `"model"` if tagging a model. Default is `None`.

- **exist_ok** (`bool`, *optional*, defaults to `False`) --
  If `True`, do not raise an error if branch already exists.</paramsdesc><paramgroups>0</paramgroups><raises>- [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) -- 
  If repository is not found (error 404): wrong repo_id/repo_type, private
  but not authenticated or repo does not exist.
- [BadRequestError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.BadRequestError) -- 
  If invalid reference for a branch. Ex: `refs/pr/5` or 'refs/foo/bar'.
- [HfHubHTTPError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.HfHubHTTPError) -- 
  If the branch already exists on the repo (error 409) and `exist_ok` is
  set to `False`.</raises><raisederrors>[RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) or [BadRequestError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.BadRequestError) or [HfHubHTTPError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.HfHubHTTPError)</raisederrors></docstring>

Create a new branch for a repo on the Hub, starting from the specified revision (defaults to `main`).
To find a revision suiting your needs, you can use [list_repo_refs()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.list_repo_refs) or [list_repo_commits()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.list_repo_commits).








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>create_collection</name><anchor>huggingface_hub.HfApi.create_collection</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L8128</source><parameters>[{"name": "title", "val": ": str"}, {"name": "namespace", "val": ": Optional[str] = None"}, {"name": "description", "val": ": Optional[str] = None"}, {"name": "private", "val": ": bool = False"}, {"name": "exists_ok", "val": ": bool = False"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **title** (`str`) --
  Title of the collection to create. Example: `"Recent models"`.
- **namespace** (`str`, *optional*) --
  Namespace of the collection to create (username or org). Will default to the owner name.
- **description** (`str`, *optional*) --
  Description of the collection to create.
- **private** (`bool`, *optional*) --
  Whether the collection should be private or not. Defaults to `False` (i.e. public collection).
- **exists_ok** (`bool`, *optional*) --
  If `True`, do not raise an error if collection already exists.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups></docstring>
Create a new Collection on the Hub.



Returns: [Collection](/docs/huggingface_hub/main/en/package_reference/collections#huggingface_hub.Collection)

<ExampleCodeBlock anchor="huggingface_hub.HfApi.create_collection.example">

Example:

```py
>>> from huggingface_hub import create_collection
>>> collection = create_collection(
...     title="ICCV 2023",
...     description="Portfolio of models, papers and demos I presented at ICCV 2023",
... )
>>> collection.slug
"username/iccv-2023-64f9a55bb3115b4f513ec026"
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>create_commit</name><anchor>huggingface_hub.HfApi.create_commit</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L3960</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "operations", "val": ": Iterable[CommitOperation]"}, {"name": "commit_message", "val": ": str"}, {"name": "commit_description", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[str, bool, None] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "create_pr", "val": ": Optional[bool] = None"}, {"name": "num_threads", "val": ": int = 5"}, {"name": "parent_commit", "val": ": Optional[str] = None"}, {"name": "run_as_future", "val": ": bool = False"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  The repository in which the commit will be created, for example:
  `"username/custom_transformers"`

- **operations** (`Iterable` of `CommitOperation()`) --
  An iterable of operations to include in the commit, either:

  - [CommitOperationAdd](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.CommitOperationAdd) to upload a file
  - [CommitOperationDelete](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.CommitOperationDelete) to delete a file
  - [CommitOperationCopy](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.CommitOperationCopy) to copy a file

  Operation objects will be mutated to include information relative to the upload. Do not reuse the
  same objects for multiple commits.

- **commit_message** (`str`) --
  The summary (first line) of the commit that will be created.

- **commit_description** (`str`, *optional*) --
  The description of the commit that will be created

- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.

- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if uploading to a dataset or
  space, `None` or `"model"` if uploading to a model. Default is
  `None`.

- **revision** (`str`, *optional*) --
  The git revision to commit from. Defaults to the head of the `"main"` branch.

- **create_pr** (`boolean`, *optional*) --
  Whether or not to create a Pull Request with that commit. Defaults to `False`.
  If `revision` is not set, PR is opened against the `"main"` branch. If
  `revision` is set and is a branch, PR is opened against this branch. If
  `revision` is set and is not a branch name (example: a commit oid), an
  `RevisionNotFoundError` is returned by the server.

- **num_threads** (`int`, *optional*) --
  Number of concurrent threads for uploading files. Defaults to 5.
  Setting it to 2 means at most 2 files will be uploaded concurrently.

- **parent_commit** (`str`, *optional*) --
  The OID / SHA of the parent commit, as a hexadecimal string.
  Shorthands (7 first characters) are also supported. If specified and `create_pr` is `False`,
  the commit will fail if `revision` does not point to `parent_commit`. If specified and `create_pr`
  is `True`, the pull request will be created from `parent_commit`. Specifying `parent_commit`
  ensures the repo has not changed before committing the changes, and can be especially useful
  if the repo is updated / committed to concurrently.
- **run_as_future** (`bool`, *optional*) --
  Whether or not to run this method in the background. Background jobs are run sequentially without
  blocking the main thread. Passing `run_as_future=True` will return a [Future](https://docs.python.org/3/library/concurrent.futures.html#future-objects)
  object. Defaults to `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[CommitInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.CommitInfo) or `Future`</rettype><retdesc>Instance of [CommitInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.CommitInfo) containing information about the newly created commit (commit hash, commit
url, pr url, commit message,...). If `run_as_future=True` is passed, returns a Future object which will
contain the result when executed.</retdesc><raises>- [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError) -- 
  If commit message is empty.
- [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError) -- 
  If parent commit is not a valid commit OID.
- [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError) -- 
  If a README.md file with an invalid metadata section is committed. In this case, the commit will fail
  early, before trying to upload any file.
- [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError) -- 
  If `create_pr` is `True` and revision is neither `None` nor `"main"`.
- [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) -- 
  If repository is not found (error 404): wrong repo_id/repo_type, private
  but not authenticated or repo does not exist.</raises><raisederrors>``ValueError`` or [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError)</raisederrors></docstring>

Creates a commit in the given repo, deleting & uploading files as needed.

> [!WARNING]
> The input list of `CommitOperation` will be mutated during the commit process. Do not reuse the same objects
> for multiple commits.

> [!WARNING]
> `create_commit` assumes that the repo already exists on the Hub. If you get a
> Client error 404, please make sure you are authenticated and that `repo_id` and
> `repo_type` are set correctly. If repo does not exist, create it first using
> [create_repo()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.create_repo).

> [!WARNING]
> `create_commit` is limited to 25k LFS files and a 1GB payload for regular files.












</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>create_discussion</name><anchor>huggingface_hub.HfApi.create_discussion</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L6209</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "title", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "description", "val": ": Optional[str] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "pull_request", "val": ": bool = False"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated
  by a `/`.
- **title** (`str`) --
  The title of the discussion. It can be up to 200 characters long,
  and must be at least 3 characters long. Leading and trailing whitespaces
  will be stripped.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.
- **description** (`str`, *optional*) --
  An optional description for the Pull Request.
  Defaults to `"Discussion opened with the huggingface_hub Python library"`
- **pull_request** (`bool`, *optional*) --
  Whether to create a Pull Request or discussion. If `True`, creates a Pull Request.
  If `False`, creates a discussion. Defaults to `False`.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if uploading to a dataset or
  space, `None` or `"model"` if uploading to a model. Default is
  `None`.</paramsdesc><paramgroups>0</paramgroups></docstring>
Creates a Discussion or Pull Request.

Pull Requests created programmatically will be in `"draft"` status.

Creating a Pull Request with changes can also be done at once with [HfApi.create_commit()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.create_commit).



Returns: [DiscussionWithDetails](/docs/huggingface_hub/main/en/package_reference/community#huggingface_hub.DiscussionWithDetails)

> [!TIP]
> Raises the following errors:
>
>     - [`HTTPError`](https://requests.readthedocs.io/en/latest/api/#requests.HTTPError)
>       if the HuggingFace API returned an error
>     - [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError)
>       if some parameter value is invalid
>     - [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError)
>       If the repository to download from cannot be found. This may be because it doesn't exist,
>       or because it is set to `private` and you do not have access.

</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>create_inference_endpoint</name><anchor>huggingface_hub.HfApi.create_inference_endpoint</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L7380</source><parameters>[{"name": "name", "val": ": str"}, {"name": "repository", "val": ": str"}, {"name": "framework", "val": ": str"}, {"name": "accelerator", "val": ": str"}, {"name": "instance_size", "val": ": str"}, {"name": "instance_type", "val": ": str"}, {"name": "region", "val": ": str"}, {"name": "vendor", "val": ": str"}, {"name": "account_id", "val": ": Optional[str] = None"}, {"name": "min_replica", "val": ": int = 1"}, {"name": "max_replica", "val": ": int = 1"}, {"name": "scale_to_zero_timeout", "val": ": Optional[int] = None"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "task", "val": ": Optional[str] = None"}, {"name": "custom_image", "val": ": Optional[dict] = None"}, {"name": "env", "val": ": Optional[dict[str, str]] = None"}, {"name": "secrets", "val": ": Optional[dict[str, str]] = None"}, {"name": "type", "val": ": InferenceEndpointType = <InferenceEndpointType.PROTECTED: 'protected'>"}, {"name": "domain", "val": ": Optional[str] = None"}, {"name": "path", "val": ": Optional[str] = None"}, {"name": "cache_http_responses", "val": ": Optional[bool] = None"}, {"name": "tags", "val": ": Optional[list[str]] = None"}, {"name": "namespace", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **name** (`str`) --
  The unique name for the new Inference Endpoint.
- **repository** (`str`) --
  The name of the model repository associated with the Inference Endpoint (e.g. `"gpt2"`).
- **framework** (`str`) --
  The machine learning framework used for the model (e.g. `"custom"`).
- **accelerator** (`str`) --
  The hardware accelerator to be used for inference (e.g. `"cpu"`).
- **instance_size** (`str`) --
  The size or type of the instance to be used for hosting the model (e.g. `"x4"`).
- **instance_type** (`str`) --
  The cloud instance type where the Inference Endpoint will be deployed (e.g. `"intel-icl"`).
- **region** (`str`) --
  The cloud region in which the Inference Endpoint will be created (e.g. `"us-east-1"`).
- **vendor** (`str`) --
  The cloud provider or vendor where the Inference Endpoint will be hosted (e.g. `"aws"`).
- **account_id** (`str`, *optional*) --
  The account ID used to link a VPC to a private Inference Endpoint (if applicable).
- **min_replica** (`int`, *optional*) --
  The minimum number of replicas (instances) to keep running for the Inference Endpoint. To enable
  scaling to zero, set this value to 0 and adjust `scale_to_zero_timeout` accordingly. Defaults to 1.
- **max_replica** (`int`, *optional*) --
  The maximum number of replicas (instances) to scale to for the Inference Endpoint. Defaults to 1.
- **scale_to_zero_timeout** (`int`, *optional*) --
  The duration in minutes before an inactive endpoint is scaled to zero, or no scaling to zero if
  set to None and `min_replica` is not 0. Defaults to None.
- **revision** (`str`, *optional*) --
  The specific model revision to deploy on the Inference Endpoint (e.g. `"6c0e6080953db56375760c0471a8c5f2929baf11"`).
- **task** (`str`, *optional*) --
  The task on which to deploy the model (e.g. `"text-classification"`).
- **custom_image** (`dict`, *optional*) --
  A custom Docker image to use for the Inference Endpoint. This is useful if you want to deploy an
  Inference Endpoint running on the `text-generation-inference` (TGI) framework (see examples).
- **env** (`dict[str, str]`, *optional*) --
  Non-secret environment variables to inject in the container environment.
- **secrets** (`dict[str, str]`, *optional*) --
  Secret values to inject in the container environment.
- **type** ([`InferenceEndpointType]`, *optional*) --
  The type of the Inference Endpoint, which can be `"protected"` (default), `"public"` or `"private"`.
- **domain** (`str`, *optional*) --
  The custom domain for the Inference Endpoint deployment, if setup the inference endpoint will be available at this domain (e.g. `"my-new-domain.cool-website.woof"`).
- **path** (`str`, *optional*) --
  The custom path to the deployed model, should start with a `/` (e.g. `"/models/google-bert/bert-base-uncased"`).
- **cache_http_responses** (`bool`, *optional*) --
  Whether to cache HTTP responses from the Inference Endpoint. Defaults to `False`.
- **tags** (`list[str]`, *optional*) --
  A list of tags to associate with the Inference Endpoint.
- **namespace** (`str`, *optional*) --
  The namespace where the Inference Endpoint will be created. Defaults to the current user's namespace.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[InferenceEndpoint](/docs/huggingface_hub/main/en/package_reference/inference_endpoints#huggingface_hub.InferenceEndpoint)</rettype><retdesc>information about the updated Inference Endpoint.</retdesc></docstring>
Create a new Inference Endpoint.







<ExampleCodeBlock anchor="huggingface_hub.HfApi.create_inference_endpoint.example">

Example:
```python
>>> from huggingface_hub import HfApi
>>> api = HfApi()
>>> endpoint = api.create_inference_endpoint(
...     "my-endpoint-name",
...     repository="gpt2",
...     framework="pytorch",
...     task="text-generation",
...     accelerator="cpu",
...     vendor="aws",
...     region="us-east-1",
...     type="protected",
...     instance_size="x2",
...     instance_type="intel-icl",
... )
>>> endpoint
InferenceEndpoint(name='my-endpoint-name', status="pending",...)

# Run inference on the endpoint
>>> endpoint.client.text_generation(...)
"..."
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.HfApi.create_inference_endpoint.example-2">

```python
# Start an Inference Endpoint running Zephyr-7b-beta on TGI
>>> from huggingface_hub import HfApi
>>> api = HfApi()
>>> endpoint = api.create_inference_endpoint(
...     "aws-zephyr-7b-beta-0486",
...     repository="HuggingFaceH4/zephyr-7b-beta",
...     framework="pytorch",
...     task="text-generation",
...     accelerator="gpu",
...     vendor="aws",
...     region="us-east-1",
...     type="protected",
...     instance_size="x1",
...     instance_type="nvidia-a10g",
...     env={
...           "MAX_BATCH_PREFILL_TOKENS": "2048",
...           "MAX_INPUT_LENGTH": "1024",
...           "MAX_TOTAL_TOKENS": "1512",
...           "MODEL_ID": "/repository"
...         },
...     custom_image={
...         "health_route": "/health",
...         "url": "ghcr.io/huggingface/text-generation-inference:1.1.0",
...     },
...    secrets={"MY_SECRET_KEY": "secret_value"},
...    tags=["dev", "text-generation"],
... )
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.HfApi.create_inference_endpoint.example-3">

```python
# Start an Inference Endpoint running ProsusAI/finbert while scaling to zero in 15 minutes
>>> from huggingface_hub import HfApi
>>> api = HfApi()
>>> endpoint = api.create_inference_endpoint(
...     "finbert-classifier",
...     repository="ProsusAI/finbert",
...     framework="pytorch",
...     task="text-classification",
...     min_replica=0,
...     scale_to_zero_timeout=15,
...     accelerator="cpu",
...     vendor="aws",
...     region="us-east-1",
...     type="protected",
...     instance_size="x2",
...     instance_type="intel-icl",
... )
>>> endpoint.wait(timeout=300)
# Run inference on the endpoint
>>> endpoint.client.text_generation(...)
TextClassificationOutputElement(label='positive', score=0.8983615040779114)
```

</ExampleCodeBlock>



</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>create_inference_endpoint_from_catalog</name><anchor>huggingface_hub.HfApi.create_inference_endpoint_from_catalog</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L7609</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "name", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "namespace", "val": ": Optional[str] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  The ID of the model in the catalog to deploy as an Inference Endpoint.
- **name** (`str`, *optional*) --
  The unique name for the new Inference Endpoint. If not provided, a random name will be generated.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
- **namespace** (`str`, *optional*) --
  The namespace where the Inference Endpoint will be created. Defaults to the current user's namespace.</paramsdesc><paramgroups>0</paramgroups><rettype>[InferenceEndpoint](/docs/huggingface_hub/main/en/package_reference/inference_endpoints#huggingface_hub.InferenceEndpoint)</rettype><retdesc>information about the new Inference Endpoint.</retdesc></docstring>
Create a new Inference Endpoint from a model in the Hugging Face Inference Catalog.

The goal of the Inference Catalog is to provide a curated list of models that are optimized for inference
and for which default configurations have been tested. See https://endpoints.huggingface.co/catalog for a list
of available models in the catalog.







> [!WARNING]
> `create_inference_endpoint_from_catalog` is experimental. Its API is subject to change in the future. Please provide feedback
> if you have any suggestions or requests.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>create_pull_request</name><anchor>huggingface_hub.HfApi.create_pull_request</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L6298</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "title", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "description", "val": ": Optional[str] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated
  by a `/`.
- **title** (`str`) --
  The title of the discussion. It can be up to 200 characters long,
  and must be at least 3 characters long. Leading and trailing whitespaces
  will be stripped.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.
- **description** (`str`, *optional*) --
  An optional description for the Pull Request.
  Defaults to `"Discussion opened with the huggingface_hub Python library"`
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if uploading to a dataset or
  space, `None` or `"model"` if uploading to a model. Default is
  `None`.</paramsdesc><paramgroups>0</paramgroups></docstring>
Creates a Pull Request . Pull Requests created programmatically will be in `"draft"` status.

Creating a Pull Request with changes can also be done at once with [HfApi.create_commit()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.create_commit);

This is a wrapper around [HfApi.create_discussion()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.create_discussion).



Returns: [DiscussionWithDetails](/docs/huggingface_hub/main/en/package_reference/community#huggingface_hub.DiscussionWithDetails)

> [!TIP]
> Raises the following errors:
>
>     - [`HTTPError`](https://requests.readthedocs.io/en/latest/api/#requests.HTTPError)
>       if the HuggingFace API returned an error
>     - [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError)
>       if some parameter value is invalid
>     - [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError)
>       If the repository to download from cannot be found. This may be because it doesn't exist,
>       or because it is set to `private` and you do not have access.

</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>create_repo</name><anchor>huggingface_hub.HfApi.create_repo</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L3595</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "token", "val": ": Union[str, bool, None] = None"}, {"name": "private", "val": ": Optional[bool] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "exist_ok", "val": ": bool = False"}, {"name": "resource_group_id", "val": ": Optional[str] = None"}, {"name": "space_sdk", "val": ": Optional[str] = None"}, {"name": "space_hardware", "val": ": Optional[SpaceHardware] = None"}, {"name": "space_storage", "val": ": Optional[SpaceStorage] = None"}, {"name": "space_sleep_time", "val": ": Optional[int] = None"}, {"name": "space_secrets", "val": ": Optional[list[dict[str, str]]] = None"}, {"name": "space_variables", "val": ": Optional[list[dict[str, str]]] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated
  by a `/`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.
- **private** (`bool`, *optional*) --
  Whether to make the repo private. If `None` (default), the repo will be public unless the organization's default is private. This value is ignored if the repo already exists.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if uploading to a dataset or
  space, `None` or `"model"` if uploading to a model. Default is
  `None`.
- **exist_ok** (`bool`, *optional*, defaults to `False`) --
  If `True`, do not raise an error if repo already exists.
- **resource_group_id** (`str`, *optional*) --
  Resource group in which to create the repo. Resource groups is only available for Enterprise Hub organizations and
  allow to define which members of the organization can access the resource. The ID of a resource group
  can be found in the URL of the resource's page on the Hub (e.g. `"66670e5163145ca562cb1988"`).
  To learn more about resource groups, see https://huggingface.co/docs/hub/en/security-resource-groups.
- **space_sdk** (`str`, *optional*) --
  Choice of SDK to use if repo_type is "space". Can be "streamlit", "gradio", "docker", or "static".
- **space_hardware** (`SpaceHardware` or `str`, *optional*) --
  Choice of Hardware if repo_type is "space". See [SpaceHardware](/docs/huggingface_hub/main/en/package_reference/space_runtime#huggingface_hub.SpaceHardware) for a complete list.
- **space_storage** (`SpaceStorage` or `str`, *optional*) --
  Choice of persistent storage tier. Example: `"small"`. See [SpaceStorage](/docs/huggingface_hub/main/en/package_reference/space_runtime#huggingface_hub.SpaceStorage) for a complete list.
- **space_sleep_time** (`int`, *optional*) --
  Number of seconds of inactivity to wait before a Space is put to sleep. Set to `-1` if you don't want
  your Space to sleep (default behavior for upgraded hardware). For free hardware, you can't configure
  the sleep time (value is fixed to 48 hours of inactivity).
  See https://huggingface.co/docs/hub/spaces-gpus#sleep-time for more details.
- **space_secrets** (`list[dict[str, str]]`, *optional*) --
  A list of secret keys to set in your Space. Each item is in the form `{"key": ..., "value": ..., "description": ...}` where description is optional.
  For more details, see https://huggingface.co/docs/hub/spaces-overview#managing-secrets.
- **space_variables** (`list[dict[str, str]]`, *optional*) --
  A list of public environment variables to set in your Space. Each item is in the form `{"key": ..., "value": ..., "description": ...}` where description is optional.
  For more details, see https://huggingface.co/docs/hub/spaces-overview#managing-secrets-and-environment-variables.</paramsdesc><paramgroups>0</paramgroups><rettype>[RepoUrl](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.RepoUrl)</rettype><retdesc>URL to the newly created repo. Value is a subclass of `str` containing
attributes like `endpoint`, `repo_type` and `repo_id`.</retdesc></docstring>
Create an empty repo on the HuggingFace Hub.








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>create_scheduled_job</name><anchor>huggingface_hub.HfApi.create_scheduled_job</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L10245</source><parameters>[{"name": "image", "val": ": str"}, {"name": "command", "val": ": list[str]"}, {"name": "schedule", "val": ": str"}, {"name": "suspend", "val": ": Optional[bool] = None"}, {"name": "concurrency", "val": ": Optional[bool] = None"}, {"name": "env", "val": ": Optional[dict[str, Any]] = None"}, {"name": "secrets", "val": ": Optional[dict[str, Any]] = None"}, {"name": "flavor", "val": ": Optional[SpaceHardware] = None"}, {"name": "timeout", "val": ": Optional[Union[int, float, str]] = None"}, {"name": "namespace", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **image** (`str`) --
  The Docker image to use.
  Examples: `"ubuntu"`, `"python:3.12"`, `"pytorch/pytorch:2.6.0-cuda12.4-cudnn9-devel"`.
  Example with an image from a Space: `"hf.co/spaces/lhoestq/duckdb"`.

- **command** (`list[str]`) --
  The command to run. Example: `["echo", "hello"]`.

- **schedule** (`str`) --
  One of "@annually", "@yearly", "@monthly", "@weekly", "@daily", "@hourly", or a
  CRON schedule expression (e.g., '0 9 * * 1' for 9 AM every Monday).

- **suspend** (`bool`, *optional*) --
  If True, the scheduled Job is suspended (paused).  Defaults to False.

- **concurrency** (`bool`, *optional*) --
  If True, multiple instances of this Job can run concurrently. Defaults to False.

- **env** (`dict[str, Any]`, *optional*) --
  Defines the environment variables for the Job.

- **secrets** (`dict[str, Any]`, *optional*) --
  Defines the secret environment variables for the Job.

- **flavor** (`str`, *optional*) --
  Flavor for the hardware, as in Hugging Face Spaces. See [SpaceHardware](/docs/huggingface_hub/main/en/package_reference/space_runtime#huggingface_hub.SpaceHardware) for possible values.
  Defaults to `"cpu-basic"`.

- **timeout** (`Union[int, float, str]`, *optional*) --
  Max duration for the Job: int/float with s (seconds, default), m (minutes), h (hours) or d (days).
  Example: `300` or `"5m"` for 5 minutes.

- **namespace** (`str`, *optional*) --
  The namespace where the Job will be created. Defaults to the current user's namespace.

- **token** `(Union[bool, str, None]`, *optional*) --
  A valid user access token. If not provided, the locally saved token will be used, which is the
  recommended authentication method. Set to `False` to disable authentication.
  Refer to: https://huggingface.co/docs/huggingface_hub/quick-start#authentication.</paramsdesc><paramgroups>0</paramgroups></docstring>

Create scheduled compute Jobs on Hugging Face infrastructure.



Example:
<ExampleCodeBlock anchor="huggingface_hub.HfApi.create_scheduled_job.example">

Create your first scheduled Job:

```python
>>> from huggingface_hub import create_scheduled_job
>>> create_scheduled_job(image="python:3.12", command=["python", "-c" ,"print('Hello from HF compute!')"], schedule="@hourly")
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.HfApi.create_scheduled_job.example-2">

Use a CRON schedule expression:

```python
>>> from huggingface_hub import create_scheduled_job
>>> create_scheduled_job(image="python:3.12", command=["python", "-c" ,"print('this runs every 5min')"], schedule="*/5 * * * *")
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.HfApi.create_scheduled_job.example-3">

Create a scheduled GPU Job:

```python
>>> from huggingface_hub import create_scheduled_job
>>> image = "pytorch/pytorch:2.6.0-cuda12.4-cudnn9-devel"
>>> command = ["python", "-c", "import torch; print(f"This code ran with the following GPU: {torch.cuda.get_device_name()}")"]
>>> create_scheduled_job(image, command, flavor="a10g-small", schedule="@hourly")
```

</ExampleCodeBlock>



</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>create_scheduled_uv_job</name><anchor>huggingface_hub.HfApi.create_scheduled_uv_job</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L10517</source><parameters>[{"name": "script", "val": ": str"}, {"name": "script_args", "val": ": Optional[list[str]] = None"}, {"name": "schedule", "val": ": str"}, {"name": "suspend", "val": ": Optional[bool] = None"}, {"name": "concurrency", "val": ": Optional[bool] = None"}, {"name": "dependencies", "val": ": Optional[list[str]] = None"}, {"name": "python", "val": ": Optional[str] = None"}, {"name": "image", "val": ": Optional[str] = None"}, {"name": "env", "val": ": Optional[dict[str, Any]] = None"}, {"name": "secrets", "val": ": Optional[dict[str, Any]] = None"}, {"name": "flavor", "val": ": Optional[SpaceHardware] = None"}, {"name": "timeout", "val": ": Optional[Union[int, float, str]] = None"}, {"name": "namespace", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "_repo", "val": ": Optional[str] = None"}]</parameters><paramsdesc>- **script** (`str`) --
  Path or URL of the UV script, or a command.

- **script_args** (`list[str]`, *optional*) --
  Arguments to pass to the script, or a command.

- **schedule** (`str`) --
  One of "@annually", "@yearly", "@monthly", "@weekly", "@daily", "@hourly", or a
  CRON schedule expression (e.g., '0 9 * * 1' for 9 AM every Monday).

- **suspend** (`bool`, *optional*) --
  If True, the scheduled Job is suspended (paused).  Defaults to False.

- **concurrency** (`bool`, *optional*) --
  If True, multiple instances of this Job can run concurrently. Defaults to False.

- **dependencies** (`list[str]`, *optional*) --
  Dependencies to use to run the UV script.

- **python** (`str`, *optional*) --
  Use a specific Python version. Default is 3.12.

- **image** (`str`, *optional*, defaults to "ghcr.io/astral-sh/uv --python3.12-bookworm"):
  Use a custom Docker image with `uv` installed.

- **env** (`dict[str, Any]`, *optional*) --
  Defines the environment variables for the Job.

- **secrets** (`dict[str, Any]`, *optional*) --
  Defines the secret environment variables for the Job.

- **flavor** (`str`, *optional*) --
  Flavor for the hardware, as in Hugging Face Spaces. See [SpaceHardware](/docs/huggingface_hub/main/en/package_reference/space_runtime#huggingface_hub.SpaceHardware) for possible values.
  Defaults to `"cpu-basic"`.

- **timeout** (`Union[int, float, str]`, *optional*) --
  Max duration for the Job: int/float with s (seconds, default), m (minutes), h (hours) or d (days).
  Example: `300` or `"5m"` for 5 minutes.

- **namespace** (`str`, *optional*) --
  The namespace where the Job will be created. Defaults to the current user's namespace.

- **token** `(Union[bool, str, None]`, *optional*) --
  A valid user access token. If not provided, the locally saved token will be used, which is the
  recommended authentication method. Set to `False` to disable authentication.
  Refer to: https://huggingface.co/docs/huggingface_hub/quick-start#authentication.</paramsdesc><paramgroups>0</paramgroups></docstring>

Run a UV script Job on Hugging Face infrastructure.



Example:

<ExampleCodeBlock anchor="huggingface_hub.HfApi.create_scheduled_uv_job.example">

Schedule a script from a URL:

```python
>>> from huggingface_hub import create_scheduled_uv_job
>>> script = "https://raw.githubusercontent.com/huggingface/trl/refs/heads/main/trl/scripts/sft.py"
>>> script_args = ["--model_name_or_path", "Qwen/Qwen2-0.5B", "--dataset_name", "trl-lib/Capybara", "--push_to_hub"]
>>> create_scheduled_uv_job(script, script_args=script_args, dependencies=["trl"], flavor="a10g-small", schedule="@weekly")
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.HfApi.create_scheduled_uv_job.example-2">

Schedule a local script:

```python
>>> from huggingface_hub import create_scheduled_uv_job
>>> script = "my_sft.py"
>>> script_args = ["--model_name_or_path", "Qwen/Qwen2-0.5B", "--dataset_name", "trl-lib/Capybara", "--push_to_hub"]
>>> create_scheduled_uv_job(script, script_args=script_args, dependencies=["trl"], flavor="a10g-small", schedule="@weekly")
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.HfApi.create_scheduled_uv_job.example-3">

Schedule a command:

```python
>>> from huggingface_hub import create_scheduled_uv_job
>>> script = "lighteval"
>>> script_args= ["endpoint", "inference-providers", "model_name=openai/gpt-oss-20b,provider=auto", "lighteval|gsm8k|0|0"]
>>> create_scheduled_uv_job(script, script_args=script_args, dependencies=["lighteval"], flavor="a10g-small", schedule="@weekly")
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>create_tag</name><anchor>huggingface_hub.HfApi.create_tag</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L5864</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "tag", "val": ": str"}, {"name": "tag_message", "val": ": Optional[str] = None"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "exist_ok", "val": ": bool = False"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  The repository in which a commit will be tagged.
  Example: `"user/my-cool-model"`.

- **tag** (`str`) --
  The name of the tag to create.

- **tag_message** (`str`, *optional*) --
  The description of the tag to create.

- **revision** (`str`, *optional*) --
  The git revision to tag. It can be a branch name or the OID/SHA of a
  commit, as a hexadecimal string. Shorthands (7 first characters) are
  also supported. Defaults to the head of the `"main"` branch.

- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.

- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if tagging a dataset or
  space, `None` or `"model"` if tagging a model. Default is
  `None`.

- **exist_ok** (`bool`, *optional*, defaults to `False`) --
  If `True`, do not raise an error if tag already exists.</paramsdesc><paramgroups>0</paramgroups><raises>- [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) -- 
  If repository is not found (error 404): wrong repo_id/repo_type, private
  but not authenticated or repo does not exist.
- [RevisionNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RevisionNotFoundError) -- 
  If revision is not found (error 404) on the repo.
- [HfHubHTTPError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.HfHubHTTPError) -- 
  If the branch already exists on the repo (error 409) and `exist_ok` is
  set to `False`.</raises><raisederrors>[RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) or [RevisionNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RevisionNotFoundError) or [HfHubHTTPError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.HfHubHTTPError)</raisederrors></docstring>

Tag a given commit of a repo on the Hub.








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>create_webhook</name><anchor>huggingface_hub.HfApi.create_webhook</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L9032</source><parameters>[{"name": "url", "val": ": Optional[str] = None"}, {"name": "job_id", "val": ": Optional[str] = None"}, {"name": "watched", "val": ": list[Union[dict, WebhookWatchedItem]]"}, {"name": "domains", "val": ": Optional[list[constants.WEBHOOK_DOMAIN_T]] = None"}, {"name": "secret", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **url** (`str`) --
  URL to send the payload to.
- **job_id** (`str`) --
  ID of the source Job to trigger with the webhook payload in the environment variable WEBHOOK_PAYLOAD.
  Additional environment variables are available for convenience: WEBHOOK_REPO_ID, WEBHOOK_REPO_TYPE and WEBHOOK_SECRET.
- **watched** (`list[WebhookWatchedItem]`) --
  List of [WebhookWatchedItem](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.WebhookWatchedItem) to be watched by the webhook. It can be users, orgs, models, datasets or spaces.
  Watched items can also be provided as plain dictionaries.
- **domains** (`list[Literal["repo", "discussion"]]`, optional) --
  List of domains to watch. It can be "repo", "discussion" or both.
- **secret** (`str`, optional) --
  A secret to sign the payload with.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved token, which is the recommended
  method for authentication (see https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[WebhookInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.WebhookInfo)</rettype><retdesc>Info about the newly created webhook.</retdesc></docstring>
Create a new webhook.

The webhook can either send a payload to a URL, or trigger a Job to run on Hugging Face infrastructure.
This function should be called with one of `url` or `job_id`, but not both.







Example:

Create a webhook that sends a payload to a URL
<ExampleCodeBlock anchor="huggingface_hub.HfApi.create_webhook.example">

```python
>>> from huggingface_hub import create_webhook
>>> payload = create_webhook(
...     watched=[{"type": "user", "name": "julien-c"}, {"type": "org", "name": "HuggingFaceH4"}],
...     url="https://webhook.site/a2176e82-5720-43ee-9e06-f91cb4c91548",
...     domains=["repo", "discussion"],
...     secret="my-secret",
... )
>>> print(payload)
WebhookInfo(
    id="654bbbc16f2ec14d77f109cc",
    url="https://webhook.site/a2176e82-5720-43ee-9e06-f91cb4c91548",
    job=None,
    watched=[WebhookWatchedItem(type="user", name="julien-c"), WebhookWatchedItem(type="org", name="HuggingFaceH4")],
    domains=["repo", "discussion"],
    secret="my-secret",
    disabled=False,
)
```

</ExampleCodeBlock>

Run a Job and then create a webhook that triggers this Job
<ExampleCodeBlock anchor="huggingface_hub.HfApi.create_webhook.example-2">

```python
>>> from huggingface_hub import create_webhook, run_job
>>> job = run_job(
...     image="ubuntu",
...     command=["bash", "-c", r"echo An event occured in $WEBHOOK_REPO_ID: $WEBHOOK_PAYLOAD"],
... )
>>> payload = create_webhook(
...     watched=[{"type": "user", "name": "julien-c"}, {"type": "org", "name": "HuggingFaceH4"}],
...     job_id=job.id,
...     domains=["repo", "discussion"],
...     secret="my-secret",
... )
>>> print(payload)
WebhookInfo(
    id="654bbbc16f2ec14d77f109cc",
    url=None,
    job=JobSpec(
        docker_image='ubuntu',
        space_id=None,
        command=['bash', '-c', 'echo An event occured in $WEBHOOK_REPO_ID: $WEBHOOK_PAYLOAD'],
        arguments=[],
        environment={},
        secrets=[],
        flavor='cpu-basic',
        timeout=None,
        tags=None,
        arch=None
    ),
    watched=[WebhookWatchedItem(type="user", name="julien-c"), WebhookWatchedItem(type="org", name="HuggingFaceH4")],
    domains=["repo", "discussion"],
    secret="my-secret",
    disabled=False,
)
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>dataset_info</name><anchor>huggingface_hub.HfApi.dataset_info</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L2556</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "timeout", "val": ": Optional[float] = None"}, {"name": "files_metadata", "val": ": bool = False"}, {"name": "expand", "val": ": Optional[list[ExpandDatasetProperty_T]] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated
  by a `/`.
- **revision** (`str`, *optional*) --
  The revision of the dataset repository from which to get the
  information.
- **timeout** (`float`, *optional*) --
  Whether to set a timeout for the request to the Hub.
- **files_metadata** (`bool`, *optional*) --
  Whether or not to retrieve metadata for files in the repository
  (size, LFS metadata, etc). Defaults to `False`.
- **expand** (`list[ExpandDatasetProperty_T]`, *optional*) --
  List properties to return in the response. When used, only the properties in the list will be returned.
  This parameter cannot be used if `files_metadata` is passed.
  Possible values are `"author"`, `"cardData"`, `"citation"`, `"createdAt"`, `"disabled"`, `"description"`, `"downloads"`, `"downloadsAllTime"`, `"gated"`, `"lastModified"`, `"likes"`, `"paperswithcode_id"`, `"private"`, `"siblings"`, `"sha"`, `"tags"`, `"trendingScore"`,`"usedStorage"`, and `"resourceGroup"`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[hf_api.DatasetInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.DatasetInfo)</rettype><retdesc>The dataset repository information.</retdesc></docstring>

Get info on one specific dataset on huggingface.co.

Dataset can be private if you pass an acceptable token.







> [!TIP]
> Raises the following errors:
>
>     - [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError)
>       If the repository to download from cannot be found. This may be because it doesn't exist,
>       or because it is set to `private` and you do not have access.
>     - [RevisionNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RevisionNotFoundError)
>       If the revision to download from cannot be found.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>delete_branch</name><anchor>huggingface_hub.HfApi.delete_branch</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L5812</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "branch", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  The repository in which a branch will be deleted.
  Example: `"user/my-cool-model"`.

- **branch** (`str`) --
  The name of the branch to delete.

- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.

- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if creating a branch on a dataset or
  space, `None` or `"model"` if tagging a model. Default is `None`.</paramsdesc><paramgroups>0</paramgroups><raises>- [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) -- 
  If repository is not found (error 404): wrong repo_id/repo_type, private
  but not authenticated or repo does not exist.
- [HfHubHTTPError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.HfHubHTTPError) -- 
  If trying to delete a protected branch. Ex: `main` cannot be deleted.
- [HfHubHTTPError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.HfHubHTTPError) -- 
  If trying to delete a branch that does not exist.</raises><raisederrors>[RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) or [HfHubHTTPError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.HfHubHTTPError)</raisederrors></docstring>

Delete a branch from a repo on the Hub.








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>delete_collection</name><anchor>huggingface_hub.HfApi.delete_collection</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L8264</source><parameters>[{"name": "collection_slug", "val": ": str"}, {"name": "missing_ok", "val": ": bool = False"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **collection_slug** (`str`) --
  Slug of the collection to delete. Example: `"TheBloke/recent-models-64f9a55bb3115b4f513ec026"`.
- **missing_ok** (`bool`, *optional*) --
  If `True`, do not raise an error if collection doesn't exists.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups></docstring>
Delete a collection on the Hub.



<ExampleCodeBlock anchor="huggingface_hub.HfApi.delete_collection.example">

Example:

```py
>>> from huggingface_hub import delete_collection
>>> collection = delete_collection("username/useless-collection-64f9a55bb3115b4f513ec026", missing_ok=True)
```

</ExampleCodeBlock>

> [!WARNING]
> This is a non-revertible action. A deleted collection cannot be restored.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>delete_collection_item</name><anchor>huggingface_hub.HfApi.delete_collection_item</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L8437</source><parameters>[{"name": "collection_slug", "val": ": str"}, {"name": "item_object_id", "val": ": str"}, {"name": "missing_ok", "val": ": bool = False"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **collection_slug** (`str`) --
  Slug of the collection to update. Example: `"TheBloke/recent-models-64f9a55bb3115b4f513ec026"`.
- **item_object_id** (`str`) --
  ID of the item in the collection. This is not the id of the item on the Hub (repo_id or paper id).
  It must be retrieved from a [CollectionItem](/docs/huggingface_hub/main/en/package_reference/collections#huggingface_hub.CollectionItem) object. Example: `collection.items[0].item_object_id`.
- **missing_ok** (`bool`, *optional*) --
  If `True`, do not raise an error if item doesn't exists.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups></docstring>
Delete an item from a collection.



<ExampleCodeBlock anchor="huggingface_hub.HfApi.delete_collection_item.example">

Example:

```py
>>> from huggingface_hub import get_collection, delete_collection_item

# Get collection first
>>> collection = get_collection("TheBloke/recent-models-64f9a55bb3115b4f513ec026")

# Delete item based on its ID
>>> delete_collection_item(
...     collection_slug="TheBloke/recent-models-64f9a55bb3115b4f513ec026",
...     item_object_id=collection.items[-1].item_object_id,
... )
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>delete_file</name><anchor>huggingface_hub.HfApi.delete_file</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L4819</source><parameters>[{"name": "path_in_repo", "val": ": str"}, {"name": "repo_id", "val": ": str"}, {"name": "token", "val": ": Union[str, bool, None] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "commit_message", "val": ": Optional[str] = None"}, {"name": "commit_description", "val": ": Optional[str] = None"}, {"name": "create_pr", "val": ": Optional[bool] = None"}, {"name": "parent_commit", "val": ": Optional[str] = None"}]</parameters><paramsdesc>- **path_in_repo** (`str`) --
  Relative filepath in the repo, for example:
  `"checkpoints/1fec34a/weights.bin"`
- **repo_id** (`str`) --
  The repository from which the file will be deleted, for example:
  `"username/custom_transformers"`
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if the file is in a dataset or
  space, `None` or `"model"` if in a model. Default is `None`.
- **revision** (`str`, *optional*) --
  The git revision to commit from. Defaults to the head of the `"main"` branch.
- **commit_message** (`str`, *optional*) --
  The summary / title / first line of the generated commit. Defaults to
  `f"Delete {path_in_repo} with huggingface_hub"`.
- **commit_description** (`str` *optional*) --
  The description of the generated commit
- **create_pr** (`boolean`, *optional*) --
  Whether or not to create a Pull Request with that commit. Defaults to `False`.
  If `revision` is not set, PR is opened against the `"main"` branch. If
  `revision` is set and is a branch, PR is opened against this branch. If
  `revision` is set and is not a branch name (example: a commit oid), an
  `RevisionNotFoundError` is returned by the server.
- **parent_commit** (`str`, *optional*) --
  The OID / SHA of the parent commit, as a hexadecimal string. Shorthands (7 first characters) are also supported.
  If specified and `create_pr` is `False`, the commit will fail if `revision` does not point to `parent_commit`.
  If specified and `create_pr` is `True`, the pull request will be created from `parent_commit`.
  Specifying `parent_commit` ensures the repo has not changed before committing the changes, and can be
  especially useful if the repo is updated / committed to concurrently.</paramsdesc><paramgroups>0</paramgroups></docstring>

Deletes a file in the given repo.



> [!TIP]
> Raises the following errors:
>
>     - [`HTTPError`](https://requests.readthedocs.io/en/latest/api/#requests.HTTPError)
>       if the HuggingFace API returned an error
>     - [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError)
>       if some parameter value is invalid
>     - [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError)
>       If the repository to download from cannot be found. This may be because it doesn't exist,
>       or because it is set to `private` and you do not have access.
>     - [RevisionNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RevisionNotFoundError)
>       If the revision to download from cannot be found.
>     - [EntryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.EntryNotFoundError)
>       If the file to download cannot be found.



</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>delete_files</name><anchor>huggingface_hub.HfApi.delete_files</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L4906</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "delete_patterns", "val": ": list[str]"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "commit_message", "val": ": Optional[str] = None"}, {"name": "commit_description", "val": ": Optional[str] = None"}, {"name": "create_pr", "val": ": Optional[bool] = None"}, {"name": "parent_commit", "val": ": Optional[str] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  The repository from which the folder will be deleted, for example:
  `"username/custom_transformers"`
- **delete_patterns** (`list[str]`) --
  List of files or folders to delete. Each string can either be
  a file path, a folder path or a Unix shell-style wildcard.
  E.g. `["file.txt", "folder/", "data/*.parquet"]`
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.
  to the stored token.
- **repo_type** (`str`, *optional*) --
  Type of the repo to delete files from. Can be `"model"`,
  `"dataset"` or `"space"`. Defaults to `"model"`.
- **revision** (`str`, *optional*) --
  The git revision to commit from. Defaults to the head of the `"main"` branch.
- **commit_message** (`str`, *optional*) --
  The summary (first line) of the generated commit. Defaults to
  `f"Delete files using huggingface_hub"`.
- **commit_description** (`str` *optional*) --
  The description of the generated commit.
- **create_pr** (`boolean`, *optional*) --
  Whether or not to create a Pull Request with that commit. Defaults to `False`.
  If `revision` is not set, PR is opened against the `"main"` branch. If
  `revision` is set and is a branch, PR is opened against this branch. If
  `revision` is set and is not a branch name (example: a commit oid), an
  `RevisionNotFoundError` is returned by the server.
- **parent_commit** (`str`, *optional*) --
  The OID / SHA of the parent commit, as a hexadecimal string. Shorthands (7 first characters) are also supported.
  If specified and `create_pr` is `False`, the commit will fail if `revision` does not point to `parent_commit`.
  If specified and `create_pr` is `True`, the pull request will be created from `parent_commit`.
  Specifying `parent_commit` ensures the repo has not changed before committing the changes, and can be
  especially useful if the repo is updated / committed to concurrently.</paramsdesc><paramgroups>0</paramgroups></docstring>

Delete files from a repository on the Hub.

If a folder path is provided, the entire folder is deleted as well as
all files it contained.




</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>delete_folder</name><anchor>huggingface_hub.HfApi.delete_folder</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L4982</source><parameters>[{"name": "path_in_repo", "val": ": str"}, {"name": "repo_id", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "commit_message", "val": ": Optional[str] = None"}, {"name": "commit_description", "val": ": Optional[str] = None"}, {"name": "create_pr", "val": ": Optional[bool] = None"}, {"name": "parent_commit", "val": ": Optional[str] = None"}]</parameters><paramsdesc>- **path_in_repo** (`str`) --
  Relative folder path in the repo, for example: `"checkpoints/1fec34a"`.
- **repo_id** (`str`) --
  The repository from which the folder will be deleted, for example:
  `"username/custom_transformers"`
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.
  to the stored token.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if the folder is in a dataset or
  space, `None` or `"model"` if in a model. Default is `None`.
- **revision** (`str`, *optional*) --
  The git revision to commit from. Defaults to the head of the `"main"` branch.
- **commit_message** (`str`, *optional*) --
  The summary / title / first line of the generated commit. Defaults to
  `f"Delete folder {path_in_repo} with huggingface_hub"`.
- **commit_description** (`str` *optional*) --
  The description of the generated commit.
- **create_pr** (`boolean`, *optional*) --
  Whether or not to create a Pull Request with that commit. Defaults to `False`.
  If `revision` is not set, PR is opened against the `"main"` branch. If
  `revision` is set and is a branch, PR is opened against this branch. If
  `revision` is set and is not a branch name (example: a commit oid), an
  `RevisionNotFoundError` is returned by the server.
- **parent_commit** (`str`, *optional*) --
  The OID / SHA of the parent commit, as a hexadecimal string. Shorthands (7 first characters) are also supported.
  If specified and `create_pr` is `False`, the commit will fail if `revision` does not point to `parent_commit`.
  If specified and `create_pr` is `True`, the pull request will be created from `parent_commit`.
  Specifying `parent_commit` ensures the repo has not changed before committing the changes, and can be
  especially useful if the repo is updated / committed to concurrently.</paramsdesc><paramgroups>0</paramgroups></docstring>

Deletes a folder in the given repo.

Simple wrapper around [create_commit()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.create_commit) method.




</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>delete_inference_endpoint</name><anchor>huggingface_hub.HfApi.delete_inference_endpoint</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L7875</source><parameters>[{"name": "name", "val": ": str"}, {"name": "namespace", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **name** (`str`) --
  The name of the Inference Endpoint to delete.
- **namespace** (`str`, *optional*) --
  The namespace in which the Inference Endpoint is located. Defaults to the current user.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups></docstring>
Delete an Inference Endpoint.

This operation is not reversible. If you don't want to be charged for an Inference Endpoint, it is preferable
to pause it with [pause_inference_endpoint()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.pause_inference_endpoint) or scale it to zero with [scale_to_zero_inference_endpoint()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.scale_to_zero_inference_endpoint).

For convenience, you can also delete an Inference Endpoint using [InferenceEndpoint.delete()](/docs/huggingface_hub/main/en/package_reference/inference_endpoints#huggingface_hub.InferenceEndpoint.delete).




</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>delete_repo</name><anchor>huggingface_hub.HfApi.delete_repo</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L3739</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "token", "val": ": Union[str, bool, None] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "missing_ok", "val": ": bool = False"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated
  by a `/`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if uploading to a dataset or
  space, `None` or `"model"` if uploading to a model.
- **missing_ok** (`bool`, *optional*, defaults to `False`) --
  If `True`, do not raise an error if repo does not exist.</paramsdesc><paramgroups>0</paramgroups><raises>- [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) -- 
  If the repository to delete from cannot be found and `missing_ok` is set to False (default).</raises><raisederrors>[RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError)</raisederrors></docstring>

Delete a repo from the HuggingFace Hub. CAUTION: this is irreversible.








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>delete_scheduled_job</name><anchor>huggingface_hub.HfApi.delete_scheduled_job</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L10429</source><parameters>[{"name": "scheduled_job_id", "val": ": str"}, {"name": "namespace", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **scheduled_job_id** (`str`) --
  ID of the scheduled Job.

- **namespace** (`str`, *optional*) --
  The namespace where the scheduled Job is. Defaults to the current user's namespace.

- **token** `(Union[bool, str, None]`, *optional*) --
  A valid user access token. If not provided, the locally saved token will be used, which is the
  recommended authentication method. Set to `False` to disable authentication.
  Refer to: https://huggingface.co/docs/huggingface_hub/quick-start#authentication.</paramsdesc><paramgroups>0</paramgroups></docstring>

Delete a scheduled compute Job on Hugging Face infrastructure.




</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>delete_space_secret</name><anchor>huggingface_hub.HfApi.delete_space_secret</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L6810</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "key", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  ID of the repo to update. Example: `"bigcode/in-the-stack"`.
- **key** (`str`) --
  Secret key. Example: `"GITHUB_API_KEY"`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups></docstring>
Deletes a secret from a Space.

Secrets allow to set secret keys or tokens to a Space without hardcoding them.
For more details, see https://huggingface.co/docs/hub/spaces-overview#managing-secrets.




</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>delete_space_storage</name><anchor>huggingface_hub.HfApi.delete_space_storage</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L7287</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  ID of the Space to update. Example: `"open-llm-leaderboard/open_llm_leaderboard"`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[SpaceRuntime](/docs/huggingface_hub/main/en/package_reference/space_runtime#huggingface_hub.SpaceRuntime)</rettype><retdesc>Runtime information about a Space including Space stage and hardware.</retdesc><raises>- `BadRequestError` -- 
  If space has no persistent storage.</raises><raisederrors>`BadRequestError`</raisederrors></docstring>
Delete persistent storage for a Space.












</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>delete_space_variable</name><anchor>huggingface_hub.HfApi.delete_space_variable</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L6900</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "key", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  ID of the repo to update. Example: `"bigcode/in-the-stack"`.
- **key** (`str`) --
  Variable key. Example: `"MODEL_REPO_ID"`
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups></docstring>
Deletes a variable from a Space.

Variables allow to set environment variables to a Space without hardcoding them.
For more details, see https://huggingface.co/docs/hub/spaces-overview#managing-secrets-and-environment-variables




</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>delete_tag</name><anchor>huggingface_hub.HfApi.delete_tag</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L5938</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "tag", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  The repository in which a tag will be deleted.
  Example: `"user/my-cool-model"`.

- **tag** (`str`) --
  The name of the tag to delete.

- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.

- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if tagging a dataset or space, `None` or
  `"model"` if tagging a model. Default is `None`.</paramsdesc><paramgroups>0</paramgroups><raises>- [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) -- 
  If repository is not found (error 404): wrong repo_id/repo_type, private
  but not authenticated or repo does not exist.
- [RevisionNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RevisionNotFoundError) -- 
  If tag is not found.</raises><raisederrors>[RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) or [RevisionNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RevisionNotFoundError)</raisederrors></docstring>

Delete a tag from a repo on the Hub.








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>delete_webhook</name><anchor>huggingface_hub.HfApi.delete_webhook</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L9349</source><parameters>[{"name": "webhook_id", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **webhook_id** (`str`) --
  The unique identifier of the webhook to delete.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved token, which is the recommended
  method for authentication (see https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`None`</rettype></docstring>
Delete a webhook.







<ExampleCodeBlock anchor="huggingface_hub.HfApi.delete_webhook.example">

Example:
```python
>>> from huggingface_hub import delete_webhook
>>> delete_webhook("654bbbc16f2ec14d77f109cc")
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>disable_webhook</name><anchor>huggingface_hub.HfApi.disable_webhook</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L9296</source><parameters>[{"name": "webhook_id", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **webhook_id** (`str`) --
  The unique identifier of the webhook to disable.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved token, which is the recommended
  method for authentication (see https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[WebhookInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.WebhookInfo)</rettype><retdesc>Info about the disabled webhook.</retdesc></docstring>
Disable a webhook (makes it "disabled").







<ExampleCodeBlock anchor="huggingface_hub.HfApi.disable_webhook.example">

Example:
```python
>>> from huggingface_hub import disable_webhook
>>> disabled_webhook = disable_webhook("654bbbc16f2ec14d77f109cc")
>>> disabled_webhook
WebhookInfo(
    id="654bbbc16f2ec14d77f109cc",
    url="https://webhook.site/a2176e82-5720-43ee-9e06-f91cb4c91548",
    jon=None,
    watched=[WebhookWatchedItem(type="user", name="julien-c"), WebhookWatchedItem(type="org", name="HuggingFaceH4")],
    domains=["repo", "discussion"],
    secret="my-secret",
    disabled=True,
)
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>duplicate_space</name><anchor>huggingface_hub.HfApi.duplicate_space</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L7133</source><parameters>[{"name": "from_id", "val": ": str"}, {"name": "to_id", "val": ": Optional[str] = None"}, {"name": "private", "val": ": Optional[bool] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "exist_ok", "val": ": bool = False"}, {"name": "hardware", "val": ": Optional[SpaceHardware] = None"}, {"name": "storage", "val": ": Optional[SpaceStorage] = None"}, {"name": "sleep_time", "val": ": Optional[int] = None"}, {"name": "secrets", "val": ": Optional[list[dict[str, str]]] = None"}, {"name": "variables", "val": ": Optional[list[dict[str, str]]] = None"}]</parameters><paramsdesc>- **from_id** (`str`) --
  ID of the Space to duplicate. Example: `"pharma/CLIP-Interrogator"`.
- **to_id** (`str`, *optional*) --
  ID of the new Space. Example: `"dog/CLIP-Interrogator"`. If not provided, the new Space will have the same
  name as the original Space, but in your account.
- **private** (`bool`, *optional*) --
  Whether the new Space should be private or not. Defaults to the same privacy as the original Space.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.
- **exist_ok** (`bool`, *optional*, defaults to `False`) --
  If `True`, do not raise an error if repo already exists.
- **hardware** (`SpaceHardware` or `str`, *optional*) --
  Choice of Hardware. Example: `"t4-medium"`. See [SpaceHardware](/docs/huggingface_hub/main/en/package_reference/space_runtime#huggingface_hub.SpaceHardware) for a complete list.
- **storage** (`SpaceStorage` or `str`, *optional*) --
  Choice of persistent storage tier. Example: `"small"`. See [SpaceStorage](/docs/huggingface_hub/main/en/package_reference/space_runtime#huggingface_hub.SpaceStorage) for a complete list.
- **sleep_time** (`int`, *optional*) --
  Number of seconds of inactivity to wait before a Space is put to sleep. Set to `-1` if you don't want
  your Space to sleep (default behavior for upgraded hardware). For free hardware, you can't configure
  the sleep time (value is fixed to 48 hours of inactivity).
  See https://huggingface.co/docs/hub/spaces-gpus#sleep-time for more details.
- **secrets** (`list[dict[str, str]]`, *optional*) --
  A list of secret keys to set in your Space. Each item is in the form `{"key": ..., "value": ..., "description": ...}` where description is optional.
  For more details, see https://huggingface.co/docs/hub/spaces-overview#managing-secrets.
- **variables** (`list[dict[str, str]]`, *optional*) --
  A list of public environment variables to set in your Space. Each item is in the form `{"key": ..., "value": ..., "description": ...}` where description is optional.
  For more details, see https://huggingface.co/docs/hub/spaces-overview#managing-secrets-and-environment-variables.</paramsdesc><paramgroups>0</paramgroups><rettype>[RepoUrl](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.RepoUrl)</rettype><retdesc>URL to the newly created repo. Value is a subclass of `str` containing
attributes like `endpoint`, `repo_type` and `repo_id`.</retdesc><raises>- [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) -- 
  If one of `from_id` or `to_id` cannot be found. This may be because it doesn't exist,
  or because it is set to `private` and you do not have access.
- `HfHubHTTPError` -- 
  If the HuggingFace API returned an error</raises><raisederrors>[RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) or `HfHubHTTPError`</raisederrors></docstring>
Duplicate a Space.

Programmatically duplicate a Space. The new Space will be created in your account and will be in the same state
as the original Space (running or paused). You can duplicate a Space no matter the current state of a Space.











<ExampleCodeBlock anchor="huggingface_hub.HfApi.duplicate_space.example">

Example:
```python
>>> from huggingface_hub import duplicate_space

# Duplicate a Space to your account
>>> duplicate_space("multimodalart/dreambooth-training")
RepoUrl('https://huggingface.co/spaces/nateraw/dreambooth-training',...)

# Can set custom destination id and visibility flag.
>>> duplicate_space("multimodalart/dreambooth-training", to_id="my-dreambooth", private=True)
RepoUrl('https://huggingface.co/spaces/nateraw/my-dreambooth',...)
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>edit_discussion_comment</name><anchor>huggingface_hub.HfApi.edit_discussion_comment</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L6653</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "discussion_num", "val": ": int"}, {"name": "comment_id", "val": ": str"}, {"name": "new_content", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated
  by a `/`.
- **discussion_num** (`int`) --
  The number of the Discussion or Pull Request . Must be a strictly positive integer.
- **comment_id** (`str`) --
  The ID of the comment to edit.
- **new_content** (`str`) --
  The new content of the comment. Comments support markdown formatting.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if uploading to a dataset or
  space, `None` or `"model"` if uploading to a model. Default is
  `None`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[DiscussionComment](/docs/huggingface_hub/main/en/package_reference/community#huggingface_hub.DiscussionComment)</rettype><retdesc>the edited comment</retdesc></docstring>
Edits a comment on a Discussion / Pull Request.







> [!TIP]
> Raises the following errors:
>
>     - [`HTTPError`](https://requests.readthedocs.io/en/latest/api/#requests.HTTPError)
>       if the HuggingFace API returned an error
>     - [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError)
>       if some parameter value is invalid
>     - [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError)
>       If the repository to download from cannot be found. This may be because it doesn't exist,
>       or because it is set to `private` and you do not have access.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>enable_webhook</name><anchor>huggingface_hub.HfApi.enable_webhook</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L9243</source><parameters>[{"name": "webhook_id", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **webhook_id** (`str`) --
  The unique identifier of the webhook to enable.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved token, which is the recommended
  method for authentication (see https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[WebhookInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.WebhookInfo)</rettype><retdesc>Info about the enabled webhook.</retdesc></docstring>
Enable a webhook (makes it "active").







<ExampleCodeBlock anchor="huggingface_hub.HfApi.enable_webhook.example">

Example:
```python
>>> from huggingface_hub import enable_webhook
>>> enabled_webhook = enable_webhook("654bbbc16f2ec14d77f109cc")
>>> enabled_webhook
WebhookInfo(
    id="654bbbc16f2ec14d77f109cc",
    job=None,
    url="https://webhook.site/a2176e82-5720-43ee-9e06-f91cb4c91548",
    watched=[WebhookWatchedItem(type="user", name="julien-c"), WebhookWatchedItem(type="org", name="HuggingFaceH4")],
    domains=["repo", "discussion"],
    secret="my-secret",
    disabled=False,
)
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>fetch_job_logs</name><anchor>huggingface_hub.HfApi.fetch_job_logs</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L9927</source><parameters>[{"name": "job_id", "val": ": str"}, {"name": "namespace", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **job_id** (`str`) --
  ID of the Job.

- **namespace** (`str`, *optional*) --
  The namespace where the Job is running. Defaults to the current user's namespace.

- **token** `(Union[bool, str, None]`, *optional*) --
  A valid user access token. If not provided, the locally saved token will be used, which is the
  recommended authentication method. Set to `False` to disable authentication.
  Refer to: https://huggingface.co/docs/huggingface_hub/quick-start#authentication.</paramsdesc><paramgroups>0</paramgroups></docstring>

Fetch all the logs from a compute Job on Hugging Face infrastructure.



<ExampleCodeBlock anchor="huggingface_hub.HfApi.fetch_job_logs.example">

Example:

```python
>>> from huggingface_hub import fetch_job_logs, run_job
>>> job = run_job(image="python:3.12", command=["python", "-c" ,"print('Hello from HF compute!')"])
>>> for log in fetch_job_logs(job.id):
...     print(log)
Hello from HF compute!
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>file_exists</name><anchor>huggingface_hub.HfApi.file_exists</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L2858</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "filename", "val": ": str"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[str, bool, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated
  by a `/`.
- **filename** (`str`) --
  The name of the file to check, for example:
  `"config.json"`
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if getting repository info from a dataset or a space,
  `None` or `"model"` if getting repository info from a model. Default is `None`.
- **revision** (`str`, *optional*) --
  The revision of the repository from which to get the information. Defaults to `"main"` branch.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><retdesc>True if the file exists, False otherwise.</retdesc></docstring>

Checks if a file exists in a repository on the Hugging Face Hub.





<ExampleCodeBlock anchor="huggingface_hub.HfApi.file_exists.example">

Examples:
```py
>>> from huggingface_hub import file_exists
>>> file_exists("bigcode/starcoder", "config.json")
True
>>> file_exists("bigcode/starcoder", "not-a-file")
False
>>> file_exists("bigcode/not-a-repo", "config.json")
False
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_collection</name><anchor>huggingface_hub.HfApi.get_collection</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L8089</source><parameters>[{"name": "collection_slug", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **collection_slug** (`str`) --
  Slug of the collection of the Hub. Example: `"TheBloke/recent-models-64f9a55bb3115b4f513ec026"`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups></docstring>
Gets information about a Collection on the Hub.



Returns: [Collection](/docs/huggingface_hub/main/en/package_reference/collections#huggingface_hub.Collection)

<ExampleCodeBlock anchor="huggingface_hub.HfApi.get_collection.example">

Example:

```py
>>> from huggingface_hub import get_collection
>>> collection = get_collection("TheBloke/recent-models-64f9a55bb3115b4f513ec026")
>>> collection.title
'Recent models'
>>> len(collection.items)
37
>>> collection.items[0]
CollectionItem(
    item_object_id='651446103cd773a050bf64c2',
    item_id='TheBloke/U-Amethyst-20B-AWQ',
    item_type='model',
    position=88,
    note=None
)
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_dataset_tags</name><anchor>huggingface_hub.HfApi.get_dataset_tags</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L1783</source><parameters>[]</parameters></docstring>

List all valid dataset tags as a nested namespace object.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_discussion_details</name><anchor>huggingface_hub.HfApi.get_discussion_details</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L6133</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "discussion_num", "val": ": int"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated
  by a `/`.
- **discussion_num** (`int`) --
  The number of the Discussion or Pull Request . Must be a strictly positive integer.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if uploading to a dataset or
  space, `None` or `"model"` if uploading to a model. Default is
  `None`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups></docstring>
Fetches a Discussion's / Pull Request 's details from the Hub.



Returns: [DiscussionWithDetails](/docs/huggingface_hub/main/en/package_reference/community#huggingface_hub.DiscussionWithDetails)

> [!TIP]
> Raises the following errors:
>
>     - [`HTTPError`](https://requests.readthedocs.io/en/latest/api/#requests.HTTPError)
>       if the HuggingFace API returned an error
>     - [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError)
>       if some parameter value is invalid
>     - [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError)
>       If the repository to download from cannot be found. This may be because it doesn't exist,
>       or because it is set to `private` and you do not have access.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_full_repo_name</name><anchor>huggingface_hub.HfApi.get_full_repo_name</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L5987</source><parameters>[{"name": "model_id", "val": ": str"}, {"name": "organization", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **model_id** (`str`) --
  The name of the model.
- **organization** (`str`, *optional*) --
  If passed, the repository name will be in the organization
  namespace instead of the user namespace.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`str`</rettype><retdesc>The repository name in the user's namespace
({username}/{model_id}) if no organization is passed, and under the
organization namespace ({organization}/{model_id}) otherwise.</retdesc></docstring>

Returns the repository name for a given model ID and optional
organization.








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_hf_file_metadata</name><anchor>huggingface_hub.HfApi.get_hf_file_metadata</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L5166</source><parameters>[{"name": "url", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "timeout", "val": ": Optional[float] = 10"}]</parameters><paramsdesc>- **url** (`str`) --
  File url, for example returned by [hf_hub_url()](/docs/huggingface_hub/main/en/package_reference/file_download#huggingface_hub.hf_hub_url).
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.
- **timeout** (`float`, *optional*, defaults to 10) --
  How many seconds to wait for the server to send metadata before giving up.</paramsdesc><paramgroups>0</paramgroups><retdesc>A [HfFileMetadata](/docs/huggingface_hub/main/en/package_reference/file_download#huggingface_hub.HfFileMetadata) object containing metadata such as location, etag, size and commit_hash.</retdesc></docstring>
Fetch metadata of a file versioned on the Hub for a given url.






</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_inference_endpoint</name><anchor>huggingface_hub.HfApi.get_inference_endpoint</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L7691</source><parameters>[{"name": "name", "val": ": str"}, {"name": "namespace", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **name** (`str`) --
  The name of the Inference Endpoint to retrieve information about.
- **namespace** (`str`, *optional*) --
  The namespace in which the Inference Endpoint is located. Defaults to the current user.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[InferenceEndpoint](/docs/huggingface_hub/main/en/package_reference/inference_endpoints#huggingface_hub.InferenceEndpoint)</rettype><retdesc>information about the requested Inference Endpoint.</retdesc></docstring>
Get information about an Inference Endpoint.







<ExampleCodeBlock anchor="huggingface_hub.HfApi.get_inference_endpoint.example">

Example:
```python
>>> from huggingface_hub import HfApi
>>> api = HfApi()
>>> endpoint = api.get_inference_endpoint("my-text-to-image")
>>> endpoint
InferenceEndpoint(name='my-text-to-image', ...)

# Get status
>>> endpoint.status
'running'
>>> endpoint.url
'https://my-text-to-image.region.vendor.endpoints.huggingface.cloud'

# Run inference
>>> endpoint.client.text_to_image(...)
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_model_tags</name><anchor>huggingface_hub.HfApi.get_model_tags</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L1774</source><parameters>[]</parameters></docstring>

List all valid model tags as a nested namespace object


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_organization_overview</name><anchor>huggingface_hub.HfApi.get_organization_overview</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L9574</source><parameters>[{"name": "organization", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **organization** (`str`) --
  Name of the organization to get an overview of.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved token, which is the recommended method
  for authentication (see https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`Organization`</rettype><retdesc>An `Organization` object with the organization's overview.</retdesc><raises>- [`HTTPError`](https://requests.readthedocs.io/en/latest/api/#requests.HTTPError) -- 
  HTTP 404 If the organization does not exist on the Hub.</raises><raisederrors>``HTTPError``</raisederrors></docstring>

Get an overview of an organization on the Hub.












</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_paths_info</name><anchor>huggingface_hub.HfApi.get_paths_info</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L3316</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "paths", "val": ": Union[list[str], str]"}, {"name": "expand", "val": ": bool = False"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[str, bool, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated by a `/`.
- **paths** (`Union[list[str], str]`, *optional*) --
  The paths to get information about. If a path do not exist, it is ignored without raising
  an exception.
- **expand** (`bool`, *optional*, defaults to `False`) --
  Whether to fetch more information about the paths (e.g. last commit and files' security scan results). This
  operation is more expensive for the server so only 50 results are returned per page (instead of 1000).
  As pagination is implemented in `huggingface_hub`, this is transparent for you except for the time it
  takes to get the results.
- **revision** (`str`, *optional*) --
  The revision of the repository from which to get the information. Defaults to `"main"` branch.
- **repo_type** (`str`, *optional*) --
  The type of the repository from which to get the information (`"model"`, `"dataset"` or `"space"`.
  Defaults to `"model"`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[Union[RepoFile, RepoFolder]]`</rettype><retdesc>The information about the paths, as a list of `RepoFile` and `RepoFolder` objects.</retdesc><raises>- [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) -- 
  If repository is not found (error 404): wrong repo_id/repo_type, private but not authenticated or repo
  does not exist.
- [RevisionNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RevisionNotFoundError) -- 
  If revision is not found (error 404) on the repo.</raises><raisederrors>[RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) or [RevisionNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RevisionNotFoundError)</raisederrors></docstring>

Get information about a repo's paths.











<ExampleCodeBlock anchor="huggingface_hub.HfApi.get_paths_info.example">

Example:
```py
>>> from huggingface_hub import get_paths_info
>>> paths_info = get_paths_info("allenai/c4", ["README.md", "en"], repo_type="dataset")
>>> paths_info
[
    RepoFile(path='README.md', size=2379, blob_id='f84cb4c97182890fc1dbdeaf1a6a468fd27b4fff', lfs=None, last_commit=None, security=None),
    RepoFolder(path='en', tree_id='dc943c4c40f53d02b31ced1defa7e5f438d5862e', last_commit=None)
]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_repo_discussions</name><anchor>huggingface_hub.HfApi.get_repo_discussions</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L6025</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "author", "val": ": Optional[str] = None"}, {"name": "discussion_type", "val": ": Optional[constants.DiscussionTypeFilter] = None"}, {"name": "discussion_status", "val": ": Optional[constants.DiscussionStatusFilter] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated
  by a `/`.
- **author** (`str`, *optional*) --
  Pass a value to filter by discussion author. `None` means no filter.
  Default is `None`.
- **discussion_type** (`str`, *optional*) --
  Set to `"pull_request"` to fetch only pull requests, `"discussion"`
  to fetch only discussions. Set to `"all"` or `None` to fetch both.
  Default is `None`.
- **discussion_status** (`str`, *optional*) --
  Set to `"open"` (respectively `"closed"`) to fetch only open
  (respectively closed) discussions. Set to `"all"` or `None`
  to fetch both.
  Default is `None`.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if fetching from a dataset or
  space, `None` or `"model"` if fetching from a model. Default is
  `None`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`Iterator[Discussion]`</rettype><retdesc>An iterator of [Discussion](/docs/huggingface_hub/main/en/package_reference/community#huggingface_hub.Discussion) objects.</retdesc></docstring>

Fetches Discussions and Pull Requests for the given repo.







Example:
<ExampleCodeBlock anchor="huggingface_hub.HfApi.get_repo_discussions.example">

Collecting all discussions of a repo in a list:

```python
>>> from huggingface_hub import get_repo_discussions
>>> discussions_list = list(get_repo_discussions(repo_id="bert-base-uncased"))
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.HfApi.get_repo_discussions.example-2">

Iterating over discussions of a repo:

```python
>>> from huggingface_hub import get_repo_discussions
>>> for discussion in get_repo_discussions(repo_id="bert-base-uncased"):
...     print(discussion.num, discussion.title)
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_safetensors_metadata</name><anchor>huggingface_hub.HfApi.get_safetensors_metadata</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L5489</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A user or an organization name and a repo name separated by a `/`.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if the file is in a dataset or space, `None` or `"model"` if in a
  model. Default is `None`.
- **revision** (`str`, *optional*) --
  The git revision to fetch the file from. Can be a branch name, a tag, or a commit hash. Defaults to the
  head of the `"main"` branch.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`SafetensorsRepoMetadata`</rettype><retdesc>information related to safetensors repo.</retdesc><raises>- `NotASafetensorsRepoError` -- 
  If the repo is not a safetensors repo i.e. doesn't have either a
  `model.safetensors` or a `model.safetensors.index.json` file.
- `SafetensorsParsingError` -- 
  If a safetensors file header couldn't be parsed correctly.</raises><raisederrors>`NotASafetensorsRepoError` or `SafetensorsParsingError`</raisederrors></docstring>

Parse metadata for a safetensors repo on the Hub.

We first check if the repo has a single safetensors file or a sharded safetensors repo. If it's a single
safetensors file, we parse the metadata from this file. If it's a sharded safetensors repo, we parse the
metadata from the index file and then parse the metadata from each shard.

To parse metadata from a single safetensors file, use [parse_safetensors_file_metadata()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.parse_safetensors_file_metadata).

For more details regarding the safetensors format, check out https://huggingface.co/docs/safetensors/index#format.











<ExampleCodeBlock anchor="huggingface_hub.HfApi.get_safetensors_metadata.example">

Example:
```py
# Parse repo with single weights file
>>> metadata = get_safetensors_metadata("bigscience/bloomz-560m")
>>> metadata
SafetensorsRepoMetadata(
    metadata=None,
    sharded=False,
    weight_map={'h.0.input_layernorm.bias': 'model.safetensors', ...},
    files_metadata={'model.safetensors': SafetensorsFileMetadata(...)}
)
>>> metadata.files_metadata["model.safetensors"].metadata
{'format': 'pt'}

# Parse repo with sharded model
>>> metadata = get_safetensors_metadata("bigscience/bloom")
Parse safetensors files: 100%|██████████████████████████████████████████| 72/72 [00:12<00:00,  5.78it/s]
>>> metadata
SafetensorsRepoMetadata(metadata={'total_size': 352494542848}, sharded=True, weight_map={...}, files_metadata={...})
>>> len(metadata.files_metadata)
72  # All safetensors files have been fetched

# Parse repo with sharded model
>>> get_safetensors_metadata("runwayml/stable-diffusion-v1-5")
NotASafetensorsRepoError: 'runwayml/stable-diffusion-v1-5' is not a safetensors repo. Couldn't find 'model.safetensors.index.json' or 'model.safetensors' files.
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_space_runtime</name><anchor>huggingface_hub.HfApi.get_space_runtime</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L6929</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  ID of the repo to update. Example: `"bigcode/in-the-stack"`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[SpaceRuntime](/docs/huggingface_hub/main/en/package_reference/space_runtime#huggingface_hub.SpaceRuntime)</rettype><retdesc>Runtime information about a Space including Space stage and hardware.</retdesc></docstring>
Gets runtime information about a Space.








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_space_variables</name><anchor>huggingface_hub.HfApi.get_space_variables</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L6836</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  ID of the repo to query. Example: `"bigcode/in-the-stack"`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups></docstring>
Gets all variables from a Space.

Variables allow to set environment variables to a Space without hardcoding them.
For more details, see https://huggingface.co/docs/hub/spaces-overview#managing-secrets-and-environment-variables




</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_user_overview</name><anchor>huggingface_hub.HfApi.get_user_overview</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L9548</source><parameters>[{"name": "username", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **username** (`str`) --
  Username of the user to get an overview of.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`User`</rettype><retdesc>A [User](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.User) object with the user's overview.</retdesc><raises>- `HfHubHTTPError` -- 
  HTTP 404 If the user does not exist on the Hub.</raises><raisederrors>`HfHubHTTPError`</raisederrors></docstring>

Get an overview of a user on the Hub.












</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_webhook</name><anchor>huggingface_hub.HfApi.get_webhook</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L8928</source><parameters>[{"name": "webhook_id", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **webhook_id** (`str`) --
  The unique identifier of the webhook to get.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved token, which is the recommended
  method for authentication (see https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[WebhookInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.WebhookInfo)</rettype><retdesc>Info about the webhook.</retdesc></docstring>
Get a webhook by its id.







<ExampleCodeBlock anchor="huggingface_hub.HfApi.get_webhook.example">

Example:
```python
>>> from huggingface_hub import get_webhook
>>> webhook = get_webhook("654bbbc16f2ec14d77f109cc")
>>> print(webhook)
WebhookInfo(
    id="654bbbc16f2ec14d77f109cc",
    job=None,
    watched=[WebhookWatchedItem(type="user", name="julien-c"), WebhookWatchedItem(type="org", name="HuggingFaceH4")],
    url="https://webhook.site/a2176e82-5720-43ee-9e06-f91cb4c91548",
    secret="my-secret",
    domains=["repo", "discussion"],
    disabled=False,
)
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>grant_access</name><anchor>huggingface_hub.HfApi.grant_access</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L8873</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "user", "val": ": str"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  The id of the repo to grant access to.
- **user** (`str`) --
  The username of the user to grant access.
- **repo_type** (`str`, *optional*) --
  The type of the repo to grant access to. Must be one of `model`, `dataset` or `space`.
  Defaults to `model`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><raises>- `HfHubHTTPError` -- 
  HTTP 400 if the repo is not gated.
- `HfHubHTTPError` -- 
  HTTP 400 if the user already has access to the repo.
- `HfHubHTTPError` -- 
  HTTP 403 if you only have read-only access to the repo. This can be the case if you don't have `write`
  or `admin` role in the organization the repo belongs to or if you passed a `read` token.
- `HfHubHTTPError` -- 
  HTTP 404 if the user does not exist on the Hub.</raises><raisederrors>`HfHubHTTPError`</raisederrors></docstring>

Grant access to a user for a given gated repo.

Granting access don't require for the user to send an access request by themselves. The user is automatically
added to the accepted list meaning they can download the files You can revoke the granted access at any time
using [cancel_access_request()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.cancel_access_request) or [reject_access_request()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.reject_access_request).

For more info about gated repos, see https://huggingface.co/docs/hub/models-gated.








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>hf_hub_download</name><anchor>huggingface_hub.HfApi.hf_hub_download</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L5240</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "filename", "val": ": str"}, {"name": "subfolder", "val": ": Optional[str] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "cache_dir", "val": ": Union[str, Path, None] = None"}, {"name": "local_dir", "val": ": Union[str, Path, None] = None"}, {"name": "force_download", "val": ": bool = False"}, {"name": "etag_timeout", "val": ": float = 10"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "local_files_only", "val": ": bool = False"}, {"name": "dry_run", "val": ": bool = False"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A user or an organization name and a repo name separated by a `/`.
- **filename** (`str`) --
  The name of the file in the repo.
- **subfolder** (`str`, *optional*) --
  An optional value corresponding to a folder inside the repository.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if downloading from a dataset or space,
  `None` or `"model"` if downloading from a model. Default is `None`.
- **revision** (`str`, *optional*) --
  An optional Git revision id which can be a branch name, a tag, or a
  commit hash.
- **cache_dir** (`str`, `Path`, *optional*) --
  Path to the folder where cached files are stored.
- **local_dir** (`str` or `Path`, *optional*) --
  If provided, the downloaded file will be placed under this directory.
- **force_download** (`bool`, *optional*, defaults to `False`) --
  Whether the file should be downloaded even if it already exists in
  the local cache.
- **etag_timeout** (`float`, *optional*, defaults to `10`) --
  When fetching ETag, how many seconds to wait for the server to send
  data before giving up which is passed to `httpx.request`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.
- **local_files_only** (`bool`, *optional*, defaults to `False`) --
  If `True`, avoid downloading the file and return the path to the
  local cached file if it exists.
- **dry_run** (`bool`, *optional*, defaults to `False`) --
  If `True`, perform a dry run without actually downloading the file. Returns a
  [DryRunFileInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.DryRunFileInfo) object containing information about what would be downloaded.</paramsdesc><paramgroups>0</paramgroups><rettype>`str` or [DryRunFileInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.DryRunFileInfo)</rettype><retdesc>- If `dry_run=False`: Local path of file or if networking is off, last version of file cached on disk.
- If `dry_run=True`: A [DryRunFileInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.DryRunFileInfo) object containing download information.</retdesc><raises>- [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) -- 
  If the repository to download from cannot be found. This may be because it doesn't exist,
  or because it is set to `private` and you do not have access.
- [RevisionNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RevisionNotFoundError) -- 
  If the revision to download from cannot be found.
- `~utils.RemoteEntryNotFoundError` -- 
  If the file to download cannot be found.
- [LocalEntryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.LocalEntryNotFoundError) -- 
  If network is disabled or unavailable and file is not found in cache.
- [`EnvironmentError`](https://docs.python.org/3/library/exceptions.html#EnvironmentError) -- 
  If `token=True` but the token cannot be found.
- [`OSError`](https://docs.python.org/3/library/exceptions.html#OSError) -- 
  If ETag cannot be determined.
- [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError) -- 
  If some parameter value is invalid.</raises><raisederrors>[RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) or [RevisionNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RevisionNotFoundError) or `~utils.RemoteEntryNotFoundError` or [LocalEntryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.LocalEntryNotFoundError) or ``EnvironmentError`` or ``OSError`` or ``ValueError``</raisederrors></docstring>
Download a given file if it's not already present in the local cache.

The new cache file layout looks like this:
- The cache directory contains one subfolder per repo_id (namespaced by repo type)
- inside each repo folder:
  - refs is a list of the latest known revision => commit_hash pairs
  - blobs contains the actual file blobs (identified by their git-sha or sha256, depending on
  whether they're LFS files or not)
  - snapshots contains one subfolder per commit, each "commit" contains the subset of the files
  that have been resolved at that particular commit. Each filename is a symlink to the blob
  at that particular commit.

<ExampleCodeBlock anchor="huggingface_hub.HfApi.hf_hub_download.example">

```
[  96]  .
└── [ 160]  models--julien-c--EsperBERTo-small
    ├── [ 160]  blobs
    │   ├── [321M]  403450e234d65943a7dcf7e05a771ce3c92faa84dd07db4ac20f592037a1e4bd
    │   ├── [ 398]  7cb18dc9bafbfcf74629a4b760af1b160957a83e
    │   └── [1.4K]  d7edf6bd2a681fb0175f7735299831ee1b22b812
    ├── [  96]  refs
    │   └── [  40]  main
    └── [ 128]  snapshots
        ├── [ 128]  2439f60ef33a0d46d85da5001d52aeda5b00ce9f
        │   ├── [  52]  README.md -> ../../blobs/d7edf6bd2a681fb0175f7735299831ee1b22b812
        │   └── [  76]  pytorch_model.bin -> ../../blobs/403450e234d65943a7dcf7e05a771ce3c92faa84dd07db4ac20f592037a1e4bd
        └── [ 128]  bbc77c8132af1cc5cf678da3f1ddf2de43606d48
            ├── [  52]  README.md -> ../../blobs/7cb18dc9bafbfcf74629a4b760af1b160957a83e
            └── [  76]  pytorch_model.bin -> ../../blobs/403450e234d65943a7dcf7e05a771ce3c92faa84dd07db4ac20f592037a1e4bd
```

</ExampleCodeBlock>

If `local_dir` is provided, the file structure from the repo will be replicated in this location. When using this
option, the `cache_dir` will not be used and a `.cache/huggingface/` folder will be created at the root of `local_dir`
to store some metadata related to the downloaded files. While this mechanism is not as robust as the main
cache-system, it's optimized for regularly pulling the latest version of a repository.












</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>hide_discussion_comment</name><anchor>huggingface_hub.HfApi.hide_discussion_comment</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L6710</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "discussion_num", "val": ": int"}, {"name": "comment_id", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated
  by a `/`.
- **discussion_num** (`int`) --
  The number of the Discussion or Pull Request . Must be a strictly positive integer.
- **comment_id** (`str`) --
  The ID of the comment to edit.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if uploading to a dataset or
  space, `None` or `"model"` if uploading to a model. Default is
  `None`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[DiscussionComment](/docs/huggingface_hub/main/en/package_reference/community#huggingface_hub.DiscussionComment)</rettype><retdesc>the hidden comment</retdesc></docstring>
Hides a comment on a Discussion / Pull Request.

> [!WARNING]
> Hidden comments' content cannot be retrieved anymore. Hiding a comment is irreversible.







> [!TIP]
> Raises the following errors:
>
>     - [`HTTPError`](https://requests.readthedocs.io/en/latest/api/#requests.HTTPError)
>       if the HuggingFace API returned an error
>     - [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError)
>       if some parameter value is invalid
>     - [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError)
>       If the repository to download from cannot be found. This may be because it doesn't exist,
>       or because it is set to `private` and you do not have access.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>inspect_job</name><anchor>huggingface_hub.HfApi.inspect_job</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L10050</source><parameters>[{"name": "job_id", "val": ": str"}, {"name": "namespace", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **job_id** (`str`) --
  ID of the Job.

- **namespace** (`str`, *optional*) --
  The namespace where the Job is running. Defaults to the current user's namespace.

- **token** `(Union[bool, str, None]`, *optional*) --
  A valid user access token. If not provided, the locally saved token will be used, which is the
  recommended authentication method. Set to `False` to disable authentication.
  Refer to: https://huggingface.co/docs/huggingface_hub/quick-start#authentication.</paramsdesc><paramgroups>0</paramgroups></docstring>

Inspect a compute Job on Hugging Face infrastructure.



<ExampleCodeBlock anchor="huggingface_hub.HfApi.inspect_job.example">

Example:

```python
>>> from huggingface_hub import inspect_job, run_job
>>> job = run_job(image="python:3.12", command=["python", "-c" ,"print('Hello from HF compute!')"])
>>> inspect_job(job.id)
JobInfo(
    id='68780d00bbe36d38803f645f',
    created_at=datetime.datetime(2025, 7, 16, 20, 35, 12, 808000, tzinfo=datetime.timezone.utc),
    docker_image='python:3.12',
    space_id=None,
    command=['python', '-c', "print('Hello from HF compute!')"],
    arguments=[],
    environment={},
    secrets={},
    flavor='cpu-basic',
    status=JobStatus(stage='RUNNING', message=None)
)
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>inspect_scheduled_job</name><anchor>huggingface_hub.HfApi.inspect_scheduled_job</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L10390</source><parameters>[{"name": "scheduled_job_id", "val": ": str"}, {"name": "namespace", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **scheduled_job_id** (`str`) --
  ID of the scheduled Job.

- **namespace** (`str`, *optional*) --
  The namespace where the scheduled Job is. Defaults to the current user's namespace.

- **token** `(Union[bool, str, None]`, *optional*) --
  A valid user access token. If not provided, the locally saved token will be used, which is the
  recommended authentication method. Set to `False` to disable authentication.
  Refer to: https://huggingface.co/docs/huggingface_hub/quick-start#authentication.</paramsdesc><paramgroups>0</paramgroups></docstring>

Inspect a scheduled compute Job on Hugging Face infrastructure.



<ExampleCodeBlock anchor="huggingface_hub.HfApi.inspect_scheduled_job.example">

Example:

```python
>>> from huggingface_hub import inspect_job, create_scheduled_job
>>> scheduled_job = create_scheduled_job(image="python:3.12", command=["python", "-c" ,"print('Hello from HF compute!')"], schedule="@hourly")
>>> inspect_scheduled_job(scheduled_job.id)
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>list_accepted_access_requests</name><anchor>huggingface_hub.HfApi.list_accepted_access_requests</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L8557</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  The id of the repo to get access requests for.
- **repo_type** (`str`, *optional*) --
  The type of the repo to get access requests for. Must be one of `model`, `dataset` or `space`.
  Defaults to `model`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[AccessRequest]`</rettype><retdesc>A list of `AccessRequest` objects. Each time contains a `username`, `email`,
`status` and `timestamp` attribute. If the gated repo has a custom form, the `fields` attribute will
be populated with user's answers.</retdesc><raises>- `HfHubHTTPError` -- 
  HTTP 400 if the repo is not gated.
- `HfHubHTTPError` -- 
  HTTP 403 if you only have read-only access to the repo. This can be the case if you don't have `write`
  or `admin` role in the organization the repo belongs to or if you passed a `read` token.</raises><raisederrors>`HfHubHTTPError`</raisederrors></docstring>

Get accepted access requests for a given gated repo.

An accepted request means the user has requested access to the repo and the request has been accepted. The user
can download any file of the repo. If the approval mode is automatic, this list should contains by default all
requests. Accepted requests can be cancelled or rejected at any time using [cancel_access_request()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.cancel_access_request) and
[reject_access_request()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.reject_access_request). A cancelled request will go back to the pending list while a rejected request will
go to the rejected list. In both cases, the user will lose access to the repo.

For more info about gated repos, see https://huggingface.co/docs/hub/models-gated.











<ExampleCodeBlock anchor="huggingface_hub.HfApi.list_accepted_access_requests.example">

Example:
```py
>>> from huggingface_hub import list_accepted_access_requests

>>> requests = list_accepted_access_requests("meta-llama/Llama-2-7b")
>>> len(requests)
411
>>> requests[0]
[
    AccessRequest(
        username='clem',
        fullname='Clem 🤗',
        email='***',
        timestamp=datetime.datetime(2023, 11, 23, 18, 4, 53, 828000, tzinfo=datetime.timezone.utc),
        status='accepted',
        fields=None,
    ),
    ...
]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>list_collections</name><anchor>huggingface_hub.HfApi.list_collections</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L8033</source><parameters>[{"name": "owner", "val": ": Union[list[str], str, None] = None"}, {"name": "item", "val": ": Union[list[str], str, None] = None"}, {"name": "sort", "val": ": Optional[Literal['lastModified', 'trending', 'upvotes']] = None"}, {"name": "limit", "val": ": Optional[int] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **owner** (`list[str]` or `str`, *optional*) --
  Filter by owner's username.
- **item** (`list[str]` or `str`, *optional*) --
  Filter collections containing a particular items. Example: `"models/teknium/OpenHermes-2.5-Mistral-7B"`, `"datasets/squad"` or `"papers/2311.12983"`.
- **sort** (`Literal["lastModified", "trending", "upvotes"]`, *optional*) --
  Sort collections by last modified, trending or upvotes.
- **limit** (`int`, *optional*) --
  Maximum number of collections to be returned.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`Iterable[Collection]`</rettype><retdesc>an iterable of [Collection](/docs/huggingface_hub/main/en/package_reference/collections#huggingface_hub.Collection) objects.</retdesc></docstring>
List collections on the Huggingface Hub, given some filters.

> [!WARNING]
> When listing collections, the item list per collection is truncated to 4 items maximum. To retrieve all items
> from a collection, you must use [get_collection()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.get_collection).








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>list_datasets</name><anchor>huggingface_hub.HfApi.list_datasets</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L1992</source><parameters>[{"name": "filter", "val": ": Union[str, Iterable[str], None] = None"}, {"name": "author", "val": ": Optional[str] = None"}, {"name": "benchmark", "val": ": Optional[Union[str, list[str]]] = None"}, {"name": "dataset_name", "val": ": Optional[str] = None"}, {"name": "gated", "val": ": Optional[bool] = None"}, {"name": "language_creators", "val": ": Optional[Union[str, list[str]]] = None"}, {"name": "language", "val": ": Optional[Union[str, list[str]]] = None"}, {"name": "multilinguality", "val": ": Optional[Union[str, list[str]]] = None"}, {"name": "size_categories", "val": ": Optional[Union[str, list[str]]] = None"}, {"name": "task_categories", "val": ": Optional[Union[str, list[str]]] = None"}, {"name": "task_ids", "val": ": Optional[Union[str, list[str]]] = None"}, {"name": "search", "val": ": Optional[str] = None"}, {"name": "sort", "val": ": Optional[Union[Literal['last_modified'], str]] = None"}, {"name": "direction", "val": ": Optional[Literal[-1]] = None"}, {"name": "limit", "val": ": Optional[int] = None"}, {"name": "expand", "val": ": Optional[list[ExpandDatasetProperty_T]] = None"}, {"name": "full", "val": ": Optional[bool] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "tags", "val": ": Optional[Union[str, list[str]]] = None"}]</parameters><paramsdesc>- **filter** (`str` or `Iterable[str]`, *optional*) --
  A string or list of string to filter datasets on the hub.
- **author** (`str`, *optional*) --
  A string which identify the author of the returned datasets.
- **benchmark** (`str` or `List`, *optional*) --
  A string or list of strings that can be used to identify datasets on
  the Hub by their official benchmark.
- **dataset_name** (`str`, *optional*) --
  A string or list of strings that can be used to identify datasets on
  the Hub by its name, such as `SQAC` or `wikineural`
- **gated** (`bool`, *optional*) --
  A boolean to filter datasets on the Hub that are gated or not. By default, all datasets are returned.
  If `gated=True` is passed, only gated datasets are returned.
  If `gated=False` is passed, only non-gated datasets are returned.
- **language_creators** (`str` or `List`, *optional*) --
  A string or list of strings that can be used to identify datasets on
  the Hub with how the data was curated, such as `crowdsourced` or
  `machine_generated`.
- **language** (`str` or `List`, *optional*) --
  A string or list of strings representing a two-character language to
  filter datasets by on the Hub.
- **multilinguality** (`str` or `List`, *optional*) --
  A string or list of strings representing a filter for datasets that
  contain multiple languages.
- **size_categories** (`str` or `List`, *optional*) --
  A string or list of strings that can be used to identify datasets on
  the Hub by the size of the dataset such as `100K<n<1M` or
  `1M<n<10M`.
- **tags** (`str` or `List`, *optional*) --
  Deprecated. Pass tags in `filter` to filter datasets by tags.
- **task_categories** (`str` or `List`, *optional*) --
  A string or list of strings that can be used to identify datasets on
  the Hub by the designed task, such as `audio_classification` or
  `named_entity_recognition`.
- **task_ids** (`str` or `List`, *optional*) --
  A string or list of strings that can be used to identify datasets on
  the Hub by the specific task such as `speech_emotion_recognition` or
  `paraphrase`.
- **search** (`str`, *optional*) --
  A string that will be contained in the returned datasets.
- **sort** (`Literal["last_modified"]` or `str`, *optional*) --
  The key with which to sort the resulting models. Possible values are "last_modified", "trending_score",
  "created_at", "downloads" and "likes".
- **direction** (`Literal[-1]` or `int`, *optional*) --
  Direction in which to sort. The value `-1` sorts by descending
  order while all other values sort by ascending order.
- **limit** (`int`, *optional*) --
  The limit on the number of datasets fetched. Leaving this option
  to `None` fetches all datasets.
- **expand** (`list[ExpandDatasetProperty_T]`, *optional*) --
  List properties to return in the response. When used, only the properties in the list will be returned.
  This parameter cannot be used if `full` is passed.
  Possible values are `"author"`, `"cardData"`, `"citation"`, `"createdAt"`, `"disabled"`, `"description"`, `"downloads"`, `"downloadsAllTime"`, `"gated"`, `"lastModified"`, `"likes"`, `"paperswithcode_id"`, `"private"`, `"siblings"`, `"sha"`, `"tags"`, `"trendingScore"`, `"usedStorage"`, and `"resourceGroup"`.
- **full** (`bool`, *optional*) --
  Whether to fetch all dataset data, including the `last_modified`,
  the `card_data` and  the files. Can contain useful information such as the
  PapersWithCode ID.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`Iterable[DatasetInfo]`</rettype><retdesc>an iterable of [huggingface_hub.hf_api.DatasetInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.DatasetInfo) objects.</retdesc></docstring>

List datasets hosted on the Huggingface Hub, given some filters.







<ExampleCodeBlock anchor="huggingface_hub.HfApi.list_datasets.example">

Example usage with the `filter` argument:

```python
>>> from huggingface_hub import HfApi

>>> api = HfApi()

# List all datasets
>>> api.list_datasets()


# List only the text classification datasets
>>> api.list_datasets(filter="task_categories:text-classification")


# List only the datasets in russian for language modeling
>>> api.list_datasets(
...     filter=("language:ru", "task_ids:language-modeling")
... )

# List FiftyOne datasets (identified by the tag "fiftyone" in dataset card)
>>> api.list_datasets(tags="fiftyone")
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.HfApi.list_datasets.example-2">

Example usage with the `search` argument:

```python
>>> from huggingface_hub import HfApi

>>> api = HfApi()

# List all datasets with "text" in their name
>>> api.list_datasets(search="text")

# List all datasets with "text" in their name made by google
>>> api.list_datasets(search="text", author="google")
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>list_inference_catalog</name><anchor>huggingface_hub.HfApi.list_inference_catalog</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L7661</source><parameters>[{"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).</paramsdesc><paramgroups>0</paramgroups><rettype>List`str`</rettype><retdesc>A list of model IDs available in the catalog.</retdesc></docstring>
List models available in the Hugging Face Inference Catalog.

The goal of the Inference Catalog is to provide a curated list of models that are optimized for inference
and for which default configurations have been tested. See https://endpoints.huggingface.co/catalog for a list
of available models in the catalog.

Use [create_inference_endpoint_from_catalog()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.create_inference_endpoint_from_catalog) to deploy a model from the catalog.







> [!WARNING]
> `list_inference_catalog` is experimental. Its API is subject to change in the future. Please provide feedback
> if you have any suggestions or requests.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>list_inference_endpoints</name><anchor>huggingface_hub.HfApi.list_inference_endpoints</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L7322</source><parameters>[{"name": "namespace", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **namespace** (`str`, *optional*) --
  The namespace to list endpoints for. Defaults to the current user. Set to `"*"` to list all endpoints
  from all namespaces (i.e. personal namespace and all orgs the user belongs to).
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>list[InferenceEndpoint](/docs/huggingface_hub/main/en/package_reference/inference_endpoints#huggingface_hub.InferenceEndpoint)</rettype><retdesc>A list of all inference endpoints for the given namespace.</retdesc></docstring>
Lists all inference endpoints for the given namespace.







<ExampleCodeBlock anchor="huggingface_hub.HfApi.list_inference_endpoints.example">

Example:
```python
>>> from huggingface_hub import HfApi
>>> api = HfApi()
>>> api.list_inference_endpoints()
[InferenceEndpoint(name='my-endpoint', ...), ...]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>list_jobs</name><anchor>huggingface_hub.HfApi.list_jobs</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L10018</source><parameters>[{"name": "timeout", "val": ": Optional[int] = None"}, {"name": "namespace", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **timeout** (`float`, *optional*) --
  Whether to set a timeout for the request to the Hub.

- **namespace** (`str`, *optional*) --
  The namespace from where it lists the jobs. Defaults to the current user's namespace.

- **token** `(Union[bool, str, None]`, *optional*) --
  A valid user access token. If not provided, the locally saved token will be used, which is the
  recommended authentication method. Set to `False` to disable authentication.
  Refer to: https://huggingface.co/docs/huggingface_hub/quick-start#authentication.</paramsdesc><paramgroups>0</paramgroups></docstring>

List compute Jobs on Hugging Face infrastructure.




</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>list_lfs_files</name><anchor>huggingface_hub.HfApi.list_lfs_files</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L3473</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  The repository for which you are listing LFS files.
- **repo_type** (`str`, *optional*) --
  Type of repository. Set to `"dataset"` or `"space"` if listing from a dataset or space, `None` or
  `"model"` if listing from a model. Default is `None`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`Iterable[LFSFileInfo]`</rettype><retdesc>An iterator of `LFSFileInfo` objects.</retdesc></docstring>

List all LFS files in a repo on the Hub.

This is primarily useful to count how much storage a repo is using and to eventually clean up large files
with [permanently_delete_lfs_files()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.permanently_delete_lfs_files). Note that this would be a permanent action that will affect all commits
referencing this deleted files and that cannot be undone.







<ExampleCodeBlock anchor="huggingface_hub.HfApi.list_lfs_files.example">

Example:
```py
>>> from huggingface_hub import HfApi
>>> api = HfApi()
>>> lfs_files = api.list_lfs_files("username/my-cool-repo")

# Filter files files to delete based on a combination of `filename`, `pushed_at`, `ref` or `size`.
# e.g. select only LFS files in the "checkpoints" folder
>>> lfs_files_to_delete = (lfs_file for lfs_file in lfs_files if lfs_file.filename.startswith("checkpoints/"))

# Permanently delete LFS files
>>> api.permanently_delete_lfs_files("username/my-cool-repo", lfs_files_to_delete)
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>list_liked_repos</name><anchor>huggingface_hub.HfApi.list_liked_repos</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L2366</source><parameters>[{"name": "user", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **user** (`str`, *optional*) --
  Name of the user for which you want to fetch the likes.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[UserLikes](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.UserLikes)</rettype><retdesc>object containing the user name and 3 lists of repo ids (1 for
models, 1 for datasets and 1 for Spaces).</retdesc><raises>- [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError) -- 
  If `user` is not passed and no token found (either from argument or from machine).</raises><raisederrors>``ValueError``</raisederrors></docstring>

List all public repos liked by a user on huggingface.co.

This list is public so token is optional. If `user` is not passed, it defaults to
the logged in user.

See also [unlike()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.unlike).











<ExampleCodeBlock anchor="huggingface_hub.HfApi.list_liked_repos.example">

Example:
```python
>>> from huggingface_hub import list_liked_repos

>>> likes = list_liked_repos("julien-c")

>>> likes.user
"julien-c"

>>> likes.models
["osanseviero/streamlit_1.15", "Xhaheen/ChatGPT_HF", ...]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>list_models</name><anchor>huggingface_hub.HfApi.list_models</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L1792</source><parameters>[{"name": "filter", "val": ": Union[str, Iterable[str], None] = None"}, {"name": "author", "val": ": Optional[str] = None"}, {"name": "apps", "val": ": Optional[Union[str, list[str]]] = None"}, {"name": "gated", "val": ": Optional[bool] = None"}, {"name": "inference", "val": ": Optional[Literal['warm']] = None"}, {"name": "inference_provider", "val": ": Optional[Union[Literal['all'], 'PROVIDER_T', list['PROVIDER_T']]] = None"}, {"name": "model_name", "val": ": Optional[str] = None"}, {"name": "trained_dataset", "val": ": Optional[Union[str, list[str]]] = None"}, {"name": "search", "val": ": Optional[str] = None"}, {"name": "pipeline_tag", "val": ": Optional[str] = None"}, {"name": "emissions_thresholds", "val": ": Optional[tuple[float, float]] = None"}, {"name": "sort", "val": ": Union[Literal['last_modified'], str, None] = None"}, {"name": "direction", "val": ": Optional[Literal[-1]] = None"}, {"name": "limit", "val": ": Optional[int] = None"}, {"name": "expand", "val": ": Optional[list[ExpandModelProperty_T]] = None"}, {"name": "full", "val": ": Optional[bool] = None"}, {"name": "cardData", "val": ": bool = False"}, {"name": "fetch_config", "val": ": bool = False"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **filter** (`str` or `Iterable[str]`, *optional*) --
  A string or list of string to filter models on the Hub.
  Models can be filtered by library, language, task, tags, and more.
- **author** (`str`, *optional*) --
  A string which identify the author (user or organization) of the
  returned models.
- **apps** (`str` or `List`, *optional*) --
  A string or list of strings to filter models on the Hub that
  support the specified apps. Example values include `"ollama"` or `["ollama", "vllm"]`.
- **gated** (`bool`, *optional*) --
  A boolean to filter models on the Hub that are gated or not. By default, all models are returned.
  If `gated=True` is passed, only gated models are returned.
  If `gated=False` is passed, only non-gated models are returned.
- **inference** (`Literal["warm"]`, *optional*) --
  If "warm", filter models on the Hub currently served by at least one provider.
- **inference_provider** (`Literal["all"]` or `str`, *optional*) --
  A string to filter models on the Hub that are served by a specific provider.
  Pass `"all"` to get all models served by at least one provider.
- **model_name** (`str`, *optional*) --
  A string that contain complete or partial names for models on the
  Hub, such as "bert" or "bert-base-cased"
- **trained_dataset** (`str` or `List`, *optional*) --
  A string tag or a list of string tags of the trained dataset for a
  model on the Hub.
- **search** (`str`, *optional*) --
  A string that will be contained in the returned model ids.
- **pipeline_tag** (`str`, *optional*) --
  A string pipeline tag to filter models on the Hub by, such as `summarization`.
- **emissions_thresholds** (`Tuple`, *optional*) --
  A tuple of two ints or floats representing a minimum and maximum
  carbon footprint to filter the resulting models with in grams.
- **sort** (`Literal["last_modified"]` or `str`, *optional*) --
  The key with which to sort the resulting models. Possible values are "last_modified", "trending_score",
  "created_at", "downloads" and "likes".
- **direction** (`Literal[-1]` or `int`, *optional*) --
  Direction in which to sort. The value `-1` sorts by descending
  order while all other values sort by ascending order.
- **limit** (`int`, *optional*) --
  The limit on the number of models fetched. Leaving this option
  to `None` fetches all models.
- **expand** (`list[ExpandModelProperty_T]`, *optional*) --
  List properties to return in the response. When used, only the properties in the list will be returned.
  This parameter cannot be used if `full`, `cardData` or `fetch_config` are passed.
  Possible values are `"author"`, `"cardData"`, `"config"`, `"createdAt"`, `"disabled"`, `"downloads"`, `"downloadsAllTime"`, `"gated"`, `"gguf"`, `"inference"`, `"inferenceProviderMapping"`, `"lastModified"`, `"library_name"`, `"likes"`, `"mask_token"`, `"model-index"`, `"pipeline_tag"`, `"private"`, `"safetensors"`, `"sha"`, `"siblings"`, `"spaces"`, `"tags"`, `"transformersInfo"`, `"trendingScore"`, `"widgetData"`, and `"resourceGroup"`.
- **full** (`bool`, *optional*) --
  Whether to fetch all model data, including the `last_modified`,
  the `sha`, the files and the `tags`. This is set to `True` by
  default when using a filter.
- **cardData** (`bool`, *optional*) --
  Whether to grab the metadata for the model as well. Can contain
  useful information such as carbon emissions, metrics, and
  datasets trained on.
- **fetch_config** (`bool`, *optional*) --
  Whether to fetch the model configs as well. This is not included
  in `full` due to its size.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`Iterable[ModelInfo]`</rettype><retdesc>an iterable of [huggingface_hub.hf_api.ModelInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.ModelInfo) objects.</retdesc></docstring>

List models hosted on the Huggingface Hub, given some filters.







<ExampleCodeBlock anchor="huggingface_hub.HfApi.list_models.example">

Example:

```python
>>> from huggingface_hub import HfApi

>>> api = HfApi()

# List all models
>>> api.list_models()

# List text classification models
>>> api.list_models(filter="text-classification")

# List models from the KerasHub library
>>> api.list_models(filter="keras-hub")

# List models served by Cohere
>>> api.list_models(inference_provider="cohere")

# List models with "bert" in their name
>>> api.list_models(search="bert")

# List models with "bert" in their name and pushed by google
>>> api.list_models(search="bert", author="google")
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>list_organization_followers</name><anchor>huggingface_hub.HfApi.list_organization_followers</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L9601</source><parameters>[{"name": "organization", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **organization** (`str`) --
  Name of the organization to get the followers of.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`Iterable[User]`</rettype><retdesc>A list of [User](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.User) objects with the followers of the organization.</retdesc><raises>- `HfHubHTTPError` -- 
  HTTP 404 If the organization does not exist on the Hub.</raises><raisederrors>`HfHubHTTPError`</raisederrors></docstring>

List followers of an organization on the Hub.












</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>list_organization_members</name><anchor>huggingface_hub.HfApi.list_organization_members</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L9630</source><parameters>[{"name": "organization", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **organization** (`str`) --
  Name of the organization to get the members of.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`Iterable[User]`</rettype><retdesc>A list of [User](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.User) objects with the members of the organization.</retdesc><raises>- `HfHubHTTPError` -- 
  HTTP 404 If the organization does not exist on the Hub.</raises><raisederrors>`HfHubHTTPError`</raisederrors></docstring>

List of members of an organization on the Hub.












</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>list_papers</name><anchor>huggingface_hub.HfApi.list_papers</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L9714</source><parameters>[{"name": "query", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **query** (`str`, *optional*) --
  A search query string to find papers.
  If provided, returns papers that match the query.
- **token** (Union[bool, str, None], *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`Iterable[PaperInfo]`</rettype><retdesc>an iterable of `huggingface_hub.hf_api.PaperInfo` objects.</retdesc></docstring>

List daily papers on the Hugging Face Hub given a search query.







<ExampleCodeBlock anchor="huggingface_hub.HfApi.list_papers.example">

Example:

```python
>>> from huggingface_hub import HfApi

>>> api = HfApi()

# List all papers with "attention" in their title
>>> api.list_papers(query="attention")
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>list_pending_access_requests</name><anchor>huggingface_hub.HfApi.list_pending_access_requests</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L8493</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  The id of the repo to get access requests for.
- **repo_type** (`str`, *optional*) --
  The type of the repo to get access requests for. Must be one of `model`, `dataset` or `space`.
  Defaults to `model`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[AccessRequest]`</rettype><retdesc>A list of `AccessRequest` objects. Each time contains a `username`, `email`,
`status` and `timestamp` attribute. If the gated repo has a custom form, the `fields` attribute will
be populated with user's answers.</retdesc><raises>- `HfHubHTTPError` -- 
  HTTP 400 if the repo is not gated.
- `HfHubHTTPError` -- 
  HTTP 403 if you only have read-only access to the repo. This can be the case if you don't have `write`
  or `admin` role in the organization the repo belongs to or if you passed a `read` token.</raises><raisederrors>`HfHubHTTPError`</raisederrors></docstring>

Get pending access requests for a given gated repo.

A pending request means the user has requested access to the repo but the request has not been processed yet.
If the approval mode is automatic, this list should be empty. Pending requests can be accepted or rejected
using [accept_access_request()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.accept_access_request) and [reject_access_request()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.reject_access_request).

For more info about gated repos, see https://huggingface.co/docs/hub/models-gated.











<ExampleCodeBlock anchor="huggingface_hub.HfApi.list_pending_access_requests.example">

Example:
```py
>>> from huggingface_hub import list_pending_access_requests, accept_access_request

# List pending requests
>>> requests = list_pending_access_requests("meta-llama/Llama-2-7b")
>>> len(requests)
411
>>> requests[0]
[
    AccessRequest(
        username='clem',
        fullname='Clem 🤗',
        email='***',
        timestamp=datetime.datetime(2023, 11, 23, 18, 4, 53, 828000, tzinfo=datetime.timezone.utc),
        status='pending',
        fields=None,
    ),
    ...
]

# Accept Clem's request
>>> accept_access_request("meta-llama/Llama-2-7b", "clem")
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>list_rejected_access_requests</name><anchor>huggingface_hub.HfApi.list_rejected_access_requests</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L8619</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  The id of the repo to get access requests for.
- **repo_type** (`str`, *optional*) --
  The type of the repo to get access requests for. Must be one of `model`, `dataset` or `space`.
  Defaults to `model`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[AccessRequest]`</rettype><retdesc>A list of `AccessRequest` objects. Each time contains a `username`, `email`,
`status` and `timestamp` attribute. If the gated repo has a custom form, the `fields` attribute will
be populated with user's answers.</retdesc><raises>- `HfHubHTTPError` -- 
  HTTP 400 if the repo is not gated.
- `HfHubHTTPError` -- 
  HTTP 403 if you only have read-only access to the repo. This can be the case if you don't have `write`
  or `admin` role in the organization the repo belongs to or if you passed a `read` token.</raises><raisederrors>`HfHubHTTPError`</raisederrors></docstring>

Get rejected access requests for a given gated repo.

A rejected request means the user has requested access to the repo and the request has been explicitly rejected
by a repo owner (either you or another user from your organization). The user cannot download any file of the
repo. Rejected requests can be accepted or cancelled at any time using [accept_access_request()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.accept_access_request) and
[cancel_access_request()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.cancel_access_request). A cancelled request will go back to the pending list while an accepted request will
go to the accepted list.

For more info about gated repos, see https://huggingface.co/docs/hub/models-gated.











<ExampleCodeBlock anchor="huggingface_hub.HfApi.list_rejected_access_requests.example">

Example:
```py
>>> from huggingface_hub import list_rejected_access_requests

>>> requests = list_rejected_access_requests("meta-llama/Llama-2-7b")
>>> len(requests)
411
>>> requests[0]
[
    AccessRequest(
        username='clem',
        fullname='Clem 🤗',
        email='***',
        timestamp=datetime.datetime(2023, 11, 23, 18, 4, 53, 828000, tzinfo=datetime.timezone.utc),
        status='rejected',
        fields=None,
    ),
    ...
]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>list_repo_commits</name><anchor>huggingface_hub.HfApi.list_repo_commits</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L3230</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "formatted", "val": ": bool = False"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated by a `/`.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if listing commits from a dataset or a Space, `None` or `"model"` if
  listing from a model. Default is `None`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.
- **revision** (`str`, *optional*) --
  The git revision to commit from. Defaults to the head of the `"main"` branch.
- **formatted** (`bool`) --
  Whether to return the HTML-formatted title and description of the commits. Defaults to False.</paramsdesc><paramgroups>0</paramgroups><rettype>list[[GitCommitInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.GitCommitInfo)]</rettype><retdesc>list of objects containing information about the commits for a repo on the Hub.</retdesc><raises>- [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) -- 
  If repository is not found (error 404): wrong repo_id/repo_type, private but not authenticated or repo
  does not exist.
- [RevisionNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RevisionNotFoundError) -- 
  If revision is not found (error 404) on the repo.</raises><raisederrors>[RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) or [RevisionNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RevisionNotFoundError)</raisederrors></docstring>

Get the list of commits of a given revision for a repo on the Hub.

Commits are sorted by date (last commit first).



<ExampleCodeBlock anchor="huggingface_hub.HfApi.list_repo_commits.example">

Example:
```py
>>> from huggingface_hub import HfApi
>>> api = HfApi()

# Commits are sorted by date (last commit first)
>>> initial_commit = api.list_repo_commits("gpt2")[-1]

# Initial commit is always a system commit containing the `.gitattributes` file.
>>> initial_commit
GitCommitInfo(
    commit_id='9b865efde13a30c13e0a33e536cf3e4a5a9d71d8',
    authors=['system'],
    created_at=datetime.datetime(2019, 2, 18, 10, 36, 15, tzinfo=datetime.timezone.utc),
    title='initial commit',
    message='',
    formatted_title=None,
    formatted_message=None
)

# Create an empty branch by deriving from initial commit
>>> api.create_branch("gpt2", "new_empty_branch", revision=initial_commit.commit_id)
```

</ExampleCodeBlock>










</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>list_repo_files</name><anchor>huggingface_hub.HfApi.list_repo_files</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L2916</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[str, bool, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated by a `/`.
- **revision** (`str`, *optional*) --
  The revision of the repository from which to get the information.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if uploading to a dataset or space, `None` or `"model"` if uploading to
  a model. Default is `None`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[str]`</rettype><retdesc>the list of files in a given repository.</retdesc></docstring>

Get the list of files in a given repo.








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>list_repo_likers</name><anchor>huggingface_hub.HfApi.list_repo_likers</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L2442</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  The repository to retrieve . Example: `"user/my-cool-model"`.

- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.

- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if uploading to a dataset or
  space, `None` or `"model"` if uploading to a model. Default is
  `None`.</paramsdesc><paramgroups>0</paramgroups><rettype>`Iterable[User]`</rettype><retdesc>an iterable of [huggingface_hub.hf_api.User](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.User) objects.</retdesc></docstring>

List all users who liked a given repo on the hugging Face Hub.

See also [list_liked_repos()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.list_liked_repos).








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>list_repo_refs</name><anchor>huggingface_hub.HfApi.list_repo_refs</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L3158</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "include_pull_requests", "val": ": bool = False"}, {"name": "token", "val": ": Union[str, bool, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated
  by a `/`.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if listing refs from a dataset or a Space,
  `None` or `"model"` if listing from a model. Default is `None`.
- **include_pull_requests** (`bool`, *optional*) --
  Whether to include refs from pull requests in the list. Defaults to `False`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[GitRefs](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.GitRefs)</rettype><retdesc>object containing all information about branches and tags for a
repo on the Hub.</retdesc></docstring>

Get the list of refs of a given repo (both tags and branches).



<ExampleCodeBlock anchor="huggingface_hub.HfApi.list_repo_refs.example">

Example:
```py
>>> from huggingface_hub import HfApi
>>> api = HfApi()
>>> api.list_repo_refs("gpt2")
GitRefs(branches=[GitRefInfo(name='main', ref='refs/heads/main', target_commit='e7da7f221d5bf496a48136c0cd264e630fe9fcc8')], converts=[], tags=[])

>>> api.list_repo_refs("bigcode/the-stack", repo_type='dataset')
GitRefs(
    branches=[
        GitRefInfo(name='main', ref='refs/heads/main', target_commit='18edc1591d9ce72aa82f56c4431b3c969b210ae3'),
        GitRefInfo(name='v1.1.a1', ref='refs/heads/v1.1.a1', target_commit='f9826b862d1567f3822d3d25649b0d6d22ace714')
    ],
    converts=[],
    tags=[
        GitRefInfo(name='v1.0', ref='refs/tags/v1.0', target_commit='c37a8cd1e382064d8aced5e05543c5f7753834da')
    ]
)
```

</ExampleCodeBlock>






</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>list_repo_tree</name><anchor>huggingface_hub.HfApi.list_repo_tree</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L2953</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "path_in_repo", "val": ": Optional[str] = None"}, {"name": "recursive", "val": ": bool = False"}, {"name": "expand", "val": ": bool = False"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[str, bool, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated by a `/`.
- **path_in_repo** (`str`, *optional*) --
  Relative path of the tree (folder) in the repo, for example:
  `"checkpoints/1fec34a/results"`. Will default to the root tree (folder) of the repository.
- **recursive** (`bool`, *optional*, defaults to `False`) --
  Whether to list tree's files and folders recursively.
- **expand** (`bool`, *optional*, defaults to `False`) --
  Whether to fetch more information about the tree's files and folders (e.g. last commit and files' security scan results). This
  operation is more expensive for the server so only 50 results are returned per page (instead of 1000).
  As pagination is implemented in `huggingface_hub`, this is transparent for you except for the time it
  takes to get the results.
- **revision** (`str`, *optional*) --
  The revision of the repository from which to get the tree. Defaults to `"main"` branch.
- **repo_type** (`str`, *optional*) --
  The type of the repository from which to get the tree (`"model"`, `"dataset"` or `"space"`.
  Defaults to `"model"`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`Iterable[Union[RepoFile, RepoFolder]]`</rettype><retdesc>The information about the tree's files and folders, as an iterable of `RepoFile` and `RepoFolder` objects. The order of the files and folders is
not guaranteed.</retdesc><raises>- [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) -- 
  If repository is not found (error 404): wrong repo_id/repo_type, private but not authenticated or repo
  does not exist.
- [RevisionNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RevisionNotFoundError) -- 
  If revision is not found (error 404) on the repo.
- `~utils.RemoteEntryNotFoundError` -- 
  If the tree (folder) does not exist (error 404) on the repo.</raises><raisederrors>[RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) or [RevisionNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RevisionNotFoundError) or `~utils.RemoteEntryNotFoundError`</raisederrors></docstring>

List a repo tree's files and folders and get information about them.











Examples:

Get information about a repo's tree.
<ExampleCodeBlock anchor="huggingface_hub.HfApi.list_repo_tree.example">

```py
>>> from huggingface_hub import list_repo_tree
>>> repo_tree = list_repo_tree("lysandre/arxiv-nlp")
>>> repo_tree
<generator object HfApi.list_repo_tree at 0x7fa4088e1ac0>
>>> list(repo_tree)
[
    RepoFile(path='.gitattributes', size=391, blob_id='ae8c63daedbd4206d7d40126955d4e6ab1c80f8f', lfs=None, last_commit=None, security=None),
    RepoFile(path='README.md', size=391, blob_id='43bd404b159de6fba7c2f4d3264347668d43af25', lfs=None, last_commit=None, security=None),
    RepoFile(path='config.json', size=554, blob_id='2f9618c3a19b9a61add74f70bfb121335aeef666', lfs=None, last_commit=None, security=None),
    RepoFile(
        path='flax_model.msgpack', size=497764107, blob_id='8095a62ccb4d806da7666fcda07467e2d150218e',
        lfs={'size': 497764107, 'sha256': 'd88b0d6a6ff9c3f8151f9d3228f57092aaea997f09af009eefd7373a77b5abb9', 'pointer_size': 134}, last_commit=None, security=None
    ),
    RepoFile(path='merges.txt', size=456318, blob_id='226b0752cac7789c48f0cb3ec53eda48b7be36cc', lfs=None, last_commit=None, security=None),
    RepoFile(
        path='pytorch_model.bin', size=548123560, blob_id='64eaa9c526867e404b68f2c5d66fd78e27026523',
        lfs={'size': 548123560, 'sha256': '9be78edb5b928eba33aa88f431551348f7466ba9f5ef3daf1d552398722a5436', 'pointer_size': 134}, last_commit=None, security=None
    ),
    RepoFile(path='vocab.json', size=898669, blob_id='b00361fece0387ca34b4b8b8539ed830d644dbeb', lfs=None, last_commit=None, security=None)]
]
```

</ExampleCodeBlock>

Get even more information about a repo's tree (last commit and files' security scan results)
<ExampleCodeBlock anchor="huggingface_hub.HfApi.list_repo_tree.example-2">

```py
>>> from huggingface_hub import list_repo_tree
>>> repo_tree = list_repo_tree("prompthero/openjourney-v4", expand=True)
>>> list(repo_tree)
[
    RepoFolder(
        path='feature_extractor',
        tree_id='aa536c4ea18073388b5b0bc791057a7296a00398',
        last_commit={
            'oid': '47b62b20b20e06b9de610e840282b7e6c3d51190',
            'title': 'Upload diffusers weights (#48)',
            'date': datetime.datetime(2023, 3, 21, 9, 5, 27, tzinfo=datetime.timezone.utc)
        }
    ),
    RepoFolder(
        path='safety_checker',
        tree_id='65aef9d787e5557373fdf714d6c34d4fcdd70440',
        last_commit={
            'oid': '47b62b20b20e06b9de610e840282b7e6c3d51190',
            'title': 'Upload diffusers weights (#48)',
            'date': datetime.datetime(2023, 3, 21, 9, 5, 27, tzinfo=datetime.timezone.utc)
        }
    ),
    RepoFile(
        path='model_index.json',
        size=582,
        blob_id='d3d7c1e8c3e78eeb1640b8e2041ee256e24c9ee1',
        lfs=None,
        last_commit={
            'oid': 'b195ed2d503f3eb29637050a886d77bd81d35f0e',
            'title': 'Fix deprecation warning by changing `CLIPFeatureExtractor` to `CLIPImageProcessor`. (#54)',
            'date': datetime.datetime(2023, 5, 15, 21, 41, 59, tzinfo=datetime.timezone.utc)
        },
        security={
            'safe': True,
            'av_scan': {'virusFound': False, 'virusNames': None},
            'pickle_import_scan': None
        }
    )
    ...
]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>list_scheduled_jobs</name><anchor>huggingface_hub.HfApi.list_scheduled_jobs</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L10358</source><parameters>[{"name": "timeout", "val": ": Optional[int] = None"}, {"name": "namespace", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **timeout** (`float`, *optional*) --
  Whether to set a timeout for the request to the Hub.

- **namespace** (`str`, *optional*) --
  The namespace from where it lists the jobs. Defaults to the current user's namespace.

- **token** `(Union[bool, str, None]`, *optional*) --
  A valid user access token. If not provided, the locally saved token will be used, which is the
  recommended authentication method. Set to `False` to disable authentication.
  Refer to: https://huggingface.co/docs/huggingface_hub/quick-start#authentication.</paramsdesc><paramgroups>0</paramgroups></docstring>

List scheduled compute Jobs on Hugging Face infrastructure.




</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>list_spaces</name><anchor>huggingface_hub.HfApi.list_spaces</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L2205</source><parameters>[{"name": "filter", "val": ": Union[str, Iterable[str], None] = None"}, {"name": "author", "val": ": Optional[str] = None"}, {"name": "search", "val": ": Optional[str] = None"}, {"name": "datasets", "val": ": Union[str, Iterable[str], None] = None"}, {"name": "models", "val": ": Union[str, Iterable[str], None] = None"}, {"name": "linked", "val": ": bool = False"}, {"name": "sort", "val": ": Union[Literal['last_modified'], str, None] = None"}, {"name": "direction", "val": ": Optional[Literal[-1]] = None"}, {"name": "limit", "val": ": Optional[int] = None"}, {"name": "expand", "val": ": Optional[list[ExpandSpaceProperty_T]] = None"}, {"name": "full", "val": ": Optional[bool] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **filter** (`str` or `Iterable`, *optional*) --
  A string tag or list of tags that can be used to identify Spaces on the Hub.
- **author** (`str`, *optional*) --
  A string which identify the author of the returned Spaces.
- **search** (`str`, *optional*) --
  A string that will be contained in the returned Spaces.
- **datasets** (`str` or `Iterable`, *optional*) --
  Whether to return Spaces that make use of a dataset.
  The name of a specific dataset can be passed as a string.
- **models** (`str` or `Iterable`, *optional*) --
  Whether to return Spaces that make use of a model.
  The name of a specific model can be passed as a string.
- **linked** (`bool`, *optional*) --
  Whether to return Spaces that make use of either a model or a dataset.
- **sort** (`Literal["last_modified"]` or `str`, *optional*) --
  The key with which to sort the resulting models. Possible values are "last_modified", "trending_score",
  "created_at" and "likes".
- **direction** (`Literal[-1]` or `int`, *optional*) --
  Direction in which to sort. The value `-1` sorts by descending
  order while all other values sort by ascending order.
- **limit** (`int`, *optional*) --
  The limit on the number of Spaces fetched. Leaving this option
  to `None` fetches all Spaces.
- **expand** (`list[ExpandSpaceProperty_T]`, *optional*) --
  List properties to return in the response. When used, only the properties in the list will be returned.
  This parameter cannot be used if `full` is passed.
  Possible values are `"author"`, `"cardData"`, `"datasets"`, `"disabled"`, `"lastModified"`, `"createdAt"`, `"likes"`, `"models"`, `"private"`, `"runtime"`, `"sdk"`, `"siblings"`, `"sha"`, `"subdomain"`, `"tags"`, `"trendingScore"`, `"usedStorage"`, and `"resourceGroup"`.
- **full** (`bool`, *optional*) --
  Whether to fetch all Spaces data, including the `last_modified`, `siblings`
  and `card_data` fields.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`Iterable[SpaceInfo]`</rettype><retdesc>an iterable of [huggingface_hub.hf_api.SpaceInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.SpaceInfo) objects.</retdesc></docstring>

List spaces hosted on the Huggingface Hub, given some filters.








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>list_user_followers</name><anchor>huggingface_hub.HfApi.list_user_followers</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L9658</source><parameters>[{"name": "username", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **username** (`str`) --
  Username of the user to get the followers of.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`Iterable[User]`</rettype><retdesc>A list of [User](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.User) objects with the followers of the user.</retdesc><raises>- `HfHubHTTPError` -- 
  HTTP 404 If the user does not exist on the Hub.</raises><raisederrors>`HfHubHTTPError`</raisederrors></docstring>

Get the list of followers of a user on the Hub.












</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>list_user_following</name><anchor>huggingface_hub.HfApi.list_user_following</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L9686</source><parameters>[{"name": "username", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **username** (`str`) --
  Username of the user to get the users followed by.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`Iterable[User]`</rettype><retdesc>A list of [User](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.User) objects with the users followed by the user.</retdesc><raises>- `HfHubHTTPError` -- 
  HTTP 404 If the user does not exist on the Hub.</raises><raisederrors>`HfHubHTTPError`</raisederrors></docstring>

Get the list of users followed by a user on the Hub.












</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>list_webhooks</name><anchor>huggingface_hub.HfApi.list_webhooks</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L8981</source><parameters>[{"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved token, which is the recommended
  method for authentication (see https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[WebhookInfo]`</rettype><retdesc>List of webhook info objects.</retdesc></docstring>
List all configured webhooks.







<ExampleCodeBlock anchor="huggingface_hub.HfApi.list_webhooks.example">

Example:
```python
>>> from huggingface_hub import list_webhooks
>>> webhooks = list_webhooks()
>>> len(webhooks)
2
>>> webhooks[0]
WebhookInfo(
    id="654bbbc16f2ec14d77f109cc",
    watched=[WebhookWatchedItem(type="user", name="julien-c"), WebhookWatchedItem(type="org", name="HuggingFaceH4")],
    url="https://webhook.site/a2176e82-5720-43ee-9e06-f91cb4c91548",
    secret="my-secret",
    domains=["repo", "discussion"],
    disabled=False,
)
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>merge_pull_request</name><anchor>huggingface_hub.HfApi.merge_pull_request</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L6600</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "discussion_num", "val": ": int"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "comment", "val": ": Optional[str] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated
  by a `/`.
- **discussion_num** (`int`) --
  The number of the Discussion or Pull Request . Must be a strictly positive integer.
- **comment** (`str`, *optional*) --
  An optional comment to post with the status change.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if uploading to a dataset or
  space, `None` or `"model"` if uploading to a model. Default is
  `None`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[DiscussionStatusChange](/docs/huggingface_hub/main/en/package_reference/community#huggingface_hub.DiscussionStatusChange)</rettype><retdesc>the status change event</retdesc></docstring>
Merges a Pull Request.







> [!TIP]
> Raises the following errors:
>
>     - [`HTTPError`](https://requests.readthedocs.io/en/latest/api/#requests.HTTPError)
>       if the HuggingFace API returned an error
>     - [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError)
>       if some parameter value is invalid
>     - [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError)
>       If the repository to download from cannot be found. This may be because it doesn't exist,
>       or because it is set to `private` and you do not have access.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>model_info</name><anchor>huggingface_hub.HfApi.model_info</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L2481</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "timeout", "val": ": Optional[float] = None"}, {"name": "securityStatus", "val": ": Optional[bool] = None"}, {"name": "files_metadata", "val": ": bool = False"}, {"name": "expand", "val": ": Optional[list[ExpandModelProperty_T]] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated
  by a `/`.
- **revision** (`str`, *optional*) --
  The revision of the model repository from which to get the
  information.
- **timeout** (`float`, *optional*) --
  Whether to set a timeout for the request to the Hub.
- **securityStatus** (`bool`, *optional*) --
  Whether to retrieve the security status from the model
  repository as well. The security status will be returned in the `security_repo_status` field.
- **files_metadata** (`bool`, *optional*) --
  Whether or not to retrieve metadata for files in the repository
  (size, LFS metadata, etc). Defaults to `False`.
- **expand** (`list[ExpandModelProperty_T]`, *optional*) --
  List properties to return in the response. When used, only the properties in the list will be returned.
  This parameter cannot be used if `securityStatus` or `files_metadata` are passed.
  Possible values are `"author"`, `"baseModels"`, `"cardData"`, `"childrenModelCount"`, `"config"`, `"createdAt"`, `"disabled"`, `"downloads"`, `"downloadsAllTime"`, `"gated"`, `"gguf"`, `"inference"`, `"inferenceProviderMapping"`, `"lastModified"`, `"library_name"`, `"likes"`, `"mask_token"`, `"model-index"`, `"pipeline_tag"`, `"private"`, `"safetensors"`, `"sha"`, `"siblings"`, `"spaces"`, `"tags"`, `"transformersInfo"`, `"trendingScore"`, `"widgetData"`, `"usedStorage"`, and `"resourceGroup"`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[huggingface_hub.hf_api.ModelInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.ModelInfo)</rettype><retdesc>The model repository information.</retdesc></docstring>

Get info on one specific model on huggingface.co

Model can be private if you pass an acceptable token or are logged in.







> [!TIP]
> Raises the following errors:
>
>     - [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError)
>       If the repository to download from cannot be found. This may be because it doesn't exist,
>       or because it is set to `private` and you do not have access.
>     - [RevisionNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RevisionNotFoundError)
>       If the revision to download from cannot be found.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>move_repo</name><anchor>huggingface_hub.HfApi.move_repo</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L3864</source><parameters>[{"name": "from_id", "val": ": str"}, {"name": "to_id", "val": ": str"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[str, bool, None] = None"}]</parameters><paramsdesc>- **from_id** (`str`) --
  A namespace (user or an organization) and a repo name separated
  by a `/`. Original repository identifier.
- **to_id** (`str`) --
  A namespace (user or an organization) and a repo name separated
  by a `/`. Final repository identifier.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if uploading to a dataset or
  space, `None` or `"model"` if uploading to a model. Default is
  `None`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups></docstring>

Moving a repository from namespace1/repo_name1 to namespace2/repo_name2

Note there are certain limitations. For more information about moving
repositories, please see
https://hf.co/docs/hub/repositories-settings#renaming-or-transferring-a-repo.



> [!TIP]
> Raises the following errors:
>
>     - [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError)
>       If the repository to download from cannot be found. This may be because it doesn't exist,
>       or because it is set to `private` and you do not have access.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>paper_info</name><anchor>huggingface_hub.HfApi.paper_info</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L9760</source><parameters>[{"name": "id", "val": ": str"}]</parameters><paramsdesc>- **id** (`str`, **optional**) --
  ArXiv id of the paper.</paramsdesc><paramgroups>0</paramgroups><rettype>`PaperInfo`</rettype><retdesc>A `PaperInfo` object.</retdesc><raises>- `HfHubHTTPError` -- 
  HTTP 404 If the paper does not exist on the Hub.</raises><raisederrors>`HfHubHTTPError`</raisederrors></docstring>

Get information for a paper on the Hub.












</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>parse_safetensors_file_metadata</name><anchor>huggingface_hub.HfApi.parse_safetensors_file_metadata</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L5629</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "filename", "val": ": str"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A user or an organization name and a repo name separated by a `/`.
- **filename** (`str`) --
  The name of the file in the repo.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if the file is in a dataset or space, `None` or `"model"` if in a
  model. Default is `None`.
- **revision** (`str`, *optional*) --
  The git revision to fetch the file from. Can be a branch name, a tag, or a commit hash. Defaults to the
  head of the `"main"` branch.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`SafetensorsFileMetadata`</rettype><retdesc>information related to a safetensors file.</retdesc><raises>- `NotASafetensorsRepoError` -- 
  If the repo is not a safetensors repo i.e. doesn't have either a
  `model.safetensors` or a `model.safetensors.index.json` file.
- `SafetensorsParsingError` -- 
  If a safetensors file header couldn't be parsed correctly.</raises><raisederrors>`NotASafetensorsRepoError` or `SafetensorsParsingError`</raisederrors></docstring>

Parse metadata from a safetensors file on the Hub.

To parse metadata from all safetensors files in a repo at once, use [get_safetensors_metadata()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.get_safetensors_metadata).

For more details regarding the safetensors format, check out https://huggingface.co/docs/safetensors/index#format.












</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>pause_inference_endpoint</name><anchor>huggingface_hub.HfApi.pause_inference_endpoint</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L7903</source><parameters>[{"name": "name", "val": ": str"}, {"name": "namespace", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **name** (`str`) --
  The name of the Inference Endpoint to pause.
- **namespace** (`str`, *optional*) --
  The namespace in which the Inference Endpoint is located. Defaults to the current user.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[InferenceEndpoint](/docs/huggingface_hub/main/en/package_reference/inference_endpoints#huggingface_hub.InferenceEndpoint)</rettype><retdesc>information about the paused Inference Endpoint.</retdesc></docstring>
Pause an Inference Endpoint.

A paused Inference Endpoint will not be charged. It can be resumed at any time using [resume_inference_endpoint()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.resume_inference_endpoint).
This is different than scaling the Inference Endpoint to zero with [scale_to_zero_inference_endpoint()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.scale_to_zero_inference_endpoint), which
would be automatically restarted when a request is made to it.

For convenience, you can also pause an Inference Endpoint using [pause_inference_endpoint()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.pause_inference_endpoint).








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>pause_space</name><anchor>huggingface_hub.HfApi.pause_space</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L7048</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  ID of the Space to pause. Example: `"Salesforce/BLIP2"`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[SpaceRuntime](/docs/huggingface_hub/main/en/package_reference/space_runtime#huggingface_hub.SpaceRuntime)</rettype><retdesc>Runtime information about your Space including `stage=PAUSED` and requested hardware.</retdesc><raises>- [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) -- 
  If your Space is not found (error 404). Most probably wrong repo_id or your space is private but you
  are not authenticated.
- [HfHubHTTPError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.HfHubHTTPError) -- 
  403 Forbidden: only the owner of a Space can pause it. If you want to manage a Space that you don't
  own, either ask the owner by opening a Discussion or duplicate the Space.
- [BadRequestError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.BadRequestError) -- 
  If your Space is a static Space. Static Spaces are always running and never billed. If you want to hide
  a static Space, you can set it to private.</raises><raisederrors>[RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) or [HfHubHTTPError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.HfHubHTTPError) or [BadRequestError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.BadRequestError)</raisederrors></docstring>
Pause your Space.

A paused Space stops executing until manually restarted by its owner. This is different from the sleeping
state in which free Spaces go after 48h of inactivity. Paused time is not billed to your account, no matter the
hardware you've selected. To restart your Space, use [restart_space()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.restart_space) and go to your Space settings page.

For more details, please visit [the docs](https://huggingface.co/docs/hub/spaces-gpus#pause).












</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>permanently_delete_lfs_files</name><anchor>huggingface_hub.HfApi.permanently_delete_lfs_files</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L3527</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "lfs_files", "val": ": Iterable[LFSFileInfo]"}, {"name": "rewrite_history", "val": ": bool = True"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  The repository for which you are listing LFS files.
- **lfs_files** (`Iterable[LFSFileInfo]`) --
  An iterable of `LFSFileInfo` items to permanently delete from the repo. Use [list_lfs_files()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.list_lfs_files) to list
  all LFS files from a repo.
- **rewrite_history** (`bool`, *optional*, default to `True`) --
  Whether to rewrite repository history to remove file pointers referencing the deleted LFS files (recommended).
- **repo_type** (`str`, *optional*) --
  Type of repository. Set to `"dataset"` or `"space"` if listing from a dataset or space, `None` or
  `"model"` if listing from a model. Default is `None`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups></docstring>

Permanently delete LFS files from a repo on the Hub.

> [!WARNING]
> This is a permanent action that will affect all commits referencing the deleted files and might corrupt your
> repository. This is a non-revertible operation. Use it only if you know what you are doing.



<ExampleCodeBlock anchor="huggingface_hub.HfApi.permanently_delete_lfs_files.example">

Example:
```py
>>> from huggingface_hub import HfApi
>>> api = HfApi()
>>> lfs_files = api.list_lfs_files("username/my-cool-repo")

# Filter files files to delete based on a combination of `filename`, `pushed_at`, `ref` or `size`.
# e.g. select only LFS files in the "checkpoints" folder
>>> lfs_files_to_delete = (lfs_file for lfs_file in lfs_files if lfs_file.filename.startswith("checkpoints/"))

# Permanently delete LFS files
>>> api.permanently_delete_lfs_files("username/my-cool-repo", lfs_files_to_delete)
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>preupload_lfs_files</name><anchor>huggingface_hub.HfApi.preupload_lfs_files</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L4247</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "additions", "val": ": Iterable[CommitOperationAdd]"}, {"name": "token", "val": ": Union[str, bool, None] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "create_pr", "val": ": Optional[bool] = None"}, {"name": "num_threads", "val": ": int = 5"}, {"name": "free_memory", "val": ": bool = True"}, {"name": "gitignore_content", "val": ": Optional[str] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  The repository in which you will commit the files, for example: `"username/custom_transformers"`.

- **operations** (`Iterable` of [CommitOperationAdd](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.CommitOperationAdd)) --
  The list of files to upload. Warning: the objects in this list will be mutated to include information
  relative to the upload. Do not reuse the same objects for multiple commits.

- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.

- **repo_type** (`str`, *optional*) --
  The type of repository to upload to (e.g. `"model"` -default-, `"dataset"` or `"space"`).

- **revision** (`str`, *optional*) --
  The git revision to commit from. Defaults to the head of the `"main"` branch.

- **create_pr** (`boolean`, *optional*) --
  Whether or not you plan to create a Pull Request with that commit. Defaults to `False`.

- **num_threads** (`int`, *optional*) --
  Number of concurrent threads for uploading files. Defaults to 5.
  Setting it to 2 means at most 2 files will be uploaded concurrently.

- **gitignore_content** (`str`, *optional*) --
  The content of the `.gitignore` file to know which files should be ignored. The order of priority
  is to first check if `gitignore_content` is passed, then check if the `.gitignore` file is present
  in the list of files to commit and finally default to the `.gitignore` file already hosted on the Hub
  (if any).</paramsdesc><paramgroups>0</paramgroups></docstring>
Pre-upload LFS files to S3 in preparation on a future commit.

This method is useful if you are generating the files to upload on-the-fly and you don't want to store them
in memory before uploading them all at once.

> [!WARNING]
> This is a power-user method. You shouldn't need to call it directly to make a normal commit.
> Use [create_commit()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.create_commit) directly instead.

> [!WARNING]
> Commit operations will be mutated during the process. In particular, the attached `path_or_fileobj` will be
> removed after the upload to save memory (and replaced by an empty `bytes` object). Do not reuse the same
> objects except to pass them to [create_commit()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.create_commit). If you don't want to remove the attached content from the
> commit operation object, pass `free_memory=False`.



<ExampleCodeBlock anchor="huggingface_hub.HfApi.preupload_lfs_files.example">

Example:
```py
>>> from huggingface_hub import CommitOperationAdd, preupload_lfs_files, create_commit, create_repo

>>> repo_id = create_repo("test_preupload").repo_id

# Generate and preupload LFS files one by one
>>> operations = [] # List of all `CommitOperationAdd` objects that will be generated
>>> for i in range(5):
...     content = ... # generate binary content
...     addition = CommitOperationAdd(path_in_repo=f"shard_{i}_of_5.bin", path_or_fileobj=content)
...     preupload_lfs_files(repo_id, additions=[addition]) # upload + free memory
...     operations.append(addition)

# Create commit
>>> create_commit(repo_id, operations=operations, commit_message="Commit all shards")
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>reject_access_request</name><anchor>huggingface_hub.HfApi.reject_access_request</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L8792</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "user", "val": ": str"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "rejection_reason", "val": ": Optional[str]"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  The id of the repo to reject access request for.
- **user** (`str`) --
  The username of the user which access request should be rejected.
- **repo_type** (`str`, *optional*) --
  The type of the repo to reject access request for. Must be one of `model`, `dataset` or `space`.
  Defaults to `model`.
- **rejection_reason** (`str`, *optional*) --
  Optional rejection reason that will be visible to the user (max 200 characters).
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><raises>- `HfHubHTTPError` -- 
  HTTP 400 if the repo is not gated.
- `HfHubHTTPError` -- 
  HTTP 403 if you only have read-only access to the repo. This can be the case if you don't have `write`
  or `admin` role in the organization the repo belongs to or if you passed a `read` token.
- `HfHubHTTPError` -- 
  HTTP 404 if the user does not exist on the Hub.
- `HfHubHTTPError` -- 
  HTTP 404 if the user access request cannot be found.
- `HfHubHTTPError` -- 
  HTTP 404 if the user access request is already in the rejected list.</raises><raisederrors>`HfHubHTTPError`</raisederrors></docstring>

Reject an access request from a user for a given gated repo.

A rejected request will go to the rejected list. The user cannot download any file of the repo. Rejected
requests can be accepted or cancelled at any time using [accept_access_request()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.accept_access_request) and [cancel_access_request()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.cancel_access_request).
A cancelled request will go back to the pending list while an accepted request will go to the accepted list.

For more info about gated repos, see https://huggingface.co/docs/hub/models-gated.








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>rename_discussion</name><anchor>huggingface_hub.HfApi.rename_discussion</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L6458</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "discussion_num", "val": ": int"}, {"name": "new_title", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated
  by a `/`.
- **discussion_num** (`int`) --
  The number of the Discussion or Pull Request . Must be a strictly positive integer.
- **new_title** (`str`) --
  The new title for the discussion
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if uploading to a dataset or
  space, `None` or `"model"` if uploading to a model. Default is
  `None`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[DiscussionTitleChange](/docs/huggingface_hub/main/en/package_reference/community#huggingface_hub.DiscussionTitleChange)</rettype><retdesc>the title change event</retdesc></docstring>
Renames a Discussion.







<ExampleCodeBlock anchor="huggingface_hub.HfApi.rename_discussion.example">

Examples:
```python
>>> new_title = "New title, fixing a typo"
>>> HfApi().rename_discussion(
...     repo_id="username/repo_name",
...     discussion_num=34
...     new_title=new_title
... )
# DiscussionTitleChange(id='deadbeef0000000', type='title-change', ...)

```

</ExampleCodeBlock>

> [!TIP]
> Raises the following errors:
>
>     - [`HTTPError`](https://requests.readthedocs.io/en/latest/api/#requests.HTTPError)
>       if the HuggingFace API returned an error
>     - [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError)
>       if some parameter value is invalid
>     - [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError)
>       If the repository to download from cannot be found. This may be because it doesn't exist,
>       or because it is set to `private` and you do not have access.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>repo_exists</name><anchor>huggingface_hub.HfApi.repo_exists</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L2767</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[str, bool, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated
  by a `/`.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if getting repository info from a dataset or a space,
  `None` or `"model"` if getting repository info from a model. Default is `None`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><retdesc>True if the repository exists, False otherwise.</retdesc></docstring>

Checks if a repository exists on the Hugging Face Hub.





<ExampleCodeBlock anchor="huggingface_hub.HfApi.repo_exists.example">

Examples:
```py
>>> from huggingface_hub import repo_exists
>>> repo_exists("google/gemma-7b")
True
>>> repo_exists("google/not-a-repo")
False
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>repo_info</name><anchor>huggingface_hub.HfApi.repo_info</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L2696</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "timeout", "val": ": Optional[float] = None"}, {"name": "files_metadata", "val": ": bool = False"}, {"name": "expand", "val": ": Optional[Union[ExpandModelProperty_T, ExpandDatasetProperty_T, ExpandSpaceProperty_T]] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated
  by a `/`.
- **revision** (`str`, *optional*) --
  The revision of the repository from which to get the
  information.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if getting repository info from a dataset or a space,
  `None` or `"model"` if getting repository info from a model. Default is `None`.
- **timeout** (`float`, *optional*) --
  Whether to set a timeout for the request to the Hub.
- **expand** (`ExpandModelProperty_T` or `ExpandDatasetProperty_T` or `ExpandSpaceProperty_T`, *optional*) --
  List properties to return in the response. When used, only the properties in the list will be returned.
  This parameter cannot be used if `files_metadata` is passed.
  For an exhaustive list of available properties, check out [model_info()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.model_info), [dataset_info()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.dataset_info) or [space_info()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.space_info).
- **files_metadata** (`bool`, *optional*) --
  Whether or not to retrieve metadata for files in the repository
  (size, LFS metadata, etc). Defaults to `False`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`Union[SpaceInfo, DatasetInfo, ModelInfo]`</rettype><retdesc>The repository information, as a
[huggingface_hub.hf_api.DatasetInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.DatasetInfo), [huggingface_hub.hf_api.ModelInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.ModelInfo)
or [huggingface_hub.hf_api.SpaceInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.SpaceInfo) object.</retdesc></docstring>

Get the info object for a given repo of a given type.







> [!TIP]
> Raises the following errors:
>
>     - [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError)
>       If the repository to download from cannot be found. This may be because it doesn't exist,
>       or because it is set to `private` and you do not have access.
>     - [RevisionNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RevisionNotFoundError)
>       If the revision to download from cannot be found.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>request_space_hardware</name><anchor>huggingface_hub.HfApi.request_space_hardware</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L6950</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "hardware", "val": ": SpaceHardware"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "sleep_time", "val": ": Optional[int] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  ID of the repo to update. Example: `"bigcode/in-the-stack"`.
- **hardware** (`str` or [SpaceHardware](/docs/huggingface_hub/main/en/package_reference/space_runtime#huggingface_hub.SpaceHardware)) --
  Hardware on which to run the Space. Example: `"t4-medium"`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.
- **sleep_time** (`int`, *optional*) --
  Number of seconds of inactivity to wait before a Space is put to sleep. Set to `-1` if you don't want
  your Space to sleep (default behavior for upgraded hardware). For free hardware, you can't configure
  the sleep time (value is fixed to 48 hours of inactivity).
  See https://huggingface.co/docs/hub/spaces-gpus#sleep-time for more details.</paramsdesc><paramgroups>0</paramgroups><rettype>[SpaceRuntime](/docs/huggingface_hub/main/en/package_reference/space_runtime#huggingface_hub.SpaceRuntime)</rettype><retdesc>Runtime information about a Space including Space stage and hardware.</retdesc></docstring>
Request new hardware for a Space.







> [!TIP]
> It is also possible to request hardware directly when creating the Space repo! See [create_repo()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.create_repo) for details.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>request_space_storage</name><anchor>huggingface_hub.HfApi.request_space_storage</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L7251</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "storage", "val": ": SpaceStorage"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  ID of the Space to update. Example: `"open-llm-leaderboard/open_llm_leaderboard"`.
- **storage** (`str` or [SpaceStorage](/docs/huggingface_hub/main/en/package_reference/space_runtime#huggingface_hub.SpaceStorage)) --
  Storage tier. Either 'small', 'medium', or 'large'.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[SpaceRuntime](/docs/huggingface_hub/main/en/package_reference/space_runtime#huggingface_hub.SpaceRuntime)</rettype><retdesc>Runtime information about a Space including Space stage and hardware.</retdesc></docstring>
Request persistent storage for a Space.







> [!TIP]
> It is not possible to decrease persistent storage after its granted. To do so, you must delete it
> via [delete_space_storage()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.delete_space_storage).


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>restart_space</name><anchor>huggingface_hub.HfApi.restart_space</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L7087</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "factory_reboot", "val": ": bool = False"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  ID of the Space to restart. Example: `"Salesforce/BLIP2"`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.
- **factory_reboot** (`bool`, *optional*) --
  If `True`, the Space will be rebuilt from scratch without caching any requirements.</paramsdesc><paramgroups>0</paramgroups><rettype>[SpaceRuntime](/docs/huggingface_hub/main/en/package_reference/space_runtime#huggingface_hub.SpaceRuntime)</rettype><retdesc>Runtime information about your Space.</retdesc><raises>- [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) -- 
  If your Space is not found (error 404). Most probably wrong repo_id or your space is private but you
  are not authenticated.
- [HfHubHTTPError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.HfHubHTTPError) -- 
  403 Forbidden: only the owner of a Space can restart it. If you want to restart a Space that you don't
  own, either ask the owner by opening a Discussion or duplicate the Space.
- [BadRequestError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.BadRequestError) -- 
  If your Space is a static Space. Static Spaces are always running and never billed. If you want to hide
  a static Space, you can set it to private.</raises><raisederrors>[RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) or [HfHubHTTPError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.HfHubHTTPError) or [BadRequestError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.BadRequestError)</raisederrors></docstring>
Restart your Space.

This is the only way to programmatically restart a Space if you've put it on Pause (see [pause_space()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.pause_space)). You
must be the owner of the Space to restart it. If you are using an upgraded hardware, your account will be
billed as soon as the Space is restarted. You can trigger a restart no matter the current state of a Space.

For more details, please visit [the docs](https://huggingface.co/docs/hub/spaces-gpus#pause).












</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>resume_inference_endpoint</name><anchor>huggingface_hub.HfApi.resume_inference_endpoint</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L7938</source><parameters>[{"name": "name", "val": ": str"}, {"name": "namespace", "val": ": Optional[str] = None"}, {"name": "running_ok", "val": ": bool = True"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **name** (`str`) --
  The name of the Inference Endpoint to resume.
- **namespace** (`str`, *optional*) --
  The namespace in which the Inference Endpoint is located. Defaults to the current user.
- **running_ok** (`bool`, *optional*) --
  If `True`, the method will not raise an error if the Inference Endpoint is already running. Defaults to
  `True`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[InferenceEndpoint](/docs/huggingface_hub/main/en/package_reference/inference_endpoints#huggingface_hub.InferenceEndpoint)</rettype><retdesc>information about the resumed Inference Endpoint.</retdesc></docstring>
Resume an Inference Endpoint.

For convenience, you can also resume an Inference Endpoint using [InferenceEndpoint.resume()](/docs/huggingface_hub/main/en/package_reference/inference_endpoints#huggingface_hub.InferenceEndpoint.resume).








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>resume_scheduled_job</name><anchor>huggingface_hub.HfApi.resume_scheduled_job</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L10488</source><parameters>[{"name": "scheduled_job_id", "val": ": str"}, {"name": "namespace", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **scheduled_job_id** (`str`) --
  ID of the scheduled Job.

- **namespace** (`str`, *optional*) --
  The namespace where the scheduled Job is. Defaults to the current user's namespace.

- **token** `(Union[bool, str, None]`, *optional*) --
  A valid user access token. If not provided, the locally saved token will be used, which is the
  recommended authentication method. Set to `False` to disable authentication.
  Refer to: https://huggingface.co/docs/huggingface_hub/quick-start#authentication.</paramsdesc><paramgroups>0</paramgroups></docstring>

Resume (unpause) a scheduled compute Job on Hugging Face infrastructure.




</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>revision_exists</name><anchor>huggingface_hub.HfApi.revision_exists</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L2811</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "revision", "val": ": str"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[str, bool, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated
  by a `/`.
- **revision** (`str`) --
  The revision of the repository to check.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if getting repository info from a dataset or a space,
  `None` or `"model"` if getting repository info from a model. Default is `None`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><retdesc>True if the repository and the revision exists, False otherwise.</retdesc></docstring>

Checks if a specific revision exists on a repo on the Hugging Face Hub.





<ExampleCodeBlock anchor="huggingface_hub.HfApi.revision_exists.example">

Examples:
```py
>>> from huggingface_hub import revision_exists
>>> revision_exists("google/gemma-7b", "float16")
True
>>> revision_exists("google/gemma-7b", "not-a-revision")
False
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>run_as_future</name><anchor>huggingface_hub.HfApi.run_as_future</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L1697</source><parameters>[{"name": "fn", "val": ": Callable[..., R]"}, {"name": "*args", "val": ""}, {"name": "**kwargs", "val": ""}]</parameters><paramsdesc>- **fn** (`Callable`) --
  The method to run in the background.
- ***args,** **kwargs --
  Arguments with which the method will be called.</paramsdesc><paramgroups>0</paramgroups><rettype>`Future`</rettype><retdesc>a [Future](https://docs.python.org/3/library/concurrent.futures.html#future-objects) instance to
get the result of the task.</retdesc></docstring>

Run a method in the background and return a Future instance.

The main goal is to run methods without blocking the main thread (e.g. to push data during a training).
Background jobs are queued to preserve order but are not ran in parallel. If you need to speed-up your scripts
by parallelizing lots of call to the API, you must setup and use your own [ThreadPoolExecutor](https://docs.python.org/3/library/concurrent.futures.html#threadpoolexecutor).

Note: Most-used methods like [upload_file()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.upload_file), [upload_folder()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.upload_folder) and [create_commit()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.create_commit) have a `run_as_future: bool`
argument to directly call them in the background. This is equivalent to calling `api.run_as_future(...)` on them
but less verbose.







<ExampleCodeBlock anchor="huggingface_hub.HfApi.run_as_future.example">

Example:
```py
>>> from huggingface_hub import HfApi
>>> api = HfApi()
>>> future = api.run_as_future(api.whoami) # instant
>>> future.done()
False
>>> future.result() # wait until complete and return result
(...)
>>> future.done()
True
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>run_job</name><anchor>huggingface_hub.HfApi.run_job</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L9844</source><parameters>[{"name": "image", "val": ": str"}, {"name": "command", "val": ": list[str]"}, {"name": "env", "val": ": Optional[dict[str, Any]] = None"}, {"name": "secrets", "val": ": Optional[dict[str, Any]] = None"}, {"name": "flavor", "val": ": Optional[SpaceHardware] = None"}, {"name": "timeout", "val": ": Optional[Union[int, float, str]] = None"}, {"name": "namespace", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **image** (`str`) --
  The Docker image to use.
  Examples: `"ubuntu"`, `"python:3.12"`, `"pytorch/pytorch:2.6.0-cuda12.4-cudnn9-devel"`.
  Example with an image from a Space: `"hf.co/spaces/lhoestq/duckdb"`.

- **command** (`list[str]`) --
  The command to run. Example: `["echo", "hello"]`.

- **env** (`dict[str, Any]`, *optional*) --
  Defines the environment variables for the Job.

- **secrets** (`dict[str, Any]`, *optional*) --
  Defines the secret environment variables for the Job.

- **flavor** (`str`, *optional*) --
  Flavor for the hardware, as in Hugging Face Spaces. See [SpaceHardware](/docs/huggingface_hub/main/en/package_reference/space_runtime#huggingface_hub.SpaceHardware) for possible values.
  Defaults to `"cpu-basic"`.

- **timeout** (`Union[int, float, str]`, *optional*) --
  Max duration for the Job: int/float with s (seconds, default), m (minutes), h (hours) or d (days).
  Example: `300` or `"5m"` for 5 minutes.

- **namespace** (`str`, *optional*) --
  The namespace where the Job will be created. Defaults to the current user's namespace.

- **token** `(Union[bool, str, None]`, *optional*) --
  A valid user access token. If not provided, the locally saved token will be used, which is the
  recommended authentication method. Set to `False` to disable authentication.
  Refer to: https://huggingface.co/docs/huggingface_hub/quick-start#authentication.</paramsdesc><paramgroups>0</paramgroups></docstring>

Run compute Jobs on Hugging Face infrastructure.



Example:
<ExampleCodeBlock anchor="huggingface_hub.HfApi.run_job.example">

Run your first Job:

```python
>>> from huggingface_hub import run_job
>>> run_job(image="python:3.12", command=["python", "-c" ,"print('Hello from HF compute!')"])
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.HfApi.run_job.example-2">

Run a GPU Job:

```python
>>> from huggingface_hub import run_job
>>> image = "pytorch/pytorch:2.6.0-cuda12.4-cudnn9-devel"
>>> command = ["python", "-c", "import torch; print(f"This code ran with the following GPU: {torch.cuda.get_device_name()}")"]
>>> run_job(image=image, command=command, flavor="a10g-small")
```

</ExampleCodeBlock>



</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>run_uv_job</name><anchor>huggingface_hub.HfApi.run_uv_job</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L10130</source><parameters>[{"name": "script", "val": ": str"}, {"name": "script_args", "val": ": Optional[list[str]] = None"}, {"name": "dependencies", "val": ": Optional[list[str]] = None"}, {"name": "python", "val": ": Optional[str] = None"}, {"name": "image", "val": ": Optional[str] = None"}, {"name": "env", "val": ": Optional[dict[str, Any]] = None"}, {"name": "secrets", "val": ": Optional[dict[str, Any]] = None"}, {"name": "flavor", "val": ": Optional[SpaceHardware] = None"}, {"name": "timeout", "val": ": Optional[Union[int, float, str]] = None"}, {"name": "namespace", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "_repo", "val": ": Optional[str] = None"}]</parameters><paramsdesc>- **script** (`str`) --
  Path or URL of the UV script, or a command.

- **script_args** (`list[str]`, *optional*) --
  Arguments to pass to the script or command.

- **dependencies** (`list[str]`, *optional*) --
  Dependencies to use to run the UV script.

- **python** (`str`, *optional*) --
  Use a specific Python version. Default is 3.12.

- **image** (`str`, *optional*, defaults to "ghcr.io/astral-sh/uv --python3.12-bookworm"):
  Use a custom Docker image with `uv` installed.

- **env** (`dict[str, Any]`, *optional*) --
  Defines the environment variables for the Job.

- **secrets** (`dict[str, Any]`, *optional*) --
  Defines the secret environment variables for the Job.

- **flavor** (`str`, *optional*) --
  Flavor for the hardware, as in Hugging Face Spaces. See [SpaceHardware](/docs/huggingface_hub/main/en/package_reference/space_runtime#huggingface_hub.SpaceHardware) for possible values.
  Defaults to `"cpu-basic"`.

- **timeout** (`Union[int, float, str]`, *optional*) --
  Max duration for the Job: int/float with s (seconds, default), m (minutes), h (hours) or d (days).
  Example: `300` or `"5m"` for 5 minutes.

- **namespace** (`str`, *optional*) --
  The namespace where the Job will be created. Defaults to the current user's namespace.

- **token** `(Union[bool, str, None]`, *optional*) --
  A valid user access token. If not provided, the locally saved token will be used, which is the
  recommended authentication method. Set to `False` to disable authentication.
  Refer to: https://huggingface.co/docs/huggingface_hub/quick-start#authentication.</paramsdesc><paramgroups>0</paramgroups></docstring>

Run a UV script Job on Hugging Face infrastructure.



Example:

<ExampleCodeBlock anchor="huggingface_hub.HfApi.run_uv_job.example">

Run a script from a URL:

```python
>>> from huggingface_hub import run_uv_job
>>> script = "https://raw.githubusercontent.com/huggingface/trl/refs/heads/main/trl/scripts/sft.py"
>>> script_args = ["--model_name_or_path", "Qwen/Qwen2-0.5B", "--dataset_name", "trl-lib/Capybara", "--push_to_hub"]
>>> run_uv_job(script, script_args=script_args, dependencies=["trl"], flavor="a10g-small")
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.HfApi.run_uv_job.example-2">

Run a local script:

```python
>>> from huggingface_hub import run_uv_job
>>> script = "my_sft.py"
>>> script_args = ["--model_name_or_path", "Qwen/Qwen2-0.5B", "--dataset_name", "trl-lib/Capybara", "--push_to_hub"]
>>> run_uv_job(script, script_args=script_args, dependencies=["trl"], flavor="a10g-small")
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.HfApi.run_uv_job.example-3">

Run a command:

```python
>>> from huggingface_hub import run_uv_job
>>> script = "lighteval"
>>> script_args= ["endpoint", "inference-providers", "model_name=openai/gpt-oss-20b,provider=auto", "lighteval|gsm8k|0|0"]
>>> run_uv_job(script, script_args=script_args, dependencies=["lighteval"], flavor="a10g-small")
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>scale_to_zero_inference_endpoint</name><anchor>huggingface_hub.HfApi.scale_to_zero_inference_endpoint</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L7984</source><parameters>[{"name": "name", "val": ": str"}, {"name": "namespace", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **name** (`str`) --
  The name of the Inference Endpoint to scale to zero.
- **namespace** (`str`, *optional*) --
  The namespace in which the Inference Endpoint is located. Defaults to the current user.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[InferenceEndpoint](/docs/huggingface_hub/main/en/package_reference/inference_endpoints#huggingface_hub.InferenceEndpoint)</rettype><retdesc>information about the scaled-to-zero Inference Endpoint.</retdesc></docstring>
Scale Inference Endpoint to zero.

An Inference Endpoint scaled to zero will not be charged. It will be resume on the next request to it, with a
cold start delay. This is different than pausing the Inference Endpoint with [pause_inference_endpoint()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.pause_inference_endpoint), which
would require a manual resume with [resume_inference_endpoint()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.resume_inference_endpoint).

For convenience, you can also scale an Inference Endpoint to zero using [InferenceEndpoint.scale_to_zero()](/docs/huggingface_hub/main/en/package_reference/inference_endpoints#huggingface_hub.InferenceEndpoint.scale_to_zero).








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>set_space_sleep_time</name><anchor>huggingface_hub.HfApi.set_space_sleep_time</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L7000</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "sleep_time", "val": ": int"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  ID of the repo to update. Example: `"bigcode/in-the-stack"`.
- **sleep_time** (`int`, *optional*) --
  Number of seconds of inactivity to wait before a Space is put to sleep. Set to `-1` if you don't want
  your Space to pause (default behavior for upgraded hardware). For free hardware, you can't configure
  the sleep time (value is fixed to 48 hours of inactivity).
  See https://huggingface.co/docs/hub/spaces-gpus#sleep-time for more details.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[SpaceRuntime](/docs/huggingface_hub/main/en/package_reference/space_runtime#huggingface_hub.SpaceRuntime)</rettype><retdesc>Runtime information about a Space including Space stage and hardware.</retdesc></docstring>
Set a custom sleep time for a Space running on upgraded hardware..

Your Space will go to sleep after X seconds of inactivity. You are not billed when your Space is in "sleep"
mode. If a new visitor lands on your Space, it will "wake it up". Only upgraded hardware can have a
configurable sleep time. To know more about the sleep stage, please refer to
https://huggingface.co/docs/hub/spaces-gpus#sleep-time.







> [!TIP]
> It is also possible to set a custom sleep time when requesting hardware with [request_space_hardware()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.request_space_hardware).


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>snapshot_download</name><anchor>huggingface_hub.HfApi.snapshot_download</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L5374</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "cache_dir", "val": ": Union[str, Path, None] = None"}, {"name": "local_dir", "val": ": Union[str, Path, None] = None"}, {"name": "etag_timeout", "val": ": float = 10"}, {"name": "force_download", "val": ": bool = False"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "local_files_only", "val": ": bool = False"}, {"name": "allow_patterns", "val": ": Optional[Union[list[str], str]] = None"}, {"name": "ignore_patterns", "val": ": Optional[Union[list[str], str]] = None"}, {"name": "max_workers", "val": ": int = 8"}, {"name": "tqdm_class", "val": ": Optional[type[base_tqdm]] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A user or an organization name and a repo name separated by a `/`.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if downloading from a dataset or space,
  `None` or `"model"` if downloading from a model. Default is `None`.
- **revision** (`str`, *optional*) --
  An optional Git revision id which can be a branch name, a tag, or a
  commit hash.
- **cache_dir** (`str`, `Path`, *optional*) --
  Path to the folder where cached files are stored.
- **local_dir** (`str` or `Path`, *optional*) --
  If provided, the downloaded files will be placed under this directory.
- **etag_timeout** (`float`, *optional*, defaults to `10`) --
  When fetching ETag, how many seconds to wait for the server to send
  data before giving up which is passed to `httpx.request`.
- **force_download** (`bool`, *optional*, defaults to `False`) --
  Whether the file should be downloaded even if it already exists in the local cache.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.
- **local_files_only** (`bool`, *optional*, defaults to `False`) --
  If `True`, avoid downloading the file and return the path to the
  local cached file if it exists.
- **allow_patterns** (`list[str]` or `str`, *optional*) --
  If provided, only files matching at least one pattern are downloaded.
- **ignore_patterns** (`list[str]` or `str`, *optional*) --
  If provided, files matching any of the patterns are not downloaded.
- **max_workers** (`int`, *optional*) --
  Number of concurrent threads to download files (1 thread = 1 file download).
  Defaults to 8.
- **tqdm_class** (`tqdm`, *optional*) --
  If provided, overwrites the default behavior for the progress bar. Passed
  argument must inherit from `tqdm.auto.tqdm` or at least mimic its behavior.
  Note that the `tqdm_class` is not passed to each individual download.
  Defaults to the custom HF progress bar that can be disabled by setting
  `HF_HUB_DISABLE_PROGRESS_BARS` environment variable.</paramsdesc><paramgroups>0</paramgroups><rettype>`str`</rettype><retdesc>folder path of the repo snapshot.</retdesc><raises>- [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) -- 
  If the repository to download from cannot be found. This may be because it doesn't exist,
  or because it is set to `private` and you do not have access.
- [RevisionNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RevisionNotFoundError) -- 
  If the revision to download from cannot be found.
- [`EnvironmentError`](https://docs.python.org/3/library/exceptions.html#EnvironmentError) -- 
  If `token=True` and the token cannot be found.
- [`OSError`](https://docs.python.org/3/library/exceptions.html#OSError) -- if
  ETag cannot be determined.
- [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError) -- 
  if some parameter value is invalid.</raises><raisederrors>[RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) or [RevisionNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RevisionNotFoundError) or ``EnvironmentError`` or ``OSError`` or ``ValueError``</raisederrors></docstring>
Download repo files.

Download a whole snapshot of a repo's files at the specified revision. This is useful when you want all files from
a repo, because you don't know which ones you will need a priori. All files are nested inside a folder in order
to keep their actual filename relative to that folder. You can also filter which files to download using
`allow_patterns` and `ignore_patterns`.

If `local_dir` is provided, the file structure from the repo will be replicated in this location. When using this
option, the `cache_dir` will not be used and a `.cache/huggingface/` folder will be created at the root of `local_dir`
to store some metadata related to the downloaded files.While this mechanism is not as robust as the main
cache-system, it's optimized for regularly pulling the latest version of a repository.

An alternative would be to clone the repo but this requires git and git-lfs to be installed and properly
configured. It is also not possible to filter which files to download when cloning a repository using git.












</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>space_info</name><anchor>huggingface_hub.HfApi.space_info</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L2626</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "timeout", "val": ": Optional[float] = None"}, {"name": "files_metadata", "val": ": bool = False"}, {"name": "expand", "val": ": Optional[list[ExpandSpaceProperty_T]] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated
  by a `/`.
- **revision** (`str`, *optional*) --
  The revision of the space repository from which to get the
  information.
- **timeout** (`float`, *optional*) --
  Whether to set a timeout for the request to the Hub.
- **files_metadata** (`bool`, *optional*) --
  Whether or not to retrieve metadata for files in the repository
  (size, LFS metadata, etc). Defaults to `False`.
- **expand** (`list[ExpandSpaceProperty_T]`, *optional*) --
  List properties to return in the response. When used, only the properties in the list will be returned.
  This parameter cannot be used if `full` is passed.
  Possible values are `"author"`, `"cardData"`, `"createdAt"`, `"datasets"`, `"disabled"`, `"lastModified"`, `"likes"`, `"models"`, `"private"`, `"runtime"`, `"sdk"`, `"siblings"`, `"sha"`, `"subdomain"`, `"tags"`, `"trendingScore"`, `"usedStorage"`, and `"resourceGroup"`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[SpaceInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.SpaceInfo)</rettype><retdesc>The space repository information.</retdesc></docstring>

Get info on one specific Space on huggingface.co.

Space can be private if you pass an acceptable token.







> [!TIP]
> Raises the following errors:
>
>     - [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError)
>       If the repository to download from cannot be found. This may be because it doesn't exist,
>       or because it is set to `private` and you do not have access.
>     - [RevisionNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RevisionNotFoundError)
>       If the revision to download from cannot be found.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>super_squash_history</name><anchor>huggingface_hub.HfApi.super_squash_history</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L3393</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "branch", "val": ": Optional[str] = None"}, {"name": "commit_message", "val": ": Optional[str] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[str, bool, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated by a `/`.
- **branch** (`str`, *optional*) --
  The branch to squash. Defaults to the head of the `"main"` branch.
- **commit_message** (`str`, *optional*) --
  The commit message to use for the squashed commit.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if listing commits from a dataset or a Space, `None` or `"model"` if
  listing from a model. Default is `None`.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><raises>- [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) -- 
  If repository is not found (error 404): wrong repo_id/repo_type, private but not authenticated or repo
  does not exist.
- [RevisionNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RevisionNotFoundError) -- 
  If the branch to squash cannot be found.
- [BadRequestError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.BadRequestError) -- 
  If invalid reference for a branch. You cannot squash history on tags.</raises><raisederrors>[RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) or [RevisionNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RevisionNotFoundError) or [BadRequestError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.BadRequestError)</raisederrors></docstring>
Squash commit history on a branch for a repo on the Hub.

Squashing the repo history is useful when you know you'll make hundreds of commits and you don't want to
clutter the history. Squashing commits can only be performed from the head of a branch.

> [!WARNING]
> Once squashed, the commit history cannot be retrieved. This is a non-revertible operation.

> [!WARNING]
> Once the history of a branch has been squashed, it is not possible to merge it back into another branch since
> their history will have diverged.







<ExampleCodeBlock anchor="huggingface_hub.HfApi.super_squash_history.example">

Example:
```py
>>> from huggingface_hub import HfApi
>>> api = HfApi()

# Create repo
>>> repo_id = api.create_repo("test-squash").repo_id

# Make a lot of commits.
>>> api.upload_file(repo_id=repo_id, path_in_repo="file.txt", path_or_fileobj=b"content")
>>> api.upload_file(repo_id=repo_id, path_in_repo="lfs.bin", path_or_fileobj=b"content")
>>> api.upload_file(repo_id=repo_id, path_in_repo="file.txt", path_or_fileobj=b"another_content")

# Squash history
>>> api.super_squash_history(repo_id=repo_id)
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>suspend_scheduled_job</name><anchor>huggingface_hub.HfApi.suspend_scheduled_job</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L10459</source><parameters>[{"name": "scheduled_job_id", "val": ": str"}, {"name": "namespace", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **scheduled_job_id** (`str`) --
  ID of the scheduled Job.

- **namespace** (`str`, *optional*) --
  The namespace where the scheduled Job is. Defaults to the current user's namespace.

- **token** `(Union[bool, str, None]`, *optional*) --
  A valid user access token. If not provided, the locally saved token will be used, which is the
  recommended authentication method. Set to `False` to disable authentication.
  Refer to: https://huggingface.co/docs/huggingface_hub/quick-start#authentication.</paramsdesc><paramgroups>0</paramgroups></docstring>

Suspend (pause) a scheduled compute Job on Hugging Face infrastructure.




</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>unlike</name><anchor>huggingface_hub.HfApi.unlike</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L2315</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "token", "val": ": Union[bool, str, None] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  The repository to unlike. Example: `"user/my-cool-model"`.

- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.

- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if unliking a dataset or space, `None` or
  `"model"` if unliking a model. Default is `None`.</paramsdesc><paramgroups>0</paramgroups><raises>- [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) -- 
  If repository is not found (error 404): wrong repo_id/repo_type, private
  but not authenticated or repo does not exist.</raises><raisederrors>[RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError)</raisederrors></docstring>

Unlike a given repo on the Hub (e.g. remove from favorite list).

To prevent spam usage, it is not possible to `like` a repository from a script.

See also [list_liked_repos()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.list_liked_repos).







<ExampleCodeBlock anchor="huggingface_hub.HfApi.unlike.example">

Example:
```python
>>> from huggingface_hub import list_liked_repos, unlike
>>> "gpt2" in list_liked_repos().models # we assume you have already liked gpt2
True
>>> unlike("gpt2")
>>> "gpt2" in list_liked_repos().models
False
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>update_collection_item</name><anchor>huggingface_hub.HfApi.update_collection_item</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L8384</source><parameters>[{"name": "collection_slug", "val": ": str"}, {"name": "item_object_id", "val": ": str"}, {"name": "note", "val": ": Optional[str] = None"}, {"name": "position", "val": ": Optional[int] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **collection_slug** (`str`) --
  Slug of the collection to update. Example: `"TheBloke/recent-models-64f9a55bb3115b4f513ec026"`.
- **item_object_id** (`str`) --
  ID of the item in the collection. This is not the id of the item on the Hub (repo_id or paper id).
  It must be retrieved from a [CollectionItem](/docs/huggingface_hub/main/en/package_reference/collections#huggingface_hub.CollectionItem) object. Example: `collection.items[0].item_object_id`.
- **note** (`str`, *optional*) --
  A note to attach to the item in the collection. The maximum size for a note is 500 characters.
- **position** (`int`, *optional*) --
  New position of the item in the collection.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups></docstring>
Update an item in a collection.



<ExampleCodeBlock anchor="huggingface_hub.HfApi.update_collection_item.example">

Example:

```py
>>> from huggingface_hub import get_collection, update_collection_item

# Get collection first
>>> collection = get_collection("TheBloke/recent-models-64f9a55bb3115b4f513ec026")

# Update item based on its ID (add note + update position)
>>> update_collection_item(
...     collection_slug="TheBloke/recent-models-64f9a55bb3115b4f513ec026",
...     item_object_id=collection.items[-1].item_object_id,
...     note="Newly updated model!"
...     position=0,
... )
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>update_collection_metadata</name><anchor>huggingface_hub.HfApi.update_collection_metadata</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L8196</source><parameters>[{"name": "collection_slug", "val": ": str"}, {"name": "title", "val": ": Optional[str] = None"}, {"name": "description", "val": ": Optional[str] = None"}, {"name": "position", "val": ": Optional[int] = None"}, {"name": "private", "val": ": Optional[bool] = None"}, {"name": "theme", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **collection_slug** (`str`) --
  Slug of the collection to update. Example: `"TheBloke/recent-models-64f9a55bb3115b4f513ec026"`.
- **title** (`str`) --
  Title of the collection to update.
- **description** (`str`, *optional*) --
  Description of the collection to update.
- **position** (`int`, *optional*) --
  New position of the collection in the list of collections of the user.
- **private** (`bool`, *optional*) --
  Whether the collection should be private or not.
- **theme** (`str`, *optional*) --
  Theme of the collection on the Hub.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups></docstring>
Update metadata of a collection on the Hub.

All arguments are optional. Only provided metadata will be updated.



Returns: [Collection](/docs/huggingface_hub/main/en/package_reference/collections#huggingface_hub.Collection)

<ExampleCodeBlock anchor="huggingface_hub.HfApi.update_collection_metadata.example">

Example:

```py
>>> from huggingface_hub import update_collection_metadata
>>> collection = update_collection_metadata(
...     collection_slug="username/iccv-2023-64f9a55bb3115b4f513ec026",
...     title="ICCV Oct. 2023"
...     description="Portfolio of models, datasets, papers and demos I presented at ICCV Oct. 2023",
...     private=False,
...     theme="pink",
... )
>>> collection.slug
"username/iccv-oct-2023-64f9a55bb3115b4f513ec026"
# ^collection slug got updated but not the trailing ID
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>update_inference_endpoint</name><anchor>huggingface_hub.HfApi.update_inference_endpoint</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L7738</source><parameters>[{"name": "name", "val": ": str"}, {"name": "accelerator", "val": ": Optional[str] = None"}, {"name": "instance_size", "val": ": Optional[str] = None"}, {"name": "instance_type", "val": ": Optional[str] = None"}, {"name": "min_replica", "val": ": Optional[int] = None"}, {"name": "max_replica", "val": ": Optional[int] = None"}, {"name": "scale_to_zero_timeout", "val": ": Optional[int] = None"}, {"name": "repository", "val": ": Optional[str] = None"}, {"name": "framework", "val": ": Optional[str] = None"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "task", "val": ": Optional[str] = None"}, {"name": "custom_image", "val": ": Optional[dict] = None"}, {"name": "env", "val": ": Optional[dict[str, str]] = None"}, {"name": "secrets", "val": ": Optional[dict[str, str]] = None"}, {"name": "domain", "val": ": Optional[str] = None"}, {"name": "path", "val": ": Optional[str] = None"}, {"name": "cache_http_responses", "val": ": Optional[bool] = None"}, {"name": "tags", "val": ": Optional[list[str]] = None"}, {"name": "namespace", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **name** (`str`) --
  The name of the Inference Endpoint to update.

- **accelerator** (`str`, *optional*) --
  The hardware accelerator to be used for inference (e.g. `"cpu"`).
- **instance_size** (`str`, *optional*) --
  The size or type of the instance to be used for hosting the model (e.g. `"x4"`).
- **instance_type** (`str`, *optional*) --
  The cloud instance type where the Inference Endpoint will be deployed (e.g. `"intel-icl"`).
- **min_replica** (`int`, *optional*) --
  The minimum number of replicas (instances) to keep running for the Inference Endpoint.
- **max_replica** (`int`, *optional*) --
  The maximum number of replicas (instances) to scale to for the Inference Endpoint.
- **scale_to_zero_timeout** (`int`, *optional*) --
  The duration in minutes before an inactive endpoint is scaled to zero.

- **repository** (`str`, *optional*) --
  The name of the model repository associated with the Inference Endpoint (e.g. `"gpt2"`).
- **framework** (`str`, *optional*) --
  The machine learning framework used for the model (e.g. `"custom"`).
- **revision** (`str`, *optional*) --
  The specific model revision to deploy on the Inference Endpoint (e.g. `"6c0e6080953db56375760c0471a8c5f2929baf11"`).
- **task** (`str`, *optional*) --
  The task on which to deploy the model (e.g. `"text-classification"`).
- **custom_image** (`dict`, *optional*) --
  A custom Docker image to use for the Inference Endpoint. This is useful if you want to deploy an
  Inference Endpoint running on the `text-generation-inference` (TGI) framework (see examples).
- **env** (`dict[str, str]`, *optional*) --
  Non-secret environment variables to inject in the container environment
- **secrets** (`dict[str, str]`, *optional*) --
  Secret values to inject in the container environment.

- **domain** (`str`, *optional*) --
  The custom domain for the Inference Endpoint deployment, if setup the inference endpoint will be available at this domain (e.g. `"my-new-domain.cool-website.woof"`).
- **path** (`str`, *optional*) --
  The custom path to the deployed model, should start with a `/` (e.g. `"/models/google-bert/bert-base-uncased"`).

- **cache_http_responses** (`bool`, *optional*) --
  Whether to cache HTTP responses from the Inference Endpoint.
- **tags** (`list[str]`, *optional*) --
  A list of tags to associate with the Inference Endpoint.

- **namespace** (`str`, *optional*) --
  The namespace where the Inference Endpoint will be updated. Defaults to the current user's namespace.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[InferenceEndpoint](/docs/huggingface_hub/main/en/package_reference/inference_endpoints#huggingface_hub.InferenceEndpoint)</rettype><retdesc>information about the updated Inference Endpoint.</retdesc></docstring>
Update an Inference Endpoint.

This method allows the update of either the compute configuration, the deployed model, the route, or any combination.
All arguments are optional but at least one must be provided.

For convenience, you can also update an Inference Endpoint using [InferenceEndpoint.update()](/docs/huggingface_hub/main/en/package_reference/inference_endpoints#huggingface_hub.InferenceEndpoint.update).








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>update_repo_settings</name><anchor>huggingface_hub.HfApi.update_repo_settings</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L3789</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "gated", "val": ": Optional[Literal['auto', 'manual', False]] = None"}, {"name": "private", "val": ": Optional[bool] = None"}, {"name": "token", "val": ": Union[str, bool, None] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated by a /.
- **gated** (`Literal["auto", "manual", False]`, *optional*) --
  The gated status for the repository. If set to `None` (default), the `gated` setting of the repository won't be updated.
  * "auto": The repository is gated, and access requests are automatically approved or denied based on predefined criteria.
  * "manual": The repository is gated, and access requests require manual approval.
  * False : The repository is not gated, and anyone can access it.
- **private** (`bool`, *optional*) --
  Whether the repository should be private.
- **token** (`Union[str, bool, None]`, *optional*) --
  A valid user access token (string). Defaults to the locally saved token,
  which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass False.
- **repo_type** (`str`, *optional*) --
  The type of the repository to update settings from (`"model"`, `"dataset"` or `"space"`).
  Defaults to `"model"`.</paramsdesc><paramgroups>0</paramgroups><raises>- [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError) -- 
  If gated is not one of "auto", "manual", or False.
- [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError) -- 
  If repo_type is not one of the values in constants.REPO_TYPES.
- [HfHubHTTPError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.HfHubHTTPError) -- 
  If the request to the Hugging Face Hub API fails.
- [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) -- 
  If the repository to download from cannot be found. This may be because it doesn't exist,
  or because it is set to `private` and you do not have access.</raises><raisederrors>``ValueError`` or [HfHubHTTPError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.HfHubHTTPError) or [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError)</raisederrors></docstring>

Update the settings of a repository, including gated access and visibility.

To give more control over how repos are used, the Hub allows repo authors to enable
access requests for their repos, and also to set the visibility of the repo to private.








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>update_webhook</name><anchor>huggingface_hub.HfApi.update_webhook</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L9162</source><parameters>[{"name": "webhook_id", "val": ": str"}, {"name": "url", "val": ": Optional[str] = None"}, {"name": "watched", "val": ": Optional[list[Union[dict, WebhookWatchedItem]]] = None"}, {"name": "domains", "val": ": Optional[list[constants.WEBHOOK_DOMAIN_T]] = None"}, {"name": "secret", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **webhook_id** (`str`) --
  The unique identifier of the webhook to be updated.
- **url** (`str`, optional) --
  The URL to which the payload will be sent.
- **watched** (`list[WebhookWatchedItem]`, optional) --
  List of items to watch. It can be users, orgs, models, datasets, or spaces.
  Refer to [WebhookWatchedItem](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.WebhookWatchedItem) for more details. Watched items can also be provided as plain dictionaries.
- **domains** (`list[Literal["repo", "discussion"]]`, optional) --
  The domains to watch. This can include "repo", "discussion", or both.
- **secret** (`str`, optional) --
  A secret to sign the payload with, providing an additional layer of security.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved token, which is the recommended
  method for authentication (see https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[WebhookInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.WebhookInfo)</rettype><retdesc>Info about the updated webhook.</retdesc></docstring>
Update an existing webhook.







<ExampleCodeBlock anchor="huggingface_hub.HfApi.update_webhook.example">

Example:
```python
>>> from huggingface_hub import update_webhook
>>> updated_payload = update_webhook(
...     webhook_id="654bbbc16f2ec14d77f109cc",
...     url="https://new.webhook.site/a2176e82-5720-43ee-9e06-f91cb4c91548",
...     watched=[{"type": "user", "name": "julien-c"}, {"type": "org", "name": "HuggingFaceH4"}],
...     domains=["repo"],
...     secret="my-secret",
... )
>>> print(updated_payload)
WebhookInfo(
    id="654bbbc16f2ec14d77f109cc",
    job=None,
    url="https://new.webhook.site/a2176e82-5720-43ee-9e06-f91cb4c91548",
    watched=[WebhookWatchedItem(type="user", name="julien-c"), WebhookWatchedItem(type="org", name="HuggingFaceH4")],
    domains=["repo"],
    secret="my-secret",
    disabled=False,
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>upload_file</name><anchor>huggingface_hub.HfApi.upload_file</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L4435</source><parameters>[{"name": "path_or_fileobj", "val": ": Union[str, Path, bytes, BinaryIO]"}, {"name": "path_in_repo", "val": ": str"}, {"name": "repo_id", "val": ": str"}, {"name": "token", "val": ": Union[str, bool, None] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "commit_message", "val": ": Optional[str] = None"}, {"name": "commit_description", "val": ": Optional[str] = None"}, {"name": "create_pr", "val": ": Optional[bool] = None"}, {"name": "parent_commit", "val": ": Optional[str] = None"}, {"name": "run_as_future", "val": ": bool = False"}]</parameters><paramsdesc>- **path_or_fileobj** (`str`, `Path`, `bytes`, or `IO`) --
  Path to a file on the local machine or binary data stream /
  fileobj / buffer.
- **path_in_repo** (`str`) --
  Relative filepath in the repo, for example:
  `"checkpoints/1fec34a/weights.bin"`
- **repo_id** (`str`) --
  The repository to which the file will be uploaded, for example:
  `"username/custom_transformers"`
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if uploading to a dataset or
  space, `None` or `"model"` if uploading to a model. Default is
  `None`.
- **revision** (`str`, *optional*) --
  The git revision to commit from. Defaults to the head of the `"main"` branch.
- **commit_message** (`str`, *optional*) --
  The summary / title / first line of the generated commit
- **commit_description** (`str` *optional*) --
  The description of the generated commit
- **create_pr** (`boolean`, *optional*) --
  Whether or not to create a Pull Request with that commit. Defaults to `False`.
  If `revision` is not set, PR is opened against the `"main"` branch. If
  `revision` is set and is a branch, PR is opened against this branch. If
  `revision` is set and is not a branch name (example: a commit oid), an
  `RevisionNotFoundError` is returned by the server.
- **parent_commit** (`str`, *optional*) --
  The OID / SHA of the parent commit, as a hexadecimal string. Shorthands (7 first characters) are also supported.
  If specified and `create_pr` is `False`, the commit will fail if `revision` does not point to `parent_commit`.
  If specified and `create_pr` is `True`, the pull request will be created from `parent_commit`.
  Specifying `parent_commit` ensures the repo has not changed before committing the changes, and can be
  especially useful if the repo is updated / committed to concurrently.
- **run_as_future** (`bool`, *optional*) --
  Whether or not to run this method in the background. Background jobs are run sequentially without
  blocking the main thread. Passing `run_as_future=True` will return a [Future](https://docs.python.org/3/library/concurrent.futures.html#future-objects)
  object. Defaults to `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[CommitInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.CommitInfo) or `Future`</rettype><retdesc>Instance of [CommitInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.CommitInfo) containing information about the newly created commit (commit hash, commit
url, pr url, commit message,...). If `run_as_future=True` is passed, returns a Future object which will
contain the result when executed.</retdesc></docstring>

Upload a local file (up to 50 GB) to the given repo. The upload is done
through a HTTP post request, and doesn't require git or git-lfs to be
installed.







> [!TIP]
> Raises the following errors:
>
>     - [`HTTPError`](https://requests.readthedocs.io/en/latest/api/#requests.HTTPError)
>       if the HuggingFace API returned an error
>     - [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError)
>       if some parameter value is invalid
>     - [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError)
>       If the repository to download from cannot be found. This may be because it doesn't exist,
>       or because it is set to `private` and you do not have access.
>     - [RevisionNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RevisionNotFoundError)
>       If the revision to download from cannot be found.

> [!WARNING]
> `upload_file` assumes that the repo already exists on the Hub. If you get a
> Client error 404, please make sure you are authenticated and that `repo_id` and
> `repo_type` are set correctly. If repo does not exist, create it first using
> [create_repo()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.create_repo).

<ExampleCodeBlock anchor="huggingface_hub.HfApi.upload_file.example">

Example:

```python
>>> from huggingface_hub import upload_file

>>> with open("./local/filepath", "rb") as fobj:
...     upload_file(
...         path_or_fileobj=fileobj,
...         path_in_repo="remote/file/path.h5",
...         repo_id="username/my-dataset",
...         repo_type="dataset",
...         token="my_token",
...     )

>>> upload_file(
...     path_or_fileobj=".\\local\\file\\path",
...     path_in_repo="remote/file/path.h5",
...     repo_id="username/my-model",
...     token="my_token",
... )

>>> upload_file(
...     path_or_fileobj=".\\local\\file\\path",
...     path_in_repo="remote/file/path.h5",
...     repo_id="username/my-model",
...     token="my_token",
...     create_pr=True,
... )
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>upload_folder</name><anchor>huggingface_hub.HfApi.upload_folder</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L4617</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "folder_path", "val": ": Union[str, Path]"}, {"name": "path_in_repo", "val": ": Optional[str] = None"}, {"name": "commit_message", "val": ": Optional[str] = None"}, {"name": "commit_description", "val": ": Optional[str] = None"}, {"name": "token", "val": ": Union[str, bool, None] = None"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "create_pr", "val": ": Optional[bool] = None"}, {"name": "parent_commit", "val": ": Optional[str] = None"}, {"name": "allow_patterns", "val": ": Optional[Union[list[str], str]] = None"}, {"name": "ignore_patterns", "val": ": Optional[Union[list[str], str]] = None"}, {"name": "delete_patterns", "val": ": Optional[Union[list[str], str]] = None"}, {"name": "run_as_future", "val": ": bool = False"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  The repository to which the file will be uploaded, for example:
  `"username/custom_transformers"`
- **folder_path** (`str` or `Path`) --
  Path to the folder to upload on the local file system
- **path_in_repo** (`str`, *optional*) --
  Relative path of the directory in the repo, for example:
  `"checkpoints/1fec34a/results"`. Will default to the root folder of the repository.
- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.
- **repo_type** (`str`, *optional*) --
  Set to `"dataset"` or `"space"` if uploading to a dataset or
  space, `None` or `"model"` if uploading to a model. Default is
  `None`.
- **revision** (`str`, *optional*) --
  The git revision to commit from. Defaults to the head of the `"main"` branch.
- **commit_message** (`str`, *optional*) --
  The summary / title / first line of the generated commit. Defaults to:
  `f"Upload {path_in_repo} with huggingface_hub"`
- **commit_description** (`str` *optional*) --
  The description of the generated commit
- **create_pr** (`boolean`, *optional*) --
  Whether or not to create a Pull Request with that commit. Defaults to `False`. If `revision` is not
  set, PR is opened against the `"main"` branch. If `revision` is set and is a branch, PR is opened
  against this branch. If `revision` is set and is not a branch name (example: a commit oid), an
  `RevisionNotFoundError` is returned by the server.
- **parent_commit** (`str`, *optional*) --
  The OID / SHA of the parent commit, as a hexadecimal string. Shorthands (7 first characters) are also supported.
  If specified and `create_pr` is `False`, the commit will fail if `revision` does not point to `parent_commit`.
  If specified and `create_pr` is `True`, the pull request will be created from `parent_commit`.
  Specifying `parent_commit` ensures the repo has not changed before committing the changes, and can be
  especially useful if the repo is updated / committed to concurrently.
- **allow_patterns** (`list[str]` or `str`, *optional*) --
  If provided, only files matching at least one pattern are uploaded.
- **ignore_patterns** (`list[str]` or `str`, *optional*) --
  If provided, files matching any of the patterns are not uploaded.
- **delete_patterns** (`list[str]` or `str`, *optional*) --
  If provided, remote files matching any of the patterns will be deleted from the repo while committing
  new files. This is useful if you don't know which files have already been uploaded.
  Note: to avoid discrepancies the `.gitattributes` file is not deleted even if it matches the pattern.
- **run_as_future** (`bool`, *optional*) --
  Whether or not to run this method in the background. Background jobs are run sequentially without
  blocking the main thread. Passing `run_as_future=True` will return a [Future](https://docs.python.org/3/library/concurrent.futures.html#future-objects)
  object. Defaults to `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>[CommitInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.CommitInfo) or `Future`</rettype><retdesc>Instance of [CommitInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.CommitInfo) containing information about the newly created commit (commit hash, commit
url, pr url, commit message,...). If `run_as_future=True` is passed, returns a Future object which will
contain the result when executed.</retdesc></docstring>

Upload a local folder to the given repo. The upload is done through a HTTP requests, and doesn't require git or
git-lfs to be installed.

The structure of the folder will be preserved. Files with the same name already present in the repository will
be overwritten. Others will be left untouched.

Use the `allow_patterns` and `ignore_patterns` arguments to specify which files to upload. These parameters
accept either a single pattern or a list of patterns. Patterns are Standard Wildcards (globbing patterns) as
documented [here](https://tldp.org/LDP/GNU-Linux-Tools-Summary/html/x11655.htm). If both `allow_patterns` and
`ignore_patterns` are provided, both constraints apply. By default, all files from the folder are uploaded.

Use the `delete_patterns` argument to specify remote files you want to delete. Input type is the same as for
`allow_patterns` (see above). If `path_in_repo` is also provided, the patterns are matched against paths
relative to this folder. For example, `upload_folder(..., path_in_repo="experiment", delete_patterns="logs/*")`
will delete any remote file under `./experiment/logs/`. Note that the `.gitattributes` file will not be deleted
even if it matches the patterns.

Any `.git/` folder present in any subdirectory will be ignored. However, please be aware that the `.gitignore`
file is not taken into account.

Uses `HfApi.create_commit` under the hood.







> [!TIP]
> Raises the following errors:
>
>     - [`HTTPError`](https://requests.readthedocs.io/en/latest/api/#requests.HTTPError)
>     if the HuggingFace API returned an error
>     - [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError)
>     if some parameter value is invalid

> [!WARNING]
> `upload_folder` assumes that the repo already exists on the Hub. If you get a Client error 404, please make
> sure you are authenticated and that `repo_id` and `repo_type` are set correctly. If repo does not exist, create
> it first using [create_repo()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.create_repo).

> [!TIP]
> When dealing with a large folder (thousands of files or hundreds of GB), we recommend using [upload_large_folder()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.upload_large_folder) instead.

<ExampleCodeBlock anchor="huggingface_hub.HfApi.upload_folder.example">

Example:

```python
# Upload checkpoints folder except the log files
>>> upload_folder(
...     folder_path="local/checkpoints",
...     path_in_repo="remote/experiment/checkpoints",
...     repo_id="username/my-dataset",
...     repo_type="datasets",
...     token="my_token",
...     ignore_patterns="**/logs/*.txt",
... )

# Upload checkpoints folder including logs while deleting existing logs from the repo
# Useful if you don't know exactly which log files have already being pushed
>>> upload_folder(
...     folder_path="local/checkpoints",
...     path_in_repo="remote/experiment/checkpoints",
...     repo_id="username/my-dataset",
...     repo_type="datasets",
...     token="my_token",
...     delete_patterns="**/logs/*.txt",
... )

# Upload checkpoints folder while creating a PR
>>> upload_folder(
...     folder_path="local/checkpoints",
...     path_in_repo="remote/experiment/checkpoints",
...     repo_id="username/my-dataset",
...     repo_type="datasets",
...     token="my_token",
...     create_pr=True,
... )
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>upload_large_folder</name><anchor>huggingface_hub.HfApi.upload_large_folder</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L5050</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "folder_path", "val": ": Union[str, Path]"}, {"name": "repo_type", "val": ": str"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "private", "val": ": Optional[bool] = None"}, {"name": "allow_patterns", "val": ": Optional[Union[list[str], str]] = None"}, {"name": "ignore_patterns", "val": ": Optional[Union[list[str], str]] = None"}, {"name": "num_workers", "val": ": Optional[int] = None"}, {"name": "print_report", "val": ": bool = True"}, {"name": "print_report_every", "val": ": int = 60"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  The repository to which the file will be uploaded.
  E.g. `"HuggingFaceTB/smollm-corpus"`.
- **folder_path** (`str` or `Path`) --
  Path to the folder to upload on the local file system.
- **repo_type** (`str`) --
  Type of the repository. Must be one of `"model"`, `"dataset"` or `"space"`.
  Unlike in all other `HfApi` methods, `repo_type` is explicitly required here. This is to avoid
  any mistake when uploading a large folder to the Hub, and therefore prevent from having to re-upload
  everything.
- **revision** (`str`, `optional`) --
  The branch to commit to. If not provided, the `main` branch will be used.
- **private** (`bool`, `optional`) --
  Whether the repository should be private.
  If `None` (default), the repo will be public unless the organization's default is private.
- **allow_patterns** (`list[str]` or `str`, *optional*) --
  If provided, only files matching at least one pattern are uploaded.
- **ignore_patterns** (`list[str]` or `str`, *optional*) --
  If provided, files matching any of the patterns are not uploaded.
- **num_workers** (`int`, *optional*) --
  Number of workers to start. Defaults to `os.cpu_count() - 2` (minimum 2).
  A higher number of workers may speed up the process if your machine allows it. However, on machines with a
  slower connection, it is recommended to keep the number of workers low to ensure better resumability.
  Indeed, partially uploaded files will have to be completely re-uploaded if the process is interrupted.
- **print_report** (`bool`, *optional*) --
  Whether to print a report of the upload progress. Defaults to True.
  Report is printed to `sys.stdout` every X seconds (60 by defaults) and overwrites the previous report.
- **print_report_every** (`int`, *optional*) --
  Frequency at which the report is printed. Defaults to 60 seconds.</paramsdesc><paramgroups>0</paramgroups></docstring>
Upload a large folder to the Hub in the most resilient way possible.

Several workers are started to upload files in an optimized way. Before being committed to a repo, files must be
hashed and be pre-uploaded if they are LFS files. Workers will perform these tasks for each file in the folder.
At each step, some metadata information about the upload process is saved in the folder under `.cache/.huggingface/`
to be able to resume the process if interrupted. The whole process might result in several commits.



> [!TIP]
> A few things to keep in mind:
>     - Repository limits still apply: https://huggingface.co/docs/hub/repositories-recommendations
>     - Do not start several processes in parallel.
>     - You can interrupt and resume the process at any time.
>     - Do not upload the same folder to several repositories. If you need to do so, you must delete the local `.cache/.huggingface/` folder first.

> [!WARNING]
> While being much more robust to upload large folders, `upload_large_folder` is more limited than [upload_folder()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.upload_folder) feature-wise. In practice:
>     - you cannot set a custom `path_in_repo`. If you want to upload to a subfolder, you need to set the proper structure locally.
>     - you cannot set a custom `commit_message` and `commit_description` since multiple commits are created.
>     - you cannot delete from the repo while uploading. Please make a separate commit first.
>     - you cannot create a PR directly. Please create a PR first (from the UI or using [create_pull_request()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.create_pull_request)) and then commit to it by passing `revision`.

**Technical details:**

`upload_large_folder` process is as follow:
1. (Check parameters and setup.)
2. Create repo if missing.
3. List local files to upload.
4. Run validation checks and display warnings if repository limits might be exceeded:
   - Warns if the total number of files exceeds 100k (recommended limit).
   - Warns if any folder contains more than 10k files (recommended limit).
   - Warns about files larger than 20GB (recommended) or 50GB (hard limit).
5. Start workers. Workers can perform the following tasks:
   - Hash a file.
   - Get upload mode (regular or LFS) for a list of files.
   - Pre-upload an LFS file.
   - Commit a bunch of files.
Once a worker finishes a task, it will move on to the next task based on the priority list (see below) until
all files are uploaded and committed.
6. While workers are up, regularly print a report to sys.stdout.

Order of priority:
1. Commit if more than 5 minutes since last commit attempt (and at least 1 file).
2. Commit if at least 150 files are ready to commit.
3. Get upload mode if at least 10 files have been hashed.
4. Pre-upload LFS file if at least 1 file and no worker is pre-uploading.
5. Hash file if at least 1 file and no worker is hashing.
6. Get upload mode if at least 1 file and no worker is getting upload mode.
7. Pre-upload LFS file if at least 1 file.
8. Hash file if at least 1 file to hash.
9. Get upload mode if at least 1 file to get upload mode.
10. Commit if at least 1 file to commit and at least 1 min since last commit attempt.
11. Commit if at least 1 file to commit and all other queues are empty.

Special rules:
- Only one worker can commit at a time.
- If no tasks are available, the worker waits for 10 seconds before checking again.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>verify_repo_checksums</name><anchor>huggingface_hub.HfApi.verify_repo_checksums</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L3085</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "repo_type", "val": ": Optional[str] = None"}, {"name": "revision", "val": ": Optional[str] = None"}, {"name": "local_dir", "val": ": Optional[Union[str, Path]] = None"}, {"name": "cache_dir", "val": ": Optional[Union[str, Path]] = None"}, {"name": "token", "val": ": Union[str, bool, None] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  A namespace (user or an organization) and a repo name separated by a `/`.
- **repo_type** (`str`, *optional*) --
  The type of the repository from which to get the tree (`"model"`, `"dataset"` or `"space"`.
  Defaults to `"model"`.
- **revision** (`str`, *optional*) --
  The revision of the repository from which to get the tree. Defaults to `"main"` branch.
- **local_dir** (`str` or `Path`, *optional*) --
  The local directory to verify.
- **cache_dir** (`str` or `Path`, *optional*) --
  The cache directory to verify.
- **token** (Union[bool, str, None], optional) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups><rettype>`FolderVerification`</rettype><retdesc>a structured result containing the verification details.</retdesc><raises>- [RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) -- 
  If repository is not found (error 404): wrong repo_id/repo_type, private but not authenticated or repo
  does not exist.
- [RevisionNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RevisionNotFoundError) -- 
  If revision is not found (error 404) on the repo.</raises><raisederrors>[RepositoryNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RepositoryNotFoundError) or [RevisionNotFoundError](/docs/huggingface_hub/main/en/package_reference/utilities#huggingface_hub.errors.RevisionNotFoundError)</raisederrors></docstring>

Verify local files for a repo against Hub checksums.












</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>whoami</name><anchor>huggingface_hub.HfApi.whoami</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L1737</source><parameters>[{"name": "token", "val": ": Union[bool, str, None] = None"}]</parameters><paramsdesc>- **token** (`bool` or `str`, *optional*) --
  A valid user access token (string). Defaults to the locally saved
  token, which is the recommended method for authentication (see
  https://huggingface.co/docs/huggingface_hub/quick-start#authentication).
  To disable authentication, pass `False`.</paramsdesc><paramgroups>0</paramgroups></docstring>

Call HF API to know "whoami".




</div></div>

## API Dataclasses

### AccessRequest[[huggingface_hub.hf_api.AccessRequest]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.hf_api.AccessRequest</name><anchor>huggingface_hub.hf_api.AccessRequest</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L420</source><parameters>[{"name": "username", "val": ": str"}, {"name": "fullname", "val": ": str"}, {"name": "email", "val": ": Optional[str]"}, {"name": "timestamp", "val": ": datetime"}, {"name": "status", "val": ": Literal['pending', 'accepted', 'rejected']"}, {"name": "fields", "val": ": Optional[dict[str, Any]] = None"}]</parameters><paramsdesc>- **username** (`str`) --
  Username of the user who requested access.
- **fullname** (`str`) --
  Fullname of the user who requested access.
- **email** (`Optional[str]`) --
  Email of the user who requested access.
  Can only be `None` in the /accepted list if the user was granted access manually.
- **timestamp** (`datetime`) --
  Timestamp of the request.
- **status** (`Literal["pending", "accepted", "rejected"]`) --
  Status of the request. Can be one of `["pending", "accepted", "rejected"]`.
- **fields** (`dict[str, Any]`, *optional*) --
  Additional fields filled by the user in the gate form.</paramsdesc><paramgroups>0</paramgroups></docstring>
Data structure containing information about a user access request.




</div>

### CommitInfo[[huggingface_hub.CommitInfo]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.CommitInfo</name><anchor>huggingface_hub.CommitInfo</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L349</source><parameters>[{"name": "*args", "val": ""}, {"name": "commit_url", "val": ": str"}, {"name": "**kwargs", "val": ""}]</parameters><paramsdesc>- **commit_url** (`str`) --
  Url where to find the commit.

- **commit_message** (`str`) --
  The summary (first line) of the commit that has been created.

- **commit_description** (`str`) --
  Description of the commit that has been created. Can be empty.

- **oid** (`str`) --
  Commit hash id. Example: `"91c54ad1727ee830252e457677f467be0bfd8a57"`.

- **pr_url** (`str`, *optional*) --
  Url to the PR that has been created, if any. Populated when `create_pr=True`
  is passed.

- **pr_revision** (`str`, *optional*) --
  Revision of the PR that has been created, if any. Populated when
  `create_pr=True` is passed. Example: `"refs/pr/1"`.

- **pr_num** (`int`, *optional*) --
  Number of the PR discussion that has been created, if any. Populated when
  `create_pr=True` is passed. Can be passed as `discussion_num` in
  [get_discussion_details()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.get_discussion_details). Example: `1`.

- **repo_url** (`RepoUrl`) --
  Repo URL of the commit containing info like repo_id, repo_type, etc.</paramsdesc><paramgroups>0</paramgroups></docstring>
Data structure containing information about a newly created commit.

Returned by any method that creates a commit on the Hub: [create_commit()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.create_commit), [upload_file()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.upload_file), [upload_folder()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.upload_folder),
[delete_file()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.delete_file), [delete_folder()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.delete_folder). It inherits from `str` for backward compatibility but using methods specific
to `str` is deprecated.




</div>

### DatasetInfo[[huggingface_hub.DatasetInfo]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.DatasetInfo</name><anchor>huggingface_hub.DatasetInfo</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L898</source><parameters>[{"name": "**kwargs", "val": ""}]</parameters><paramsdesc>- **id** (`str`) --
  ID of dataset.
- **author** (`str`) --
  Author of the dataset.
- **sha** (`str`) --
  Repo SHA at this particular revision.
- **created_at** (`datetime`, *optional*) --
  Date of creation of the repo on the Hub. Note that the lowest value is `2022-03-02T23:29:04.000Z`,
  corresponding to the date when we began to store creation dates.
- **last_modified** (`datetime`, *optional*) --
  Date of last commit to the repo.
- **private** (`bool`) --
  Is the repo private.
- **disabled** (`bool`, *optional*) --
  Is the repo disabled.
- **gated** (`Literal["auto", "manual", False]`, *optional*) --
  Is the repo gated.
  If so, whether there is manual or automatic approval.
- **downloads** (`int`) --
  Number of downloads of the dataset over the last 30 days.
- **downloads_all_time** (`int`) --
  Cumulated number of downloads of the model since its creation.
- **likes** (`int`) --
  Number of likes of the dataset.
- **tags** (`list[str]`) --
  List of tags of the dataset.
- **card_data** (`DatasetCardData`, *optional*) --
  Model Card Metadata  as a [huggingface_hub.repocard_data.DatasetCardData](/docs/huggingface_hub/main/en/package_reference/cards#huggingface_hub.DatasetCardData) object.
- **siblings** (`list[RepoSibling]`) --
  List of [huggingface_hub.hf_api.RepoSibling](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.hf_api.RepoSibling) objects that constitute the dataset.
- **paperswithcode_id** (`str`, *optional*) --
  Papers with code ID of the dataset.
- **trending_score** (`int`, *optional*) --
  Trending score of the dataset.</paramsdesc><paramgroups>0</paramgroups></docstring>

Contains information about a dataset on the Hub. This object is returned by [dataset_info()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.dataset_info) and [list_datasets()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.list_datasets).

> [!TIP]
> Most attributes of this class are optional. This is because the data returned by the Hub depends on the query made.
> In general, the more specific the query, the more information is returned. On the contrary, when listing datasets
> using [list_datasets()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.list_datasets) only a subset of the attributes are returned.




</div>

### DryRunFileInfo[[huggingface_hub.DryRunFileInfo]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.DryRunFileInfo</name><anchor>huggingface_hub.DryRunFileInfo</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/file_download.py#L154</source><parameters>[{"name": "commit_hash", "val": ": str"}, {"name": "file_size", "val": ": int"}, {"name": "filename", "val": ": str"}, {"name": "local_path", "val": ": str"}, {"name": "is_cached", "val": ": bool"}, {"name": "will_download", "val": ": bool"}]</parameters><paramsdesc>- **commit_hash** (`str`) --
  The commit_hash related to the file.
- **file_size** (`int`) --
  Size of the file. In case of an LFS file, contains the size of the actual LFS file, not the pointer.
- **filename** (`str`) --
  Name of the file in the repo.
- **is_cached** (`bool`) --
  Whether the file is already cached locally.
- **will_download** (`bool`) --
  Whether the file will be downloaded if `hf_hub_download` is called with `dry_run=False`.
  In practice, will_download is `True` if the file is not cached or if `force_download=True`.</paramsdesc><paramgroups>0</paramgroups></docstring>
Information returned when performing a dry run of a file download.

Returned by [hf_hub_download()](/docs/huggingface_hub/main/en/package_reference/file_download#huggingface_hub.hf_hub_download) when `dry_run=True`.




</div>

### GitRefInfo[[huggingface_hub.GitRefInfo]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.GitRefInfo</name><anchor>huggingface_hub.GitRefInfo</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L1246</source><parameters>[{"name": "name", "val": ": str"}, {"name": "ref", "val": ": str"}, {"name": "target_commit", "val": ": str"}]</parameters><paramsdesc>- **name** (`str`) --
  Name of the reference (e.g. tag name or branch name).
- **ref** (`str`) --
  Full git ref on the Hub (e.g. `"refs/heads/main"` or `"refs/tags/v1.0"`).
- **target_commit** (`str`) --
  OID of the target commit for the ref (e.g. `"e7da7f221d5bf496a48136c0cd264e630fe9fcc8"`)</paramsdesc><paramgroups>0</paramgroups></docstring>

Contains information about a git reference for a repo on the Hub.




</div>

### GitCommitInfo[[huggingface_hub.GitCommitInfo]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.GitCommitInfo</name><anchor>huggingface_hub.GitCommitInfo</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L1291</source><parameters>[{"name": "commit_id", "val": ": str"}, {"name": "authors", "val": ": list[str]"}, {"name": "created_at", "val": ": datetime"}, {"name": "title", "val": ": str"}, {"name": "message", "val": ": str"}, {"name": "formatted_title", "val": ": Optional[str]"}, {"name": "formatted_message", "val": ": Optional[str]"}]</parameters><paramsdesc>- **commit_id** (`str`) --
  OID of the commit (e.g. `"e7da7f221d5bf496a48136c0cd264e630fe9fcc8"`)
- **authors** (`list[str]`) --
  List of authors of the commit.
- **created_at** (`datetime`) --
  Datetime when the commit was created.
- **title** (`str`) --
  Title of the commit. This is a free-text value entered by the authors.
- **message** (`str`) --
  Description of the commit. This is a free-text value entered by the authors.
- **formatted_title** (`str`) --
  Title of the commit formatted as HTML. Only returned if `formatted=True` is set.
- **formatted_message** (`str`) --
  Description of the commit formatted as HTML. Only returned if `formatted=True` is set.</paramsdesc><paramgroups>0</paramgroups></docstring>

Contains information about a git commit for a repo on the Hub. Check out [list_repo_commits()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.list_repo_commits) for more details.




</div>

### GitRefs[[huggingface_hub.GitRefs]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.GitRefs</name><anchor>huggingface_hub.GitRefs</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L1265</source><parameters>[{"name": "branches", "val": ": list[GitRefInfo]"}, {"name": "converts", "val": ": list[GitRefInfo]"}, {"name": "tags", "val": ": list[GitRefInfo]"}, {"name": "pull_requests", "val": ": Optional[list[GitRefInfo]] = None"}]</parameters><paramsdesc>- **branches** (`list[GitRefInfo]`) --
  A list of [GitRefInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.GitRefInfo) containing information about branches on the repo.
- **converts** (`list[GitRefInfo]`) --
  A list of [GitRefInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.GitRefInfo) containing information about "convert" refs on the repo.
  Converts are refs used (internally) to push preprocessed data in Dataset repos.
- **tags** (`list[GitRefInfo]`) --
  A list of [GitRefInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.GitRefInfo) containing information about tags on the repo.
- **pull_requests** (`list[GitRefInfo]`, *optional*) --
  A list of [GitRefInfo](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.GitRefInfo) containing information about pull requests on the repo.
  Only returned if `include_prs=True` is set.</paramsdesc><paramgroups>0</paramgroups></docstring>

Contains information about all git references for a repo on the Hub.

Object is returned by [list_repo_refs()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.list_repo_refs).




</div>

### InferenceProviderMapping[[huggingface_hub.hf_api.InferenceProviderMapping]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.hf_api.InferenceProviderMapping</name><anchor>huggingface_hub.hf_api.InferenceProviderMapping</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L679</source><parameters>[{"name": "**kwargs", "val": ""}]</parameters></docstring>


</div>

### LFSFileInfo[[huggingface_hub.hf_api.LFSFileInfo]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.hf_api.LFSFileInfo</name><anchor>huggingface_hub.hf_api.LFSFileInfo</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L1556</source><parameters>[{"name": "**kwargs", "val": ""}]</parameters><paramsdesc>- **file_oid** (`str`) --
  SHA-256 object ID of the file. This is the identifier to pass when permanently deleting the file.
- **filename** (`str`) --
  Possible filename for the LFS object. See the note above for more information.
- **oid** (`str`) --
  OID of the LFS object.
- **pushed_at** (`datetime`) --
  Date the LFS object was pushed to the repo.
- **ref** (`str`, *optional*) --
  Ref where the LFS object has been pushed (if any).
- **size** (`int`) --
  Size of the LFS object.</paramsdesc><paramgroups>0</paramgroups></docstring>

Contains information about a file stored as LFS on a repo on the Hub.

Used in the context of listing and permanently deleting LFS files from a repo to free-up space.
See [list_lfs_files()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.list_lfs_files) and [permanently_delete_lfs_files()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.permanently_delete_lfs_files) for more details.

Git LFS files are tracked using SHA-256 object IDs, rather than file paths, to optimize performance
This approach is necessary because a single object can be referenced by multiple paths across different commits,
making it impractical to search and resolve these connections. Check out [our documentation](https://huggingface.co/docs/hub/storage-limits#advanced-track-lfs-file-references)
to learn how to know which filename(s) is(are) associated with each SHA.



<ExampleCodeBlock anchor="huggingface_hub.hf_api.LFSFileInfo.example">

Example:
```py
>>> from huggingface_hub import HfApi
>>> api = HfApi()
>>> lfs_files = api.list_lfs_files("username/my-cool-repo")

# Filter files files to delete based on a combination of `filename`, `pushed_at`, `ref` or `size`.
# e.g. select only LFS files in the "checkpoints" folder
>>> lfs_files_to_delete = (lfs_file for lfs_file in lfs_files if lfs_file.filename.startswith("checkpoints/"))

# Permanently delete LFS files
>>> api.permanently_delete_lfs_files("username/my-cool-repo", lfs_files_to_delete)
```

</ExampleCodeBlock>


</div>

### ModelInfo[[huggingface_hub.ModelInfo]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.ModelInfo</name><anchor>huggingface_hub.ModelInfo</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L704</source><parameters>[{"name": "**kwargs", "val": ""}]</parameters><paramsdesc>- **id** (`str`) --
  ID of model.
- **author** (`str`, *optional*) --
  Author of the model.
- **sha** (`str`, *optional*) --
  Repo SHA at this particular revision.
- **created_at** (`datetime`, *optional*) --
  Date of creation of the repo on the Hub. Note that the lowest value is `2022-03-02T23:29:04.000Z`,
  corresponding to the date when we began to store creation dates.
- **last_modified** (`datetime`, *optional*) --
  Date of last commit to the repo.
- **private** (`bool`) --
  Is the repo private.
- **disabled** (`bool`, *optional*) --
  Is the repo disabled.
- **downloads** (`int`) --
  Number of downloads of the model over the last 30 days.
- **downloads_all_time** (`int`) --
  Cumulated number of downloads of the model since its creation.
- **gated** (`Literal["auto", "manual", False]`, *optional*) --
  Is the repo gated.
  If so, whether there is manual or automatic approval.
- **gguf** (`dict`, *optional*) --
  GGUF information of the model.
- **inference** (`Literal["warm"]`, *optional*) --
  Status of the model on Inference Providers. Warm if the model is served by at least one provider.
- **inference_provider_mapping** (`list[InferenceProviderMapping]`, *optional*) --
  A list of `InferenceProviderMapping` ordered after the user's provider order.
- **likes** (`int`) --
  Number of likes of the model.
- **library_name** (`str`, *optional*) --
  Library associated with the model.
- **tags** (`list[str]`) --
  List of tags of the model. Compared to `card_data.tags`, contains extra tags computed by the Hub
  (e.g. supported libraries, model's arXiv).
- **pipeline_tag** (`str`, *optional*) --
  Pipeline tag associated with the model.
- **mask_token** (`str`, *optional*) --
  Mask token used by the model.
- **widget_data** (`Any`, *optional*) --
  Widget data associated with the model.
- **model_index** (`dict`, *optional*) --
  Model index for evaluation.
- **config** (`dict`, *optional*) --
  Model configuration.
- **transformers_info** (`TransformersInfo`, *optional*) --
  Transformers-specific info (auto class, processor, etc.) associated with the model.
- **trending_score** (`int`, *optional*) --
  Trending score of the model.
- **card_data** (`ModelCardData`, *optional*) --
  Model Card Metadata  as a [huggingface_hub.repocard_data.ModelCardData](/docs/huggingface_hub/main/en/package_reference/cards#huggingface_hub.ModelCardData) object.
- **siblings** (`list[RepoSibling]`) --
  List of [huggingface_hub.hf_api.RepoSibling](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.hf_api.RepoSibling) objects that constitute the model.
- **spaces** (`list[str]`, *optional*) --
  List of spaces using the model.
- **safetensors** (`SafeTensorsInfo`, *optional*) --
  Model's safetensors information.
- **security_repo_status** (`dict`, *optional*) --
  Model's security scan status.</paramsdesc><paramgroups>0</paramgroups></docstring>

Contains information about a model on the Hub. This object is returned by [model_info()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.model_info) and [list_models()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.list_models).

> [!TIP]
> Most attributes of this class are optional. This is because the data returned by the Hub depends on the query made.
> In general, the more specific the query, the more information is returned. On the contrary, when listing models
> using [list_models()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.list_models) only a subset of the attributes are returned.




</div>

### RepoSibling[[huggingface_hub.hf_api.RepoSibling]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.hf_api.RepoSibling</name><anchor>huggingface_hub.hf_api.RepoSibling</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L558</source><parameters>[{"name": "rfilename", "val": ": str"}, {"name": "size", "val": ": Optional[int] = None"}, {"name": "blob_id", "val": ": Optional[str] = None"}, {"name": "lfs", "val": ": Optional[BlobLfsInfo] = None"}]</parameters><paramsdesc>- **rfilename** (str) --
  file name, relative to the repo root.
- **size** (`int`, *optional*) --
  The file's size, in bytes. This attribute is defined when `files_metadata` argument of [repo_info()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.repo_info) is set
  to `True`. It's `None` otherwise.
- **blob_id** (`str`, *optional*) --
  The file's git OID. This attribute is defined when `files_metadata` argument of [repo_info()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.repo_info) is set to
  `True`. It's `None` otherwise.
- **lfs** (`BlobLfsInfo`, *optional*) --
  The file's LFS metadata. This attribute is defined when`files_metadata` argument of [repo_info()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.repo_info) is set to
  `True` and the file is stored with Git LFS. It's `None` otherwise.</paramsdesc><paramgroups>0</paramgroups></docstring>

Contains basic information about a repo file inside a repo on the Hub.

> [!TIP]
> All attributes of this class are optional except `rfilename`. This is because only the file names are returned when
> listing repositories on the Hub (with [list_models()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.list_models), [list_datasets()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.list_datasets) or [list_spaces()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.list_spaces)). If you need more
> information like file size, blob id or lfs details, you must request them specifically from one repo at a time
> (using [model_info()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.model_info), [dataset_info()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.dataset_info) or [space_info()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.space_info)) as it adds more constraints on the backend server to
> retrieve these.




</div>

### RepoFile[[huggingface_hub.hf_api.RepoFile]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.hf_api.RepoFile</name><anchor>huggingface_hub.hf_api.RepoFile</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L590</source><parameters>[{"name": "**kwargs", "val": ""}]</parameters><paramsdesc>- **path** (str) --
  file path relative to the repo root.
- **size** (`int`) --
  The file's size, in bytes.
- **blob_id** (`str`) --
  The file's git OID.
- **lfs** (`BlobLfsInfo`, *optional*) --
  The file's LFS metadata.
- **last_commit** (`LastCommitInfo`, *optional*) --
  The file's last commit metadata. Only defined if [list_repo_tree()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.list_repo_tree) and [get_paths_info()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.get_paths_info)
  are called with `expand=True`.
- **security** (`BlobSecurityInfo`, *optional*) --
  The file's security scan metadata. Only defined if [list_repo_tree()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.list_repo_tree) and [get_paths_info()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.get_paths_info)
  are called with `expand=True`.</paramsdesc><paramgroups>0</paramgroups></docstring>

Contains information about a file on the Hub.




</div>

### RepoUrl[[huggingface_hub.RepoUrl]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.RepoUrl</name><anchor>huggingface_hub.RepoUrl</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L496</source><parameters>[{"name": "url", "val": ": Any"}, {"name": "endpoint", "val": ": Optional[str] = None"}]</parameters><paramsdesc>- **url** (`Any`) --
  String value of the repo url.
- **endpoint** (`str`, *optional*) --
  Endpoint of the Hub. Defaults to <https://huggingface.co>.</paramsdesc><paramgroups>0</paramgroups><raises>- [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError) -- 
  If URL cannot be parsed.
- [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError) -- 
  If `repo_type` is unknown.</raises><raisederrors>``ValueError``</raisederrors></docstring>
Subclass of `str` describing a repo URL on the Hub.

`RepoUrl` is returned by `HfApi.create_repo`. It inherits from `str` for backward
compatibility. At initialization, the URL is parsed to populate properties:
- endpoint (`str`)
- namespace (`Optional[str]`)
- repo_name (`str`)
- repo_id (`str`)
- repo_type (`Literal["model", "dataset", "space"]`)
- url (`str`)



<ExampleCodeBlock anchor="huggingface_hub.RepoUrl.example">

Example:
```py
>>> RepoUrl('https://huggingface.co/gpt2')
RepoUrl('https://huggingface.co/gpt2', endpoint='https://huggingface.co', repo_type='model', repo_id='gpt2')

>>> RepoUrl('https://hub-ci.huggingface.co/datasets/dummy_user/dummy_dataset', endpoint='https://hub-ci.huggingface.co')
RepoUrl('https://hub-ci.huggingface.co/datasets/dummy_user/dummy_dataset', endpoint='https://hub-ci.huggingface.co', repo_type='dataset', repo_id='dummy_user/dummy_dataset')

>>> RepoUrl('hf://datasets/my-user/my-dataset')
RepoUrl('hf://datasets/my-user/my-dataset', endpoint='https://huggingface.co', repo_type='dataset', repo_id='user/dataset')

>>> HfApi.create_repo("dummy_model")
RepoUrl('https://huggingface.co/Wauplin/dummy_model', endpoint='https://huggingface.co', repo_type='model', repo_id='Wauplin/dummy_model')
```

</ExampleCodeBlock>






</div>

### SafetensorsRepoMetadata[[huggingface_hub.utils.SafetensorsRepoMetadata]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.utils.SafetensorsRepoMetadata</name><anchor>huggingface_hub.utils.SafetensorsRepoMetadata</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/utils/_safetensors.py#L74</source><parameters>[{"name": "metadata", "val": ": typing.Optional[dict]"}, {"name": "sharded", "val": ": bool"}, {"name": "weight_map", "val": ": dict"}, {"name": "files_metadata", "val": ": dict"}]</parameters><paramsdesc>- **metadata** (`dict`, *optional*) --
  The metadata contained in the 'model.safetensors.index.json' file, if it exists. Only populated for sharded
  models.
- **sharded** (`bool`) --
  Whether the repo contains a sharded model or not.
- **weight_map** (`dict[str, str]`) --
  A map of all weights. Keys are tensor names and values are filenames of the files containing the tensors.
- **files_metadata** (`dict[str, SafetensorsFileMetadata]`) --
  A map of all files metadata. Keys are filenames and values are the metadata of the corresponding file, as
  a `SafetensorsFileMetadata` object.
- **parameter_count** (`dict[str, int]`) --
  A map of the number of parameters per data type. Keys are data types and values are the number of parameters
  of that data type.</paramsdesc><paramgroups>0</paramgroups></docstring>
Metadata for a Safetensors repo.

A repo is considered to be a Safetensors repo if it contains either a 'model.safetensors' weight file (non-shared
model) or a 'model.safetensors.index.json' index file (sharded model) at its root.

This class is returned by [get_safetensors_metadata()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.get_safetensors_metadata).

For more details regarding the safetensors format, check out https://huggingface.co/docs/safetensors/index#format.




</div>

### SafetensorsFileMetadata[[huggingface_hub.utils.SafetensorsFileMetadata]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.utils.SafetensorsFileMetadata</name><anchor>huggingface_hub.utils.SafetensorsFileMetadata</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/utils/_safetensors.py#L44</source><parameters>[{"name": "metadata", "val": ": dict"}, {"name": "tensors", "val": ": dict"}]</parameters><paramsdesc>- **metadata** (`dict`) --
  The metadata contained in the file.
- **tensors** (`dict[str, TensorInfo]`) --
  A map of all tensors. Keys are tensor names and values are information about the corresponding tensor, as a
  `TensorInfo` object.
- **parameter_count** (`dict[str, int]`) --
  A map of the number of parameters per data type. Keys are data types and values are the number of parameters
  of that data type.</paramsdesc><paramgroups>0</paramgroups></docstring>
Metadata for a Safetensors file hosted on the Hub.

This class is returned by [parse_safetensors_file_metadata()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.parse_safetensors_file_metadata).

For more details regarding the safetensors format, check out https://huggingface.co/docs/safetensors/index#format.




</div>

### SpaceInfo[[huggingface_hub.SpaceInfo]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.SpaceInfo</name><anchor>huggingface_hub.SpaceInfo</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L1012</source><parameters>[{"name": "**kwargs", "val": ""}]</parameters><paramsdesc>- **id** (`str`) --
  ID of the Space.
- **author** (`str`, *optional*) --
  Author of the Space.
- **sha** (`str`, *optional*) --
  Repo SHA at this particular revision.
- **created_at** (`datetime`, *optional*) --
  Date of creation of the repo on the Hub. Note that the lowest value is `2022-03-02T23:29:04.000Z`,
  corresponding to the date when we began to store creation dates.
- **last_modified** (`datetime`, *optional*) --
  Date of last commit to the repo.
- **private** (`bool`) --
  Is the repo private.
- **gated** (`Literal["auto", "manual", False]`, *optional*) --
  Is the repo gated.
  If so, whether there is manual or automatic approval.
- **disabled** (`bool`, *optional*) --
  Is the Space disabled.
- **host** (`str`, *optional*) --
  Host URL of the Space.
- **subdomain** (`str`, *optional*) --
  Subdomain of the Space.
- **likes** (`int`) --
  Number of likes of the Space.
- **tags** (`list[str]`) --
  List of tags of the Space.
- **siblings** (`list[RepoSibling]`) --
  List of [huggingface_hub.hf_api.RepoSibling](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.hf_api.RepoSibling) objects that constitute the Space.
- **card_data** (`SpaceCardData`, *optional*) --
  Space Card Metadata  as a [huggingface_hub.repocard_data.SpaceCardData](/docs/huggingface_hub/main/en/package_reference/cards#huggingface_hub.SpaceCardData) object.
- **runtime** (`SpaceRuntime`, *optional*) --
  Space runtime information as a [huggingface_hub.hf_api.SpaceRuntime](/docs/huggingface_hub/main/en/package_reference/space_runtime#huggingface_hub.SpaceRuntime) object.
- **sdk** (`str`, *optional*) --
  SDK used by the Space.
- **models** (`list[str]`, *optional*) --
  List of models used by the Space.
- **datasets** (`list[str]`, *optional*) --
  List of datasets used by the Space.
- **trending_score** (`int`, *optional*) --
  Trending score of the Space.</paramsdesc><paramgroups>0</paramgroups></docstring>

Contains information about a Space on the Hub. This object is returned by [space_info()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.space_info) and [list_spaces()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.list_spaces).

> [!TIP]
> Most attributes of this class are optional. This is because the data returned by the Hub depends on the query made.
> In general, the more specific the query, the more information is returned. On the contrary, when listing spaces
> using [list_spaces()](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi.list_spaces) only a subset of the attributes are returned.




</div>

### TensorInfo[[huggingface_hub.utils.TensorInfo]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.utils.TensorInfo</name><anchor>huggingface_hub.utils.TensorInfo</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/utils/_safetensors.py#L14</source><parameters>[{"name": "dtype", "val": ": typing.Literal['F64', 'F32', 'F16', 'BF16', 'I64', 'I32', 'I16', 'I8', 'U8', 'BOOL']"}, {"name": "shape", "val": ": list"}, {"name": "data_offsets", "val": ": tuple"}]</parameters><paramsdesc>- **dtype** (`str`) --
  The data type of the tensor ("F64", "F32", "F16", "BF16", "I64", "I32", "I16", "I8", "U8", "BOOL").
- **shape** (`list[int]`) --
  The shape of the tensor.
- **data_offsets** (`tuple[int, int]`) --
  The offsets of the data in the file as a tuple `[BEGIN, END]`.
- **parameter_count** (`int`) --
  The number of parameters in the tensor.</paramsdesc><paramgroups>0</paramgroups></docstring>
Information about a tensor.

For more details regarding the safetensors format, check out https://huggingface.co/docs/safetensors/index#format.




</div>

### User[[huggingface_hub.User]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.User</name><anchor>huggingface_hub.User</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L1411</source><parameters>[{"name": "**kwargs", "val": ""}]</parameters><paramsdesc>- **username** (`str`) --
  Name of the user on the Hub (unique).
- **fullname** (`str`) --
  User's full name.
- **avatar_url** (`str`) --
  URL of the user's avatar.
- **details** (`str`, *optional*) --
  User's details.
- **is_following** (`bool`, *optional*) --
  Whether the authenticated user is following this user.
- **is_pro** (`bool`, *optional*) --
  Whether the user is a pro user.
- **num_models** (`int`, *optional*) --
  Number of models created by the user.
- **num_datasets** (`int`, *optional*) --
  Number of datasets created by the user.
- **num_spaces** (`int`, *optional*) --
  Number of spaces created by the user.
- **num_discussions** (`int`, *optional*) --
  Number of discussions initiated by the user.
- **num_papers** (`int`, *optional*) --
  Number of papers authored by the user.
- **num_upvotes** (`int`, *optional*) --
  Number of upvotes received by the user.
- **num_likes** (`int`, *optional*) --
  Number of likes given by the user.
- **num_following** (`int`, *optional*) --
  Number of users this user is following.
- **num_followers** (`int`, *optional*) --
  Number of users following this user.
- **orgs** (list of `Organization`) --
  List of organizations the user is part of.</paramsdesc><paramgroups>0</paramgroups></docstring>

Contains information about a user on the Hub.




</div>

### UserLikes[[huggingface_hub.UserLikes]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.UserLikes</name><anchor>huggingface_hub.UserLikes</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L1324</source><parameters>[{"name": "user", "val": ": str"}, {"name": "total", "val": ": int"}, {"name": "datasets", "val": ": list[str]"}, {"name": "models", "val": ": list[str]"}, {"name": "spaces", "val": ": list[str]"}]</parameters><paramsdesc>- **user** (`str`) --
  Name of the user for which we fetched the likes.
- **total** (`int`) --
  Total number of likes.
- **datasets** (`list[str]`) --
  List of datasets liked by the user (as repo_ids).
- **models** (`list[str]`) --
  List of models liked by the user (as repo_ids).
- **spaces** (`list[str]`) --
  List of spaces liked by the user (as repo_ids).</paramsdesc><paramgroups>0</paramgroups></docstring>

Contains information about a user likes on the Hub.




</div>

### WebhookInfo[[huggingface_hub.WebhookInfo]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.WebhookInfo</name><anchor>huggingface_hub.WebhookInfo</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L465</source><parameters>[{"name": "id", "val": ": str"}, {"name": "url", "val": ": Optional[str]"}, {"name": "job", "val": ": Optional[JobSpec]"}, {"name": "watched", "val": ": list[WebhookWatchedItem]"}, {"name": "domains", "val": ": list[constants.WEBHOOK_DOMAIN_T]"}, {"name": "secret", "val": ": Optional[str]"}, {"name": "disabled", "val": ": bool"}]</parameters><paramsdesc>- **id** (`str`) --
  ID of the webhook.
- **url** (`str`, *optional*) --
  URL of the webhook.
- **job** (`JobSpec`, *optional*) --
  Specifications of the Job to trigger.
- **watched** (`list[WebhookWatchedItem]`) --
  List of items watched by the webhook, see [WebhookWatchedItem](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.WebhookWatchedItem).
- **domains** (`list[WEBHOOK_DOMAIN_T]`) --
  List of domains the webhook is watching. Can be one of `["repo", "discussions"]`.
- **secret** (`str`, *optional*) --
  Secret of the webhook.
- **disabled** (`bool`) --
  Whether the webhook is disabled or not.</paramsdesc><paramgroups>0</paramgroups></docstring>
Data structure containing information about a webhook.

One of `url` or `job` is specified, but not both.




</div>

### WebhookWatchedItem[[huggingface_hub.WebhookWatchedItem]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.WebhookWatchedItem</name><anchor>huggingface_hub.WebhookWatchedItem</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/hf_api.py#L450</source><parameters>[{"name": "type", "val": ": Literal['dataset', 'model', 'org', 'space', 'user']"}, {"name": "name", "val": ": str"}]</parameters><paramsdesc>- **type** (`Literal["dataset", "model", "org", "space", "user"]`) --
  Type of the item to be watched. Can be one of `["dataset", "model", "org", "space", "user"]`.
- **name** (`str`) --
  Name of the item to be watched. Can be the username, organization name, model name, dataset name or space name.</paramsdesc><paramgroups>0</paramgroups></docstring>
Data structure containing information about the items watched by a webhook.




</div>

## CommitOperation[[huggingface_hub.CommitOperationAdd]]

Below are the supported values for `CommitOperation()`:

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.CommitOperationAdd</name><anchor>huggingface_hub.CommitOperationAdd</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/_commit_api.py#L125</source><parameters>[{"name": "path_in_repo", "val": ": str"}, {"name": "path_or_fileobj", "val": ": typing.Union[str, pathlib.Path, bytes, typing.BinaryIO]"}]</parameters><paramsdesc>- **path_in_repo** (`str`) --
  Relative filepath in the repo, for example: `"checkpoints/1fec34a/weights.bin"`
- **path_or_fileobj** (`str`, `Path`, `bytes`, or `BinaryIO`) --
  Either:
  - a path to a local file (as `str` or `pathlib.Path`) to upload
  - a buffer of bytes (`bytes`) holding the content of the file to upload
  - a "file object" (subclass of `io.BufferedIOBase`), typically obtained
    with `open(path, "rb")`. It must support `seek()` and `tell()` methods.</paramsdesc><paramgroups>0</paramgroups><raises>- [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError) -- 
  If `path_or_fileobj` is not one of `str`, `Path`, `bytes` or `io.BufferedIOBase`.
- [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError) -- 
  If `path_or_fileobj` is a `str` or `Path` but not a path to an existing file.
- [`ValueError`](https://docs.python.org/3/library/exceptions.html#ValueError) -- 
  If `path_or_fileobj` is a `io.BufferedIOBase` but it doesn't support both
  `seek()` and `tell()`.</raises><raisederrors>``ValueError``</raisederrors></docstring>

Data structure holding necessary info to upload a file to a repository on the Hub.









<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>as_file</name><anchor>huggingface_hub.CommitOperationAdd.as_file</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/_commit_api.py#L207</source><parameters>[{"name": "with_tqdm", "val": ": bool = False"}]</parameters><paramsdesc>- **with_tqdm** (`bool`, *optional*, defaults to `False`) --
  If True, iterating over the file object will display a progress bar. Only
  works if the file-like object is a path to a file. Pure bytes and buffers
  are not supported.</paramsdesc><paramgroups>0</paramgroups></docstring>

A context manager that yields a file-like object allowing to read the underlying
data behind `path_or_fileobj`.



<ExampleCodeBlock anchor="huggingface_hub.CommitOperationAdd.as_file.example">

Example:

```python
>>> operation = CommitOperationAdd(
...        path_in_repo="remote/dir/weights.h5",
...        path_or_fileobj="./local/weights.h5",
... )
CommitOperationAdd(path_in_repo='remote/dir/weights.h5', path_or_fileobj='./local/weights.h5')

>>> with operation.as_file() as file:
...     content = file.read()

>>> with operation.as_file(with_tqdm=True) as file:
...     while True:
...         data = file.read(1024)
...         if not data:
...              break
config.json: 100%|█████████████████████████| 8.19k/8.19k [00:02<00:00, 3.72kB/s]

>>> with operation.as_file(with_tqdm=True) as file:
...     httpx.put(..., data=file)
config.json: 100%|█████████████████████████| 8.19k/8.19k [00:02<00:00, 3.72kB/s]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>b64content</name><anchor>huggingface_hub.CommitOperationAdd.b64content</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/_commit_api.py#L257</source><parameters>[]</parameters></docstring>

The base64-encoded content of `path_or_fileobj`

Returns: `bytes`


</div></div>

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.CommitOperationDelete</name><anchor>huggingface_hub.CommitOperationDelete</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/_commit_api.py#L58</source><parameters>[{"name": "path_in_repo", "val": ": str"}, {"name": "is_folder", "val": ": typing.Union[bool, typing.Literal['auto']] = 'auto'"}]</parameters><paramsdesc>- **path_in_repo** (`str`) --
  Relative filepath in the repo, for example: `"checkpoints/1fec34a/weights.bin"`
  for a file or `"checkpoints/1fec34a/"` for a folder.
- **is_folder** (`bool` or `Literal["auto"]`, *optional*) --
  Whether the Delete Operation applies to a folder or not. If "auto", the path
  type (file or folder) is guessed automatically by looking if path ends with
  a "/" (folder) or not (file). To explicitly set the path type, you can set
  `is_folder=True` or `is_folder=False`.</paramsdesc><paramgroups>0</paramgroups></docstring>

Data structure holding necessary info to delete a file or a folder from a repository
on the Hub.




</div>

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.CommitOperationCopy</name><anchor>huggingface_hub.CommitOperationCopy</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/_commit_api.py#L89</source><parameters>[{"name": "src_path_in_repo", "val": ": str"}, {"name": "path_in_repo", "val": ": str"}, {"name": "src_revision", "val": ": typing.Optional[str] = None"}, {"name": "_src_oid", "val": ": typing.Optional[str] = None"}, {"name": "_dest_oid", "val": ": typing.Optional[str] = None"}]</parameters><paramsdesc>- **src_path_in_repo** (`str`) --
  Relative filepath in the repo of the file to be copied, e.g. `"checkpoints/1fec34a/weights.bin"`.
- **path_in_repo** (`str`) --
  Relative filepath in the repo where to copy the file, e.g. `"checkpoints/1fec34a/weights_copy.bin"`.
- **src_revision** (`str`, *optional*) --
  The git revision of the file to be copied. Can be any valid git revision.
  Default to the target commit revision.</paramsdesc><paramgroups>0</paramgroups></docstring>

Data structure holding necessary info to copy a file in a repository on the Hub.

Limitations:
- Only LFS files can be copied. To copy a regular file, you need to download it locally and re-upload it
- Cross-repository copies are not supported.

Note: you can combine a [CommitOperationCopy](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.CommitOperationCopy) and a [CommitOperationDelete](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.CommitOperationDelete) to rename an LFS file on the Hub.




</div>

## CommitScheduler[[huggingface_hub.CommitScheduler]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.CommitScheduler</name><anchor>huggingface_hub.CommitScheduler</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/_commit_scheduler.py#L29</source><parameters>[{"name": "repo_id", "val": ": str"}, {"name": "folder_path", "val": ": typing.Union[str, pathlib.Path]"}, {"name": "every", "val": ": typing.Union[int, float] = 5"}, {"name": "path_in_repo", "val": ": typing.Optional[str] = None"}, {"name": "repo_type", "val": ": typing.Optional[str] = None"}, {"name": "revision", "val": ": typing.Optional[str] = None"}, {"name": "private", "val": ": typing.Optional[bool] = None"}, {"name": "token", "val": ": typing.Optional[str] = None"}, {"name": "allow_patterns", "val": ": typing.Union[list[str], str, NoneType] = None"}, {"name": "ignore_patterns", "val": ": typing.Union[list[str], str, NoneType] = None"}, {"name": "squash_history", "val": ": bool = False"}, {"name": "hf_api", "val": ": typing.Optional[ForwardRef('HfApi')] = None"}]</parameters><paramsdesc>- **repo_id** (`str`) --
  The id of the repo to commit to.
- **folder_path** (`str` or `Path`) --
  Path to the local folder to upload regularly.
- **every** (`int` or `float`, *optional*) --
  The number of minutes between each commit. Defaults to 5 minutes.
- **path_in_repo** (`str`, *optional*) --
  Relative path of the directory in the repo, for example: `"checkpoints/"`. Defaults to the root folder
  of the repository.
- **repo_type** (`str`, *optional*) --
  The type of the repo to commit to. Defaults to `model`.
- **revision** (`str`, *optional*) --
  The revision of the repo to commit to. Defaults to `main`.
- **private** (`bool`, *optional*) --
  Whether to make the repo private. If `None` (default), the repo will be public unless the organization's default is private. This value is ignored if the repo already exists.
- **token** (`str`, *optional*) --
  The token to use to commit to the repo. Defaults to the token saved on the machine.
- **allow_patterns** (`list[str]` or `str`, *optional*) --
  If provided, only files matching at least one pattern are uploaded.
- **ignore_patterns** (`list[str]` or `str`, *optional*) --
  If provided, files matching any of the patterns are not uploaded.
- **squash_history** (`bool`, *optional*) --
  Whether to squash the history of the repo after each commit. Defaults to `False`. Squashing commits is
  useful to avoid degraded performances on the repo when it grows too large.
- **hf_api** (`HfApi`, *optional*) --
  The [HfApi](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.HfApi) client to use to commit to the Hub. Can be set with custom settings (user agent, token,...).</paramsdesc><paramgroups>0</paramgroups></docstring>

Scheduler to upload a local folder to the Hub at regular intervals (e.g. push to hub every 5 minutes).

The recommended way to use the scheduler is to use it as a context manager. This ensures that the scheduler is
properly stopped and the last commit is triggered when the script ends. The scheduler can also be stopped manually
with the `stop` method. Checkout the [upload guide](https://huggingface.co/docs/huggingface_hub/guides/upload#scheduled-uploads)
to learn more about how to use it.



<ExampleCodeBlock anchor="huggingface_hub.CommitScheduler.example">

Example:
```py
>>> from pathlib import Path
>>> from huggingface_hub import CommitScheduler

# Scheduler uploads every 10 minutes
>>> csv_path = Path("watched_folder/data.csv")
>>> CommitScheduler(repo_id="test_scheduler", repo_type="dataset", folder_path=csv_path.parent, every=10)

>>> with csv_path.open("a") as f:
...     f.write("first line")

# Some time later (...)
>>> with csv_path.open("a") as f:
...     f.write("second line")
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.CommitScheduler.example-2">

Example using a context manager:
```py
>>> from pathlib import Path
>>> from huggingface_hub import CommitScheduler

>>> with CommitScheduler(repo_id="test_scheduler", repo_type="dataset", folder_path="watched_folder", every=10) as scheduler:
...     csv_path = Path("watched_folder/data.csv")
...     with csv_path.open("a") as f:
...         f.write("first line")
...     (...)
...     with csv_path.open("a") as f:
...         f.write("second line")

# Scheduler is now stopped and last commit have been triggered
```

</ExampleCodeBlock>



<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>push_to_hub</name><anchor>huggingface_hub.CommitScheduler.push_to_hub</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/_commit_scheduler.py#L204</source><parameters>[]</parameters></docstring>

Push folder to the Hub and return the commit info.

> [!WARNING]
> This method is not meant to be called directly. It is run in the background by the scheduler, respecting a
> queue mechanism to avoid concurrent commits. Making a direct call to the method might lead to concurrency
> issues.

The default behavior of `push_to_hub` is to assume an append-only folder. It lists all files in the folder and
uploads only changed files. If no changes are found, the method returns without committing anything. If you want
to change this behavior, you can inherit from [CommitScheduler](/docs/huggingface_hub/main/en/package_reference/hf_api#huggingface_hub.CommitScheduler) and override this method. This can be useful
for example to compress data together in a single file before committing. For more details and examples, check
out our [integration guide](https://huggingface.co/docs/huggingface_hub/main/en/guides/upload#scheduled-uploads).


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>stop</name><anchor>huggingface_hub.CommitScheduler.stop</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/_commit_scheduler.py#L157</source><parameters>[]</parameters></docstring>
Stop the scheduler.

A stopped scheduler cannot be restarted. Mostly for tests purposes.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>trigger</name><anchor>huggingface_hub.CommitScheduler.trigger</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/_commit_scheduler.py#L181</source><parameters>[]</parameters></docstring>
Trigger a `push_to_hub` and return a future.

This method is automatically called every `every` minutes. You can also call it manually to trigger a commit
immediately, without waiting for the next scheduled commit.


</div></div>

<EditOnGithub source="https://github.com/huggingface/huggingface_hub/blob/main/docs/source/en/package_reference/hf_api.md" />