# Inference

Inference is the process of using a trained model to make predictions on new data. Because this process can be compute-intensive, running on a dedicated or external service can be an interesting option.
The `huggingface_hub` library provides a unified interface to run inference across multiple services for models hosted on the Hugging Face Hub:

1.  [Inference Providers](https://huggingface.co/docs/inference-providers/index): a streamlined, unified access to hundreds of machine learning models, powered by our serverless inference partners. This new approach builds on our previous Serverless Inference API, offering more models, improved performance, and greater reliability thanks to world-class providers. Refer to the [documentation](https://huggingface.co/docs/inference-providers/index#partners) for a list of supported providers.
2.  [Inference Endpoints](https://huggingface.co/docs/inference-endpoints/index): a product to easily deploy models to production. Inference is run by Hugging Face in a dedicated, fully managed infrastructure on a cloud provider of your choice.
3.  Local endpoints: you can also run inference with local inference servers like [llama.cpp](https://github.com/ggerganov/llama.cpp), [Ollama](https://ollama.com/), [vLLM](https://github.com/vllm-project/vllm), [LiteLLM](https://docs.litellm.ai/docs/simple_proxy), or [Text Generation Inference (TGI)](https://github.com/huggingface/text-generation-inference) by connecting the client to these local endpoints.

These services can be called with the [InferenceClient](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceClient) object. Please refer to [this guide](../guides/inference)
for more information on how to use it.

## Inference Client[[huggingface_hub.InferenceClient]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.InferenceClient</name><anchor>huggingface_hub.InferenceClient</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L123</source><parameters>[{"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "provider", "val": ": typing.Union[typing.Literal['black-forest-labs', 'cerebras', 'clarifai', 'cohere', 'fal-ai', 'featherless-ai', 'fireworks-ai', 'groq', 'hf-inference', 'hyperbolic', 'nebius', 'novita', 'nscale', 'openai', 'publicai', 'replicate', 'sambanova', 'scaleway', 'together', 'wavespeed', 'zai-org'], typing.Literal['auto'], NoneType] = None"}, {"name": "token", "val": ": typing.Optional[str] = None"}, {"name": "timeout", "val": ": typing.Optional[float] = None"}, {"name": "headers", "val": ": typing.Optional[dict[str, str]] = None"}, {"name": "cookies", "val": ": typing.Optional[dict[str, str]] = None"}, {"name": "bill_to", "val": ": typing.Optional[str] = None"}, {"name": "base_url", "val": ": typing.Optional[str] = None"}, {"name": "api_key", "val": ": typing.Optional[str] = None"}]</parameters><paramsdesc>- **model** (`str`, `optional`) --
  The model to run inference with. Can be a model id hosted on the Hugging Face Hub, e.g. `meta-llama/Meta-Llama-3-8B-Instruct`
  or a URL to a deployed Inference Endpoint. Defaults to None, in which case a recommended model is
  automatically selected for the task.
  Note: for better compatibility with OpenAI's client, `model` has been aliased as `base_url`. Those 2
  arguments are mutually exclusive. If a URL is passed as `model` or `base_url` for chat completion, the `(/v1)/chat/completions` suffix path will be appended to the URL.
- **provider** (`str`, *optional*) --
  Name of the provider to use for inference. Can be `"black-forest-labs"`, `"cerebras"`, `"clarifai"`, `"cohere"`, `"fal-ai"`, `"featherless-ai"`, `"fireworks-ai"`, `"groq"`, `"hf-inference"`, `"hyperbolic"`, `"nebius"`, `"novita"`, `"nscale"`, `"openai"`, `"publicai"`, `"replicate"`, `"sambanova"`, `"scaleway"`, `"together"`, `"wavespeed"` or `"zai-org"`.
  Defaults to "auto" i.e. the first of the providers available for the model, sorted by the user's order in https://hf.co/settings/inference-providers.
  If model is a URL or `base_url` is passed, then `provider` is not used.
- **token** (`str`, *optional*) --
  Hugging Face token. Will default to the locally saved token if not provided.
  Note: for better compatibility with OpenAI's client, `token` has been aliased as `api_key`. Those 2
  arguments are mutually exclusive and have the exact same behavior.
- **timeout** (`float`, `optional`) --
  The maximum number of seconds to wait for a response from the server. Defaults to None, meaning it will loop until the server is available.
- **headers** (`dict[str, str]`, `optional`) --
  Additional headers to send to the server. By default only the authorization and user-agent headers are sent.
  Values in this dictionary will override the default values.
- **bill_to** (`str`, `optional`) --
  The billing account to use for the requests. By default the requests are billed on the user's account.
  Requests can only be billed to an organization the user is a member of, and which has subscribed to Enterprise Hub.
- **cookies** (`dict[str, str]`, `optional`) --
  Additional cookies to send to the server.
- **base_url** (`str`, `optional`) --
  Base URL to run inference. This is a duplicated argument from `model` to make [InferenceClient](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceClient)
  follow the same pattern as `openai.OpenAI` client. Cannot be used if `model` is set. Defaults to None.
- **api_key** (`str`, `optional`) --
  Token to use for authentication. This is a duplicated argument from `token` to make [InferenceClient](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceClient)
  follow the same pattern as `openai.OpenAI` client. Cannot be used if `token` is set. Defaults to None.</paramsdesc><paramgroups>0</paramgroups></docstring>

Initialize a new Inference Client.

[InferenceClient](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceClient) aims to provide a unified experience to perform inference. The client can be used
seamlessly with either the (free) Inference API, self-hosted Inference Endpoints, or third-party Inference Providers.





<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>audio_classification</name><anchor>huggingface_hub.InferenceClient.audio_classification</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L300</source><parameters>[{"name": "audio", "val": ": typing.Union[bytes, typing.BinaryIO, str, pathlib.Path, ForwardRef('Image'), bytearray, memoryview]"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "top_k", "val": ": typing.Optional[int] = None"}, {"name": "function_to_apply", "val": ": typing.Optional[ForwardRef('AudioClassificationOutputTransform')] = None"}]</parameters><paramsdesc>- **audio** (Union[str, Path, bytes, BinaryIO]) --
  The audio content to classify. It can be raw audio bytes, a local audio file, or a URL pointing to an
  audio file.
- **model** (`str`, *optional*) --
  The model to use for audio classification. Can be a model ID hosted on the Hugging Face Hub
  or a URL to a deployed Inference Endpoint. If not provided, the default recommended model for
  audio classification will be used.
- **top_k** (`int`, *optional*) --
  When specified, limits the output to the top K most probable classes.
- **function_to_apply** (`"AudioClassificationOutputTransform"`, *optional*) --
  The function to apply to the model outputs in order to retrieve the scores.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[AudioClassificationOutputElement]`</rettype><retdesc>List of [AudioClassificationOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.AudioClassificationOutputElement) items containing the predicted labels and their confidence.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Perform audio classification on the provided audio content.











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.audio_classification.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()
>>> client.audio_classification("audio.flac")
[
    AudioClassificationOutputElement(score=0.4976358711719513, label='hap'),
    AudioClassificationOutputElement(score=0.3677836060523987, label='neu'),
    ...
]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>audio_to_audio</name><anchor>huggingface_hub.InferenceClient.audio_to_audio</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L357</source><parameters>[{"name": "audio", "val": ": typing.Union[bytes, typing.BinaryIO, str, pathlib.Path, ForwardRef('Image'), bytearray, memoryview]"}, {"name": "model", "val": ": typing.Optional[str] = None"}]</parameters><paramsdesc>- **audio** (Union[str, Path, bytes, BinaryIO]) --
  The audio content for the model. It can be raw audio bytes, a local audio file, or a URL pointing to an
  audio file.
- **model** (`str`, *optional*) --
  The model can be any model which takes an audio file and returns another audio file. Can be a model ID hosted on the Hugging Face Hub
  or a URL to a deployed Inference Endpoint. If not provided, the default recommended model for
  audio_to_audio will be used.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[AudioToAudioOutputElement]`</rettype><retdesc>A list of [AudioToAudioOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.AudioToAudioOutputElement) items containing audios label, content-type, and audio content in blob.</retdesc><raises>- ``InferenceTimeoutError`` -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>``InferenceTimeoutError`` or `HfHubHTTPError`</raisederrors></docstring>

Performs multiple tasks related to audio-to-audio depending on the model (eg: speech enhancement, source separation).











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.audio_to_audio.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()
>>> audio_output = client.audio_to_audio("audio.flac")
>>> for i, item in enumerate(audio_output):
>>>     with open(f"output_{i}.flac", "wb") as f:
            f.write(item.blob)
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>automatic_speech_recognition</name><anchor>huggingface_hub.InferenceClient.automatic_speech_recognition</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L409</source><parameters>[{"name": "audio", "val": ": typing.Union[bytes, typing.BinaryIO, str, pathlib.Path, ForwardRef('Image'), bytearray, memoryview]"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "extra_body", "val": ": typing.Optional[dict] = None"}]</parameters><paramsdesc>- **audio** (Union[str, Path, bytes, BinaryIO]) --
  The content to transcribe. It can be raw audio bytes, local audio file, or a URL to an audio file.
- **model** (`str`, *optional*) --
  The model to use for ASR. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. If not provided, the default recommended model for ASR will be used.
- **extra_body** (`dict`, *optional*) --
  Additional provider-specific parameters to pass to the model. Refer to the provider's documentation
  for supported parameters.</paramsdesc><paramgroups>0</paramgroups><rettype>[AutomaticSpeechRecognitionOutput](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.AutomaticSpeechRecognitionOutput)</rettype><retdesc>An item containing the transcribed text and optionally the timestamp chunks.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Perform automatic speech recognition (ASR or audio-to-text) on the given audio content.











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.automatic_speech_recognition.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()
>>> client.automatic_speech_recognition("hello_world.flac").text
"hello world"
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>chat_completion</name><anchor>huggingface_hub.InferenceClient.chat_completion</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L535</source><parameters>[{"name": "messages", "val": ": list"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "stream", "val": ": bool = False"}, {"name": "frequency_penalty", "val": ": typing.Optional[float] = None"}, {"name": "logit_bias", "val": ": typing.Optional[list[float]] = None"}, {"name": "logprobs", "val": ": typing.Optional[bool] = None"}, {"name": "max_tokens", "val": ": typing.Optional[int] = None"}, {"name": "n", "val": ": typing.Optional[int] = None"}, {"name": "presence_penalty", "val": ": typing.Optional[float] = None"}, {"name": "response_format", "val": ": typing.Union[huggingface_hub.inference._generated.types.chat_completion.ChatCompletionInputResponseFormatText, huggingface_hub.inference._generated.types.chat_completion.ChatCompletionInputResponseFormatJSONSchema, huggingface_hub.inference._generated.types.chat_completion.ChatCompletionInputResponseFormatJSONObject, NoneType] = None"}, {"name": "seed", "val": ": typing.Optional[int] = None"}, {"name": "stop", "val": ": typing.Optional[list[str]] = None"}, {"name": "stream_options", "val": ": typing.Optional[huggingface_hub.inference._generated.types.chat_completion.ChatCompletionInputStreamOptions] = None"}, {"name": "temperature", "val": ": typing.Optional[float] = None"}, {"name": "tool_choice", "val": ": typing.Union[huggingface_hub.inference._generated.types.chat_completion.ChatCompletionInputToolChoiceClass, ForwardRef('ChatCompletionInputToolChoiceEnum'), NoneType] = None"}, {"name": "tool_prompt", "val": ": typing.Optional[str] = None"}, {"name": "tools", "val": ": typing.Optional[list[huggingface_hub.inference._generated.types.chat_completion.ChatCompletionInputTool]] = None"}, {"name": "top_logprobs", "val": ": typing.Optional[int] = None"}, {"name": "top_p", "val": ": typing.Optional[float] = None"}, {"name": "extra_body", "val": ": typing.Optional[dict] = None"}]</parameters><paramsdesc>- **messages** (List of [ChatCompletionInputMessage](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ChatCompletionInputMessage)) --
  Conversation history consisting of roles and content pairs.
- **model** (`str`, *optional*) --
  The model to use for chat-completion. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. If not provided, the default recommended model for chat-based text-generation will be used.
  See https://huggingface.co/tasks/text-generation for more details.
  If `model` is a model ID, it is passed to the server as the `model` parameter. If you want to define a
  custom URL while setting `model` in the request payload, you must set `base_url` when initializing [InferenceClient](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceClient).
- **frequency_penalty** (`float`, *optional*) --
  Penalizes new tokens based on their existing frequency
  in the text so far. Range: [-2.0, 2.0]. Defaults to 0.0.
- **logit_bias** (`list[float]`, *optional*) --
  Adjusts the likelihood of specific tokens appearing in the generated output.
- **logprobs** (`bool`, *optional*) --
  Whether to return log probabilities of the output tokens or not. If true, returns the log
  probabilities of each output token returned in the content of message.
- **max_tokens** (`int`, *optional*) --
  Maximum number of tokens allowed in the response. Defaults to 100.
- **n** (`int`, *optional*) --
  The number of completions to generate for each prompt.
- **presence_penalty** (`float`, *optional*) --
  Number between -2.0 and 2.0. Positive values penalize new tokens based on whether they appear in the
  text so far, increasing the model's likelihood to talk about new topics.
- **response_format** (`ChatCompletionInputGrammarType()`, *optional*) --
  Grammar constraints. Can be either a JSONSchema or a regex.
- **seed** (Optional`int`, *optional*) --
  Seed for reproducible control flow. Defaults to None.
- **stop** (`list[str]`, *optional*) --
  Up to four strings which trigger the end of the response.
  Defaults to None.
- **stream** (`bool`, *optional*) --
  Enable realtime streaming of responses. Defaults to False.
- **stream_options** ([ChatCompletionInputStreamOptions](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ChatCompletionInputStreamOptions), *optional*) --
  Options for streaming completions.
- **temperature** (`float`, *optional*) --
  Controls randomness of the generations. Lower values ensure
  less random completions. Range: [0, 2]. Defaults to 1.0.
- **top_logprobs** (`int`, *optional*) --
  An integer between 0 and 5 specifying the number of most likely tokens to return at each token
  position, each with an associated log probability. logprobs must be set to true if this parameter is
  used.
- **top_p** (`float`, *optional*) --
  Fraction of the most likely next words to sample from.
  Must be between 0 and 1. Defaults to 1.0.
- **tool_choice** ([ChatCompletionInputToolChoiceClass](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ChatCompletionInputToolChoiceClass) or `ChatCompletionInputToolChoiceEnum()`, *optional*) --
  The tool to use for the completion. Defaults to "auto".
- **tool_prompt** (`str`, *optional*) --
  A prompt to be appended before the tools.
- **tools** (List of [ChatCompletionInputTool](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ChatCompletionInputTool), *optional*) --
  A list of tools the model may call. Currently, only functions are supported as a tool. Use this to
  provide a list of functions the model may generate JSON inputs for.
- **extra_body** (`dict`, *optional*) --
  Additional provider-specific parameters to pass to the model. Refer to the provider's documentation
  for supported parameters.</paramsdesc><paramgroups>0</paramgroups><rettype>[ChatCompletionOutput](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ChatCompletionOutput) or Iterable of [ChatCompletionStreamOutput](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ChatCompletionStreamOutput)</rettype><retdesc>Generated text returned from the server:
- if `stream=False`, the generated text is returned as a [ChatCompletionOutput](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ChatCompletionOutput) (default).
- if `stream=True`, the generated text is returned token by token as a sequence of [ChatCompletionStreamOutput](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ChatCompletionStreamOutput).</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

A method for completing conversations using a specified language model.

> [!TIP]
> The `client.chat_completion` method is aliased as `client.chat.completions.create` for compatibility with OpenAI's client.
> Inputs and outputs are strictly the same and using either syntax will yield the same results.
> Check out the [Inference guide](https://huggingface.co/docs/huggingface_hub/guides/inference#openai-compatibility)
> for more details about OpenAI's compatibility.

> [!TIP]
> You can pass provider-specific parameters to the model by using the `extra_body` argument.











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.chat_completion.example">

Example:

```py
>>> from huggingface_hub import InferenceClient
>>> messages = [{"role": "user", "content": "What is the capital of France?"}]
>>> client = InferenceClient("meta-llama/Meta-Llama-3-8B-Instruct")
>>> client.chat_completion(messages, max_tokens=100)
ChatCompletionOutput(
    choices=[
        ChatCompletionOutputComplete(
            finish_reason='eos_token',
            index=0,
            message=ChatCompletionOutputMessage(
                role='assistant',
                content='The capital of France is Paris.',
                name=None,
                tool_calls=None
            ),
            logprobs=None
        )
    ],
    created=1719907176,
    id='',
    model='meta-llama/Meta-Llama-3-8B-Instruct',
    object='text_completion',
    system_fingerprint='2.0.4-sha-f426a33',
    usage=ChatCompletionOutputUsage(
        completion_tokens=8,
        prompt_tokens=17,
        total_tokens=25
    )
)
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.chat_completion.example-2">

Example using streaming:
```py
>>> from huggingface_hub import InferenceClient
>>> messages = [{"role": "user", "content": "What is the capital of France?"}]
>>> client = InferenceClient("meta-llama/Meta-Llama-3-8B-Instruct")
>>> for token in client.chat_completion(messages, max_tokens=10, stream=True):
...     print(token)
ChatCompletionStreamOutput(choices=[ChatCompletionStreamOutputChoice(delta=ChatCompletionStreamOutputDelta(content='The', role='assistant'), index=0, finish_reason=None)], created=1710498504)
ChatCompletionStreamOutput(choices=[ChatCompletionStreamOutputChoice(delta=ChatCompletionStreamOutputDelta(content=' capital', role='assistant'), index=0, finish_reason=None)], created=1710498504)
(...)
ChatCompletionStreamOutput(choices=[ChatCompletionStreamOutputChoice(delta=ChatCompletionStreamOutputDelta(content=' may', role='assistant'), index=0, finish_reason=None)], created=1710498504)
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.chat_completion.example-3">

Example using OpenAI's syntax:
```py
# instead of `from openai import OpenAI`
from huggingface_hub import InferenceClient

# instead of `client = OpenAI(...)`
client = InferenceClient(
    base_url=...,
    api_key=...,
)

output = client.chat.completions.create(
    model="meta-llama/Meta-Llama-3-8B-Instruct",
    messages=[
        {"role": "system", "content": "You are a helpful assistant."},
        {"role": "user", "content": "Count to 10"},
    ],
    stream=True,
    max_tokens=1024,
)

for chunk in output:
    print(chunk.choices[0].delta.content)
```

</ExampleCodeBlock>

Example using a third-party provider directly with extra (provider-specific) parameters. Usage will be billed on your Together AI account.
<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.chat_completion.example-4">

```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient(
...     provider="together",  # Use Together AI provider
...     api_key="<together_api_key>",  # Pass your Together API key directly
... )
>>> client.chat_completion(
...     model="meta-llama/Meta-Llama-3-8B-Instruct",
...     messages=[{"role": "user", "content": "What is the capital of France?"}],
...     extra_body={"safety_model": "Meta-Llama/Llama-Guard-7b"},
... )
```

</ExampleCodeBlock>

Example using a third-party provider through Hugging Face Routing. Usage will be billed on your Hugging Face account.
<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.chat_completion.example-5">

```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient(
...     provider="sambanova",  # Use Sambanova provider
...     api_key="hf_...",  # Pass your HF token
... )
>>> client.chat_completion(
...     model="meta-llama/Meta-Llama-3-8B-Instruct",
...     messages=[{"role": "user", "content": "What is the capital of France?"}],
... )
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.chat_completion.example-6">

Example using Image + Text as input:
```py
>>> from huggingface_hub import InferenceClient

# provide a remote URL
>>> image_url ="https://cdn.britannica.com/61/93061-050-99147DCE/Statue-of-Liberty-Island-New-York-Bay.jpg"
# or a base64-encoded image
>>> image_path = "/path/to/image.jpeg"
>>> with open(image_path, "rb") as f:
...     base64_image = base64.b64encode(f.read()).decode("utf-8")
>>> image_url = f"data:image/jpeg;base64,{base64_image}"

>>> client = InferenceClient("meta-llama/Llama-3.2-11B-Vision-Instruct")
>>> output = client.chat.completions.create(
...     messages=[
...         {
...             "role": "user",
...             "content": [
...                 {
...                     "type": "image_url",
...                     "image_url": {"url": image_url},
...                 },
...                 {
...                     "type": "text",
...                     "text": "Describe this image in one sentence.",
...                 },
...             ],
...         },
...     ],
... )
>>> output
The image depicts the iconic Statue of Liberty situated in New York Harbor, New York, on a clear day.
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.chat_completion.example-7">

Example using tools:
```py
>>> client = InferenceClient("meta-llama/Meta-Llama-3-70B-Instruct")
>>> messages = [
...     {
...         "role": "system",
...         "content": "Don't make assumptions about what values to plug into functions. Ask for clarification if a user request is ambiguous.",
...     },
...     {
...         "role": "user",
...         "content": "What's the weather like the next 3 days in San Francisco, CA?",
...     },
... ]
>>> tools = [
...     {
...         "type": "function",
...         "function": {
...             "name": "get_current_weather",
...             "description": "Get the current weather",
...             "parameters": {
...                 "type": "object",
...                 "properties": {
...                     "location": {
...                         "type": "string",
...                         "description": "The city and state, e.g. San Francisco, CA",
...                     },
...                     "format": {
...                         "type": "string",
...                         "enum": ["celsius", "fahrenheit"],
...                         "description": "The temperature unit to use. Infer this from the users location.",
...                     },
...                 },
...                 "required": ["location", "format"],
...             },
...         },
...     },
...     {
...         "type": "function",
...         "function": {
...             "name": "get_n_day_weather_forecast",
...             "description": "Get an N-day weather forecast",
...             "parameters": {
...                 "type": "object",
...                 "properties": {
...                     "location": {
...                         "type": "string",
...                         "description": "The city and state, e.g. San Francisco, CA",
...                     },
...                     "format": {
...                         "type": "string",
...                         "enum": ["celsius", "fahrenheit"],
...                         "description": "The temperature unit to use. Infer this from the users location.",
...                     },
...                     "num_days": {
...                         "type": "integer",
...                         "description": "The number of days to forecast",
...                     },
...                 },
...                 "required": ["location", "format", "num_days"],
...             },
...         },
...     },
... ]

>>> response = client.chat_completion(
...     model="meta-llama/Meta-Llama-3-70B-Instruct",
...     messages=messages,
...     tools=tools,
...     tool_choice="auto",
...     max_tokens=500,
... )
>>> response.choices[0].message.tool_calls[0].function
ChatCompletionOutputFunctionDefinition(
    arguments={
        'location': 'San Francisco, CA',
        'format': 'fahrenheit',
        'num_days': 3
    },
    name='get_n_day_weather_forecast',
    description=None
)
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.chat_completion.example-8">

Example using response_format:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient("meta-llama/Meta-Llama-3-70B-Instruct")
>>> messages = [
...     {
...         "role": "user",
...         "content": "I saw a puppy a cat and a raccoon during my bike ride in the park. What did I see and when?",
...     },
... ]
>>> response_format = {
...     "type": "json",
...     "value": {
...         "properties": {
...             "location": {"type": "string"},
...             "activity": {"type": "string"},
...             "animals_seen": {"type": "integer", "minimum": 1, "maximum": 5},
...             "animals": {"type": "array", "items": {"type": "string"}},
...         },
...         "required": ["location", "activity", "animals_seen", "animals"],
...     },
... }
>>> response = client.chat_completion(
...     messages=messages,
...     response_format=response_format,
...     max_tokens=500,
... )
>>> response.choices[0].message.content
'{

y": "bike ride",
": ["puppy", "cat", "raccoon"],
_seen": 3,
n": "park"}'
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>document_question_answering</name><anchor>huggingface_hub.InferenceClient.document_question_answering</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L937</source><parameters>[{"name": "image", "val": ": typing.Union[bytes, typing.BinaryIO, str, pathlib.Path, ForwardRef('Image'), bytearray, memoryview]"}, {"name": "question", "val": ": str"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "doc_stride", "val": ": typing.Optional[int] = None"}, {"name": "handle_impossible_answer", "val": ": typing.Optional[bool] = None"}, {"name": "lang", "val": ": typing.Optional[str] = None"}, {"name": "max_answer_len", "val": ": typing.Optional[int] = None"}, {"name": "max_question_len", "val": ": typing.Optional[int] = None"}, {"name": "max_seq_len", "val": ": typing.Optional[int] = None"}, {"name": "top_k", "val": ": typing.Optional[int] = None"}, {"name": "word_boxes", "val": ": typing.Optional[list[typing.Union[list[float], str]]] = None"}]</parameters><paramsdesc>- **image** (`Union[str, Path, bytes, BinaryIO]`) --
  The input image for the context. It can be raw bytes, an image file, or a URL to an online image.
- **question** (`str`) --
  Question to be answered.
- **model** (`str`, *optional*) --
  The model to use for the document question answering task. Can be a model ID hosted on the Hugging Face Hub or a URL to
  a deployed Inference Endpoint. If not provided, the default recommended document question answering model will be used.
  Defaults to None.
- **doc_stride** (`int`, *optional*) --
  If the words in the document are too long to fit with the question for the model, it will be split in
  several chunks with some overlap. This argument controls the size of that overlap.
- **handle_impossible_answer** (`bool`, *optional*) --
  Whether to accept impossible as an answer
- **lang** (`str`, *optional*) --
  Language to use while running OCR. Defaults to english.
- **max_answer_len** (`int`, *optional*) --
  The maximum length of predicted answers (e.g., only answers with a shorter length are considered).
- **max_question_len** (`int`, *optional*) --
  The maximum length of the question after tokenization. It will be truncated if needed.
- **max_seq_len** (`int`, *optional*) --
  The maximum length of the total sentence (context + question) in tokens of each chunk passed to the
  model. The context will be split in several chunks (using doc_stride as overlap) if needed.
- **top_k** (`int`, *optional*) --
  The number of answers to return (will be chosen by order of likelihood). Can return less than top_k
  answers if there are not enough options available within the context.
- **word_boxes** (`list[Union[list[float], str`, *optional*) --
  A list of words and bounding boxes (normalized 0->1000). If provided, the inference will skip the OCR
  step and use the provided bounding boxes instead.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[DocumentQuestionAnsweringOutputElement]`</rettype><retdesc>a list of [DocumentQuestionAnsweringOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.DocumentQuestionAnsweringOutputElement) items containing the predicted label, associated probability, word ids, and page number.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Answer questions on document images.











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.document_question_answering.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()
>>> client.document_question_answering(image="https://huggingface.co/spaces/impira/docquery/resolve/2359223c1837a7587402bda0f2643382a6eefeab/invoice.png", question="What is the invoice number?")
[DocumentQuestionAnsweringOutputElement(answer='us-001', end=16, score=0.9999666213989258, start=16)]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>feature_extraction</name><anchor>huggingface_hub.InferenceClient.feature_extraction</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L1024</source><parameters>[{"name": "text", "val": ": str"}, {"name": "normalize", "val": ": typing.Optional[bool] = None"}, {"name": "prompt_name", "val": ": typing.Optional[str] = None"}, {"name": "truncate", "val": ": typing.Optional[bool] = None"}, {"name": "truncation_direction", "val": ": typing.Optional[typing.Literal['Left', 'Right']] = None"}, {"name": "model", "val": ": typing.Optional[str] = None"}]</parameters><paramsdesc>- **text** (*str*) --
  The text to embed.
- **model** (*str*, *optional*) --
  The model to use for the feature extraction task. Can be a model ID hosted on the Hugging Face Hub or a URL to
  a deployed Inference Endpoint. If not provided, the default recommended feature extraction model will be used.
  Defaults to None.
- **normalize** (*bool*, *optional*) --
  Whether to normalize the embeddings or not.
  Only available on server powered by Text-Embedding-Inference.
- **prompt_name** (*str*, *optional*) --
  The name of the prompt that should be used by for encoding. If not set, no prompt will be applied.
  Must be a key in the *Sentence Transformers* configuration *prompts* dictionary.
  For example if `prompt_name` is "query" and the `prompts` is &amp;lcub;"query": "query: ",...},
  then the sentence "What is the capital of France?" will be encoded as "query: What is the capital of France?"
  because the prompt text will be prepended before any text to encode.
- **truncate** (*bool*, *optional*) --
  Whether to truncate the embeddings or not.
  Only available on server powered by Text-Embedding-Inference.
- **truncation_direction** (*Literal["Left", "Right"]*, *optional*) --
  Which side of the input should be truncated when *truncate=True* is passed.</paramsdesc><paramgroups>0</paramgroups><rettype>*np.ndarray*</rettype><retdesc>The embedding representing the input text as a float32 numpy array.</retdesc><raises>- [*InferenceTimeoutError*] -- 
  If the model is unavailable or the request times out.
- [*HfHubHTTPError*] -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[*InferenceTimeoutError*] or [*HfHubHTTPError*]</raisederrors></docstring>

Generate embeddings for a given text.











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.feature_extraction.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()
>>> client.feature_extraction("Hi, who are you?")
array([[ 2.424802  ,  2.93384   ,  1.1750331 , ...,  1.240499, -0.13776633, -0.7889173 ],
[-0.42943227, -0.6364878 , -1.693462  , ...,  0.41978157, -2.4336355 ,  0.6162071 ],
...,
[ 0.28552425, -0.928395  , -1.2077185 , ...,  0.76810825, -2.1069427 ,  0.6236161 ]], dtype=float32)
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>fill_mask</name><anchor>huggingface_hub.InferenceClient.fill_mask</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L1097</source><parameters>[{"name": "text", "val": ": str"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "targets", "val": ": typing.Optional[list[str]] = None"}, {"name": "top_k", "val": ": typing.Optional[int] = None"}]</parameters><paramsdesc>- **text** (`str`) --
  a string to be filled from, must contain the [MASK] token (check model card for exact name of the mask).
- **model** (`str`, *optional*) --
  The model to use for the fill mask task. Can be a model ID hosted on the Hugging Face Hub or a URL to
  a deployed Inference Endpoint. If not provided, the default recommended fill mask model will be used.
- **targets** (`list[str`, *optional*) --
  When passed, the model will limit the scores to the passed targets instead of looking up in the whole
  vocabulary. If the provided targets are not in the model vocab, they will be tokenized and the first
  resulting token will be used (with a warning, and that might be slower).
- **top_k** (`int`, *optional*) --
  When passed, overrides the number of predictions to return.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[FillMaskOutputElement]`</rettype><retdesc>a list of [FillMaskOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.FillMaskOutputElement) items containing the predicted label, associated
probability, token reference, and completed text.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Fill in a hole with a missing word (token to be precise).











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.fill_mask.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()
>>> client.fill_mask("The goal of life is <mask>.")
[
    FillMaskOutputElement(score=0.06897063553333282, token=11098, token_str=' happiness', sequence='The goal of life is happiness.'),
    FillMaskOutputElement(score=0.06554922461509705, token=45075, token_str=' immortality', sequence='The goal of life is immortality.')
]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_endpoint_info</name><anchor>huggingface_hub.InferenceClient.get_endpoint_info</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L3270</source><parameters>[{"name": "model", "val": ": typing.Optional[str] = None"}]</parameters><paramsdesc>- **model** (`str`, *optional*) --
  The model to use for inference. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. This parameter overrides the model defined at the instance level. Defaults to None.</paramsdesc><paramgroups>0</paramgroups><rettype>`dict[str, Any]`</rettype><retdesc>Information about the endpoint.</retdesc></docstring>

Get information about the deployed endpoint.

This endpoint is only available on endpoints powered by Text-Generation-Inference (TGI) or Text-Embedding-Inference (TEI).
Endpoints powered by `transformers` return an empty payload.







<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.get_endpoint_info.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient("meta-llama/Meta-Llama-3-70B-Instruct")
>>> client.get_endpoint_info()
{
    'model_id': 'meta-llama/Meta-Llama-3-70B-Instruct',
    'model_sha': None,
    'model_dtype': 'torch.float16',
    'model_device_type': 'cuda',
    'model_pipeline_tag': None,
    'max_concurrent_requests': 128,
    'max_best_of': 2,
    'max_stop_sequences': 4,
    'max_input_length': 8191,
    'max_total_tokens': 8192,
    'waiting_served_ratio': 0.3,
    'max_batch_total_tokens': 1259392,
    'max_waiting_tokens': 20,
    'max_batch_size': None,
    'validation_workers': 32,
    'max_client_batch_size': 4,
    'version': '2.0.2',
    'sha': 'dccab72549635c7eb5ddb17f43f0b7cdff07c214',
    'docker_label': 'sha-dccab72'
}
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>health_check</name><anchor>huggingface_hub.InferenceClient.health_check</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L3328</source><parameters>[{"name": "model", "val": ": typing.Optional[str] = None"}]</parameters><paramsdesc>- **model** (`str`, *optional*) --
  URL of the Inference Endpoint. This parameter overrides the model defined at the instance level. Defaults to None.</paramsdesc><paramgroups>0</paramgroups><rettype>`bool`</rettype><retdesc>True if everything is working fine.</retdesc></docstring>

Check the health of the deployed endpoint.

Health check is only available with Inference Endpoints powered by Text-Generation-Inference (TGI) or Text-Embedding-Inference (TEI).







<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.health_check.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient("https://jzgu0buei5.us-east-1.aws.endpoints.huggingface.cloud")
>>> client.health_check()
True
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>image_classification</name><anchor>huggingface_hub.InferenceClient.image_classification</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L1153</source><parameters>[{"name": "image", "val": ": typing.Union[bytes, typing.BinaryIO, str, pathlib.Path, ForwardRef('Image'), bytearray, memoryview]"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "function_to_apply", "val": ": typing.Optional[ForwardRef('ImageClassificationOutputTransform')] = None"}, {"name": "top_k", "val": ": typing.Optional[int] = None"}]</parameters><paramsdesc>- **image** (`Union[str, Path, bytes, BinaryIO, PIL.Image.Image]`) --
  The image to classify. It can be raw bytes, an image file, a URL to an online image, or a PIL Image.
- **model** (`str`, *optional*) --
  The model to use for image classification. Can be a model ID hosted on the Hugging Face Hub or a URL to a
  deployed Inference Endpoint. If not provided, the default recommended model for image classification will be used.
- **function_to_apply** (`"ImageClassificationOutputTransform"`, *optional*) --
  The function to apply to the model outputs in order to retrieve the scores.
- **top_k** (`int`, *optional*) --
  When specified, limits the output to the top K most probable classes.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[ImageClassificationOutputElement]`</rettype><retdesc>a list of [ImageClassificationOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ImageClassificationOutputElement) items containing the predicted label and associated probability.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Perform image classification on the given image using the specified model.











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.image_classification.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()
>>> client.image_classification("https://upload.wikimedia.org/wikipedia/commons/thumb/4/43/Cute_dog.jpg/320px-Cute_dog.jpg")
[ImageClassificationOutputElement(label='Blenheim spaniel', score=0.9779096841812134), ...]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>image_segmentation</name><anchor>huggingface_hub.InferenceClient.image_segmentation</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L1203</source><parameters>[{"name": "image", "val": ": typing.Union[bytes, typing.BinaryIO, str, pathlib.Path, ForwardRef('Image'), bytearray, memoryview]"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "mask_threshold", "val": ": typing.Optional[float] = None"}, {"name": "overlap_mask_area_threshold", "val": ": typing.Optional[float] = None"}, {"name": "subtask", "val": ": typing.Optional[ForwardRef('ImageSegmentationSubtask')] = None"}, {"name": "threshold", "val": ": typing.Optional[float] = None"}]</parameters><paramsdesc>- **image** (`Union[str, Path, bytes, BinaryIO, PIL.Image.Image]`) --
  The image to segment. It can be raw bytes, an image file, a URL to an online image, or a PIL Image.
- **model** (`str`, *optional*) --
  The model to use for image segmentation. Can be a model ID hosted on the Hugging Face Hub or a URL to a
  deployed Inference Endpoint. If not provided, the default recommended model for image segmentation will be used.
- **mask_threshold** (`float`, *optional*) --
  Threshold to use when turning the predicted masks into binary values.
- **overlap_mask_area_threshold** (`float`, *optional*) --
  Mask overlap threshold to eliminate small, disconnected segments.
- **subtask** (`"ImageSegmentationSubtask"`, *optional*) --
  Segmentation task to be performed, depending on model capabilities.
- **threshold** (`float`, *optional*) --
  Probability threshold to filter out predicted masks.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[ImageSegmentationOutputElement]`</rettype><retdesc>A list of [ImageSegmentationOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ImageSegmentationOutputElement) items containing the segmented masks and associated attributes.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Perform image segmentation on the given image using the specified model.

> [!WARNING]
> You must have `PIL` installed if you want to work with images (`pip install Pillow`).











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.image_segmentation.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()
>>> client.image_segmentation("cat.jpg")
[ImageSegmentationOutputElement(score=0.989008, label='LABEL_184', mask=<PIL.PngImagePlugin.PngImageFile image mode=L size=400x300 at 0x7FDD2B129CC0>), ...]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>image_to_image</name><anchor>huggingface_hub.InferenceClient.image_to_image</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L1271</source><parameters>[{"name": "image", "val": ": typing.Union[bytes, typing.BinaryIO, str, pathlib.Path, ForwardRef('Image'), bytearray, memoryview]"}, {"name": "prompt", "val": ": typing.Optional[str] = None"}, {"name": "negative_prompt", "val": ": typing.Optional[str] = None"}, {"name": "num_inference_steps", "val": ": typing.Optional[int] = None"}, {"name": "guidance_scale", "val": ": typing.Optional[float] = None"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "target_size", "val": ": typing.Optional[huggingface_hub.inference._generated.types.image_to_image.ImageToImageTargetSize] = None"}, {"name": "**kwargs", "val": ""}]</parameters><paramsdesc>- **image** (`Union[str, Path, bytes, BinaryIO, PIL.Image.Image]`) --
  The input image for translation. It can be raw bytes, an image file, a URL to an online image, or a PIL Image.
- **prompt** (`str`, *optional*) --
  The text prompt to guide the image generation.
- **negative_prompt** (`str`, *optional*) --
  One prompt to guide what NOT to include in image generation.
- **num_inference_steps** (`int`, *optional*) --
  For diffusion models. The number of denoising steps. More denoising steps usually lead to a higher
  quality image at the expense of slower inference.
- **guidance_scale** (`float`, *optional*) --
  For diffusion models. A higher guidance scale value encourages the model to generate images closely
  linked to the text prompt at the expense of lower image quality.
- **model** (`str`, *optional*) --
  The model to use for inference. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. This parameter overrides the model defined at the instance level. Defaults to None.
- **target_size** (`ImageToImageTargetSize`, *optional*) --
  The size in pixels of the output image. This parameter is only supported by some providers and for
  specific models. It will be ignored when unsupported.</paramsdesc><paramgroups>0</paramgroups><rettype>`Image`</rettype><retdesc>The translated image.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Perform image-to-image translation using a specified model.

> [!WARNING]
> You must have `PIL` installed if you want to work with images (`pip install Pillow`).











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.image_to_image.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()
>>> image = client.image_to_image("cat.jpg", prompt="turn the cat into a tiger")
>>> image.save("tiger.jpg")
```

</ExampleCodeBlock>



</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>image_to_text</name><anchor>huggingface_hub.InferenceClient.image_to_text</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L1426</source><parameters>[{"name": "image", "val": ": typing.Union[bytes, typing.BinaryIO, str, pathlib.Path, ForwardRef('Image'), bytearray, memoryview]"}, {"name": "model", "val": ": typing.Optional[str] = None"}]</parameters><paramsdesc>- **image** (`Union[str, Path, bytes, BinaryIO, PIL.Image.Image]`) --
  The input image to caption. It can be raw bytes, an image file, a URL to an online image, or a PIL Image.
- **model** (`str`, *optional*) --
  The model to use for inference. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. This parameter overrides the model defined at the instance level. Defaults to None.</paramsdesc><paramgroups>0</paramgroups><rettype>[ImageToTextOutput](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ImageToTextOutput)</rettype><retdesc>The generated text.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Takes an input image and return text.

Models can have very different outputs depending on your use case (image captioning, optical character recognition
(OCR), Pix2Struct, etc.). Please have a look to the model card to learn more about a model's specificities.











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.image_to_text.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()
>>> client.image_to_text("cat.jpg")
'a cat standing in a grassy field '
>>> client.image_to_text("https://upload.wikimedia.org/wikipedia/commons/thumb/4/43/Cute_dog.jpg/320px-Cute_dog.jpg")
'a dog laying on the grass next to a flower pot '
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>image_to_video</name><anchor>huggingface_hub.InferenceClient.image_to_video</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L1347</source><parameters>[{"name": "image", "val": ": typing.Union[bytes, typing.BinaryIO, str, pathlib.Path, ForwardRef('Image'), bytearray, memoryview]"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "prompt", "val": ": typing.Optional[str] = None"}, {"name": "negative_prompt", "val": ": typing.Optional[str] = None"}, {"name": "num_frames", "val": ": typing.Optional[float] = None"}, {"name": "num_inference_steps", "val": ": typing.Optional[int] = None"}, {"name": "guidance_scale", "val": ": typing.Optional[float] = None"}, {"name": "seed", "val": ": typing.Optional[int] = None"}, {"name": "target_size", "val": ": typing.Optional[huggingface_hub.inference._generated.types.image_to_video.ImageToVideoTargetSize] = None"}, {"name": "**kwargs", "val": ""}]</parameters><paramsdesc>- **image** (`Union[str, Path, bytes, BinaryIO, PIL.Image.Image]`) --
  The input image to generate a video from. It can be raw bytes, an image file, a URL to an online image, or a PIL Image.
- **model** (`str`, *optional*) --
  The model to use for inference. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. This parameter overrides the model defined at the instance level. Defaults to None.
- **prompt** (`str`, *optional*) --
  The text prompt to guide the video generation.
- **negative_prompt** (`str`, *optional*) --
  One prompt to guide what NOT to include in video generation.
- **num_frames** (`float`, *optional*) --
  The num_frames parameter determines how many video frames are generated.
- **num_inference_steps** (`int`, *optional*) --
  For diffusion models. The number of denoising steps. More denoising steps usually lead to a higher
  quality image at the expense of slower inference.
- **guidance_scale** (`float`, *optional*) --
  For diffusion models. A higher guidance scale value encourages the model to generate videos closely
  linked to the text prompt at the expense of lower image quality.
- **seed** (`int`, *optional*) --
  The seed to use for the video generation.
- **target_size** (`ImageToVideoTargetSize`, *optional*) --
  The size in pixel of the output video frames.
- **num_inference_steps** (`int`, *optional*) --
  The number of denoising steps. More denoising steps usually lead to a higher quality video at the
  expense of slower inference.
- **seed** (`int`, *optional*) --
  Seed for the random number generator.</paramsdesc><paramgroups>0</paramgroups><rettype>`bytes`</rettype><retdesc>The generated video.</retdesc></docstring>

Generate a video from an input image.







<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.image_to_video.example">

Examples:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()
>>> video = client.image_to_video("cat.jpg", model="Wan-AI/Wan2.2-I2V-A14B", prompt="turn the cat into a tiger")
>>> with open("tiger.mp4", "wb") as f:
...     f.write(video)
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>object_detection</name><anchor>huggingface_hub.InferenceClient.object_detection</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L1472</source><parameters>[{"name": "image", "val": ": typing.Union[bytes, typing.BinaryIO, str, pathlib.Path, ForwardRef('Image'), bytearray, memoryview]"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "threshold", "val": ": typing.Optional[float] = None"}]</parameters><paramsdesc>- **image** (`Union[str, Path, bytes, BinaryIO, PIL.Image.Image]`) --
  The image to detect objects on. It can be raw bytes, an image file, a URL to an online image, or a PIL Image.
- **model** (`str`, *optional*) --
  The model to use for object detection. Can be a model ID hosted on the Hugging Face Hub or a URL to a
  deployed Inference Endpoint. If not provided, the default recommended model for object detection (DETR) will be used.
- **threshold** (`float`, *optional*) --
  The probability necessary to make a prediction.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[ObjectDetectionOutputElement]`</rettype><retdesc>A list of [ObjectDetectionOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ObjectDetectionOutputElement) items containing the bounding boxes and associated attributes.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.
- ``ValueError`` -- 
  If the request output is not a List.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError` or ``ValueError``</raisederrors></docstring>

Perform object detection on the given image using the specified model.

> [!WARNING]
> You must have `PIL` installed if you want to work with images (`pip install Pillow`).











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.object_detection.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()
>>> client.object_detection("people.jpg")
[ObjectDetectionOutputElement(score=0.9486683011054993, label='person', box=ObjectDetectionBoundingBox(xmin=59, ymin=39, xmax=420, ymax=510)), ...]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>question_answering</name><anchor>huggingface_hub.InferenceClient.question_answering</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L1520</source><parameters>[{"name": "question", "val": ": str"}, {"name": "context", "val": ": str"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "align_to_words", "val": ": typing.Optional[bool] = None"}, {"name": "doc_stride", "val": ": typing.Optional[int] = None"}, {"name": "handle_impossible_answer", "val": ": typing.Optional[bool] = None"}, {"name": "max_answer_len", "val": ": typing.Optional[int] = None"}, {"name": "max_question_len", "val": ": typing.Optional[int] = None"}, {"name": "max_seq_len", "val": ": typing.Optional[int] = None"}, {"name": "top_k", "val": ": typing.Optional[int] = None"}]</parameters><paramsdesc>- **question** (`str`) --
  Question to be answered.
- **context** (`str`) --
  The context of the question.
- **model** (`str`) --
  The model to use for the question answering task. Can be a model ID hosted on the Hugging Face Hub or a URL to
  a deployed Inference Endpoint.
- **align_to_words** (`bool`, *optional*) --
  Attempts to align the answer to real words. Improves quality on space separated languages. Might hurt
  on non-space-separated languages (like Japanese or Chinese)
- **doc_stride** (`int`, *optional*) --
  If the context is too long to fit with the question for the model, it will be split in several chunks
  with some overlap. This argument controls the size of that overlap.
- **handle_impossible_answer** (`bool`, *optional*) --
  Whether to accept impossible as an answer.
- **max_answer_len** (`int`, *optional*) --
  The maximum length of predicted answers (e.g., only answers with a shorter length are considered).
- **max_question_len** (`int`, *optional*) --
  The maximum length of the question after tokenization. It will be truncated if needed.
- **max_seq_len** (`int`, *optional*) --
  The maximum length of the total sentence (context + question) in tokens of each chunk passed to the
  model. The context will be split in several chunks (using docStride as overlap) if needed.
- **top_k** (`int`, *optional*) --
  The number of answers to return (will be chosen by order of likelihood). Note that we return less than
  topk answers if there are not enough options available within the context.</paramsdesc><paramgroups>0</paramgroups><rettype>Union[`QuestionAnsweringOutputElement`, list[QuestionAnsweringOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.QuestionAnsweringOutputElement)]</rettype><retdesc>When top_k is 1 or not provided, it returns a single `QuestionAnsweringOutputElement`.
When top_k is greater than 1, it returns a list of `QuestionAnsweringOutputElement`.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Retrieve the answer to a question from a given text.











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.question_answering.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()
>>> client.question_answering(question="What's my name?", context="My name is Clara and I live in Berkeley.")
QuestionAnsweringOutputElement(answer='Clara', end=16, score=0.9326565265655518, start=11)
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>sentence_similarity</name><anchor>huggingface_hub.InferenceClient.sentence_similarity</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L1604</source><parameters>[{"name": "sentence", "val": ": str"}, {"name": "other_sentences", "val": ": list"}, {"name": "model", "val": ": typing.Optional[str] = None"}]</parameters><paramsdesc>- **sentence** (`str`) --
  The main sentence to compare to others.
- **other_sentences** (`list[str]`) --
  The list of sentences to compare to.
- **model** (`str`, *optional*) --
  The model to use for the sentence similarity task. Can be a model ID hosted on the Hugging Face Hub or a URL to
  a deployed Inference Endpoint. If not provided, the default recommended sentence similarity model will be used.
  Defaults to None.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[float]`</rettype><retdesc>The embedding representing the input text.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Compute the semantic similarity between a sentence and a list of other sentences by comparing their embeddings.











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.sentence_similarity.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()
>>> client.sentence_similarity(
...     "Machine learning is so easy.",
...     other_sentences=[
...         "Deep learning is so straightforward.",
...         "This is so difficult, like rocket science.",
...         "I can't believe how much I struggled with this.",
...     ],
... )
[0.7785726189613342, 0.45876261591911316, 0.2906220555305481]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>summarization</name><anchor>huggingface_hub.InferenceClient.summarization</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L1657</source><parameters>[{"name": "text", "val": ": str"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "clean_up_tokenization_spaces", "val": ": typing.Optional[bool] = None"}, {"name": "generate_parameters", "val": ": typing.Optional[dict[str, typing.Any]] = None"}, {"name": "truncation", "val": ": typing.Optional[ForwardRef('SummarizationTruncationStrategy')] = None"}]</parameters><paramsdesc>- **text** (`str`) --
  The input text to summarize.
- **model** (`str`, *optional*) --
  The model to use for inference. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. If not provided, the default recommended model for summarization will be used.
- **clean_up_tokenization_spaces** (`bool`, *optional*) --
  Whether to clean up the potential extra spaces in the text output.
- **generate_parameters** (`dict[str, Any]`, *optional*) --
  Additional parametrization of the text generation algorithm.
- **truncation** (`"SummarizationTruncationStrategy"`, *optional*) --
  The truncation strategy to use.</paramsdesc><paramgroups>0</paramgroups><rettype>[SummarizationOutput](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.SummarizationOutput)</rettype><retdesc>The generated summary text.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Generate a summary of a given text using a specified model.











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.summarization.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()
>>> client.summarization("The Eiffel tower...")
SummarizationOutput(generated_text="The Eiffel tower is one of the most famous landmarks in the world....")
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>table_question_answering</name><anchor>huggingface_hub.InferenceClient.table_question_answering</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L1715</source><parameters>[{"name": "table", "val": ": dict"}, {"name": "query", "val": ": str"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "padding", "val": ": typing.Optional[ForwardRef('Padding')] = None"}, {"name": "sequential", "val": ": typing.Optional[bool] = None"}, {"name": "truncation", "val": ": typing.Optional[bool] = None"}]</parameters><paramsdesc>- **table** (`str`) --
  A table of data represented as a dict of lists where entries are headers and the lists are all the
  values, all lists must have the same size.
- **query** (`str`) --
  The query in plain text that you want to ask the table.
- **model** (`str`) --
  The model to use for the table-question-answering task. Can be a model ID hosted on the Hugging Face
  Hub or a URL to a deployed Inference Endpoint.
- **padding** (`"Padding"`, *optional*) --
  Activates and controls padding.
- **sequential** (`bool`, *optional*) --
  Whether to do inference sequentially or as a batch. Batching is faster, but models like SQA require the
  inference to be done sequentially to extract relations within sequences, given their conversational
  nature.
- **truncation** (`bool`, *optional*) --
  Activates and controls truncation.</paramsdesc><paramgroups>0</paramgroups><rettype>[TableQuestionAnsweringOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.TableQuestionAnsweringOutputElement)</rettype><retdesc>a table question answering output containing the answer, coordinates, cells and the aggregator used.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Retrieve the answer to a question from information given in a table.











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.table_question_answering.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()
>>> query = "How many stars does the transformers repository have?"
>>> table = {"Repository": ["Transformers", "Datasets", "Tokenizers"], "Stars": ["36542", "4512", "3934"]}
>>> client.table_question_answering(table, query, model="google/tapas-base-finetuned-wtq")
TableQuestionAnsweringOutputElement(answer='36542', coordinates=[[0, 1]], cells=['36542'], aggregator='AVERAGE')
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>tabular_classification</name><anchor>huggingface_hub.InferenceClient.tabular_classification</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L1777</source><parameters>[{"name": "table", "val": ": dict"}, {"name": "model", "val": ": typing.Optional[str] = None"}]</parameters><paramsdesc>- **table** (`dict[str, Any]`) --
  Set of attributes to classify.
- **model** (`str`, *optional*) --
  The model to use for the tabular classification task. Can be a model ID hosted on the Hugging Face Hub or a URL to
  a deployed Inference Endpoint. If not provided, the default recommended tabular classification model will be used.
  Defaults to None.</paramsdesc><paramgroups>0</paramgroups><rettype>`List`</rettype><retdesc>a list of labels, one per row in the initial table.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Classifying a target category (a group) based on a set of attributes.











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.tabular_classification.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()
>>> table = {
...     "fixed_acidity": ["7.4", "7.8", "10.3"],
...     "volatile_acidity": ["0.7", "0.88", "0.32"],
...     "citric_acid": ["0", "0", "0.45"],
...     "residual_sugar": ["1.9", "2.6", "6.4"],
...     "chlorides": ["0.076", "0.098", "0.073"],
...     "free_sulfur_dioxide": ["11", "25", "5"],
...     "total_sulfur_dioxide": ["34", "67", "13"],
...     "density": ["0.9978", "0.9968", "0.9976"],
...     "pH": ["3.51", "3.2", "3.23"],
...     "sulphates": ["0.56", "0.68", "0.82"],
...     "alcohol": ["9.4", "9.8", "12.6"],
... }
>>> client.tabular_classification(table=table, model="julien-c/wine-quality")
["5", "5", "5"]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>tabular_regression</name><anchor>huggingface_hub.InferenceClient.tabular_regression</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L1832</source><parameters>[{"name": "table", "val": ": dict"}, {"name": "model", "val": ": typing.Optional[str] = None"}]</parameters><paramsdesc>- **table** (`dict[str, Any]`) --
  Set of attributes stored in a table. The attributes used to predict the target can be both numerical and categorical.
- **model** (`str`, *optional*) --
  The model to use for the tabular regression task. Can be a model ID hosted on the Hugging Face Hub or a URL to
  a deployed Inference Endpoint. If not provided, the default recommended tabular regression model will be used.
  Defaults to None.</paramsdesc><paramgroups>0</paramgroups><rettype>`List`</rettype><retdesc>a list of predicted numerical target values.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Predicting a numerical target value given a set of attributes/features in a table.











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.tabular_regression.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()
>>> table = {
...     "Height": ["11.52", "12.48", "12.3778"],
...     "Length1": ["23.2", "24", "23.9"],
...     "Length2": ["25.4", "26.3", "26.5"],
...     "Length3": ["30", "31.2", "31.1"],
...     "Species": ["Bream", "Bream", "Bream"],
...     "Width": ["4.02", "4.3056", "4.6961"],
... }
>>> client.tabular_regression(table, model="scikit-learn/Fish-Weight")
[110, 120, 130]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>text_classification</name><anchor>huggingface_hub.InferenceClient.text_classification</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L1882</source><parameters>[{"name": "text", "val": ": str"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "top_k", "val": ": typing.Optional[int] = None"}, {"name": "function_to_apply", "val": ": typing.Optional[ForwardRef('TextClassificationOutputTransform')] = None"}]</parameters><paramsdesc>- **text** (`str`) --
  A string to be classified.
- **model** (`str`, *optional*) --
  The model to use for the text classification task. Can be a model ID hosted on the Hugging Face Hub or a URL to
  a deployed Inference Endpoint. If not provided, the default recommended text classification model will be used.
  Defaults to None.
- **top_k** (`int`, *optional*) --
  When specified, limits the output to the top K most probable classes.
- **function_to_apply** (`"TextClassificationOutputTransform"`, *optional*) --
  The function to apply to the model outputs in order to retrieve the scores.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[TextClassificationOutputElement]`</rettype><retdesc>a list of [TextClassificationOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.TextClassificationOutputElement) items containing the predicted label and associated probability.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Perform text classification (e.g. sentiment-analysis) on the given text.











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.text_classification.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()
>>> client.text_classification("I like you")
[
    TextClassificationOutputElement(label='POSITIVE', score=0.9998695850372314),
    TextClassificationOutputElement(label='NEGATIVE', score=0.0001304351753788069),
]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>text_generation</name><anchor>huggingface_hub.InferenceClient.text_generation</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L2090</source><parameters>[{"name": "prompt", "val": ": str"}, {"name": "details", "val": ": typing.Optional[bool] = None"}, {"name": "stream", "val": ": typing.Optional[bool] = None"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "adapter_id", "val": ": typing.Optional[str] = None"}, {"name": "best_of", "val": ": typing.Optional[int] = None"}, {"name": "decoder_input_details", "val": ": typing.Optional[bool] = None"}, {"name": "do_sample", "val": ": typing.Optional[bool] = None"}, {"name": "frequency_penalty", "val": ": typing.Optional[float] = None"}, {"name": "grammar", "val": ": typing.Optional[huggingface_hub.inference._generated.types.text_generation.TextGenerationInputGrammarType] = None"}, {"name": "max_new_tokens", "val": ": typing.Optional[int] = None"}, {"name": "repetition_penalty", "val": ": typing.Optional[float] = None"}, {"name": "return_full_text", "val": ": typing.Optional[bool] = None"}, {"name": "seed", "val": ": typing.Optional[int] = None"}, {"name": "stop", "val": ": typing.Optional[list[str]] = None"}, {"name": "stop_sequences", "val": ": typing.Optional[list[str]] = None"}, {"name": "temperature", "val": ": typing.Optional[float] = None"}, {"name": "top_k", "val": ": typing.Optional[int] = None"}, {"name": "top_n_tokens", "val": ": typing.Optional[int] = None"}, {"name": "top_p", "val": ": typing.Optional[float] = None"}, {"name": "truncate", "val": ": typing.Optional[int] = None"}, {"name": "typical_p", "val": ": typing.Optional[float] = None"}, {"name": "watermark", "val": ": typing.Optional[bool] = None"}]</parameters><paramsdesc>- **prompt** (`str`) --
  Input text.
- **details** (`bool`, *optional*) --
  By default, text_generation returns a string. Pass `details=True` if you want a detailed output (tokens,
  probabilities, seed, finish reason, etc.). Only available for models running on with the
  `text-generation-inference` backend.
- **stream** (`bool`, *optional*) --
  By default, text_generation returns the full generated text. Pass `stream=True` if you want a stream of
  tokens to be returned. Only available for models running on with the `text-generation-inference`
  backend.
- **model** (`str`, *optional*) --
  The model to use for inference. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. This parameter overrides the model defined at the instance level. Defaults to None.
- **adapter_id** (`str`, *optional*) --
  Lora adapter id.
- **best_of** (`int`, *optional*) --
  Generate best_of sequences and return the one if the highest token logprobs.
- **decoder_input_details** (`bool`, *optional*) --
  Return the decoder input token logprobs and ids. You must set `details=True` as well for it to be taken
  into account. Defaults to `False`.
- **do_sample** (`bool`, *optional*) --
  Activate logits sampling
- **frequency_penalty** (`float`, *optional*) --
  Number between -2.0 and 2.0. Positive values penalize new tokens based on their existing frequency in
  the text so far, decreasing the model's likelihood to repeat the same line verbatim.
- **grammar** ([TextGenerationInputGrammarType](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.TextGenerationInputGrammarType), *optional*) --
  Grammar constraints. Can be either a JSONSchema or a regex.
- **max_new_tokens** (`int`, *optional*) --
  Maximum number of generated tokens. Defaults to 100.
- **repetition_penalty** (`float`, *optional*) --
  The parameter for repetition penalty. 1.0 means no penalty. See [this
  paper](https://arxiv.org/pdf/1909.05858.pdf) for more details.
- **return_full_text** (`bool`, *optional*) --
  Whether to prepend the prompt to the generated text
- **seed** (`int`, *optional*) --
  Random sampling seed
- **stop** (`list[str]`, *optional*) --
  Stop generating tokens if a member of `stop` is generated.
- **stop_sequences** (`list[str]`, *optional*) --
  Deprecated argument. Use `stop` instead.
- **temperature** (`float`, *optional*) --
  The value used to module the logits distribution.
- **top_n_tokens** (`int`, *optional*) --
  Return information about the `top_n_tokens` most likely tokens at each generation step, instead of
  just the sampled token.
- **top_k** (`int`, *optional`) --
  The number of highest probability vocabulary tokens to keep for top-k-filtering.
- **top_p** (`float`, *optional`) --
  If set to < 1, only the smallest set of most probable tokens with probabilities that add up to `top_p` or
  higher are kept for generation.
- **truncate** (`int`, *optional`) --
  Truncate inputs tokens to the given size.
- **typical_p** (`float`, *optional`) --
  Typical Decoding mass
  See [Typical Decoding for Natural Language Generation](https://arxiv.org/abs/2202.00666) for more information
- **watermark** (`bool`, *optional*) --
  Watermarking with [A Watermark for Large Language Models](https://arxiv.org/abs/2301.10226)</paramsdesc><paramgroups>0</paramgroups><rettype>`Union[str, TextGenerationOutput, Iterable[str], Iterable[TextGenerationStreamOutput]]`</rettype><retdesc>Generated text returned from the server:
- if `stream=False` and `details=False`, the generated text is returned as a `str` (default)
- if `stream=True` and `details=False`, the generated text is returned token by token as a `Iterable[str]`
- if `stream=False` and `details=True`, the generated text is returned with more details as a [TextGenerationOutput](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.TextGenerationOutput)
- if `details=True` and `stream=True`, the generated text is returned token by token as a iterable of [TextGenerationStreamOutput](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.TextGenerationStreamOutput)</retdesc><raises>- ``ValidationError`` -- 
  If input values are not valid. No HTTP call is made to the server.
- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>``ValidationError`` or [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Given a prompt, generate the following text.

> [!TIP]
> If you want to generate a response from chat messages, you should use the [InferenceClient.chat_completion()](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceClient.chat_completion) method.
> It accepts a list of messages instead of a single text prompt and handles the chat templating for you.











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.text_generation.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()

# Case 1: generate text
>>> client.text_generation("The huggingface_hub library is ", max_new_tokens=12)
'100% open source and built to be easy to use.'

# Case 2: iterate over the generated tokens. Useful for large generation.
>>> for token in client.text_generation("The huggingface_hub library is ", max_new_tokens=12, stream=True):
...     print(token)
100
%
open
source
and
built
to
be
easy
to
use
.

# Case 3: get more details about the generation process.
>>> client.text_generation("The huggingface_hub library is ", max_new_tokens=12, details=True)
TextGenerationOutput(
    generated_text='100% open source and built to be easy to use.',
    details=TextGenerationDetails(
        finish_reason='length',
        generated_tokens=12,
        seed=None,
        prefill=[
            TextGenerationPrefillOutputToken(id=487, text='The', logprob=None),
            TextGenerationPrefillOutputToken(id=53789, text=' hugging', logprob=-13.171875),
            (...)
            TextGenerationPrefillOutputToken(id=204, text=' ', logprob=-7.0390625)
        ],
        tokens=[
            TokenElement(id=1425, text='100', logprob=-1.0175781, special=False),
            TokenElement(id=16, text='%', logprob=-0.0463562, special=False),
            (...)
            TokenElement(id=25, text='.', logprob=-0.5703125, special=False)
        ],
        best_of_sequences=None
    )
)

# Case 4: iterate over the generated tokens with more details.
# Last object is more complete, containing the full generated text and the finish reason.
>>> for details in client.text_generation("The huggingface_hub library is ", max_new_tokens=12, details=True, stream=True):
...     print(details)
...
TextGenerationStreamOutput(token=TokenElement(id=1425, text='100', logprob=-1.0175781, special=False), generated_text=None, details=None)
TextGenerationStreamOutput(token=TokenElement(id=16, text='%', logprob=-0.0463562, special=False), generated_text=None, details=None)
TextGenerationStreamOutput(token=TokenElement(id=1314, text=' open', logprob=-1.3359375, special=False), generated_text=None, details=None)
TextGenerationStreamOutput(token=TokenElement(id=3178, text=' source', logprob=-0.28100586, special=False), generated_text=None, details=None)
TextGenerationStreamOutput(token=TokenElement(id=273, text=' and', logprob=-0.5961914, special=False), generated_text=None, details=None)
TextGenerationStreamOutput(token=TokenElement(id=3426, text=' built', logprob=-1.9423828, special=False), generated_text=None, details=None)
TextGenerationStreamOutput(token=TokenElement(id=271, text=' to', logprob=-1.4121094, special=False), generated_text=None, details=None)
TextGenerationStreamOutput(token=TokenElement(id=314, text=' be', logprob=-1.5224609, special=False), generated_text=None, details=None)
TextGenerationStreamOutput(token=TokenElement(id=1833, text=' easy', logprob=-2.1132812, special=False), generated_text=None, details=None)
TextGenerationStreamOutput(token=TokenElement(id=271, text=' to', logprob=-0.08520508, special=False), generated_text=None, details=None)
TextGenerationStreamOutput(token=TokenElement(id=745, text=' use', logprob=-0.39453125, special=False), generated_text=None, details=None)
TextGenerationStreamOutput(token=TokenElement(
    id=25,
    text='.',
    logprob=-0.5703125,
    special=False),
    generated_text='100% open source and built to be easy to use.',
    details=TextGenerationStreamOutputStreamDetails(finish_reason='length', generated_tokens=12, seed=None)
)

# Case 5: generate constrained output using grammar
>>> response = client.text_generation(
...     prompt="I saw a puppy a cat and a raccoon during my bike ride in the park",
...     model="HuggingFaceH4/zephyr-orpo-141b-A35b-v0.1",
...     max_new_tokens=100,
...     repetition_penalty=1.3,
...     grammar={
...         "type": "json",
...         "value": {
...             "properties": {
...                 "location": {"type": "string"},
...                 "activity": {"type": "string"},
...                 "animals_seen": {"type": "integer", "minimum": 1, "maximum": 5},
...                 "animals": {"type": "array", "items": {"type": "string"}},
...             },
...             "required": ["location", "activity", "animals_seen", "animals"],
...         },
...     },
... )
>>> json.loads(response)
{
    "activity": "bike riding",
    "animals": ["puppy", "cat", "raccoon"],
    "animals_seen": 3,
    "location": "park"
}
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>text_to_image</name><anchor>huggingface_hub.InferenceClient.text_to_image</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L2429</source><parameters>[{"name": "prompt", "val": ": str"}, {"name": "negative_prompt", "val": ": typing.Optional[str] = None"}, {"name": "height", "val": ": typing.Optional[int] = None"}, {"name": "width", "val": ": typing.Optional[int] = None"}, {"name": "num_inference_steps", "val": ": typing.Optional[int] = None"}, {"name": "guidance_scale", "val": ": typing.Optional[float] = None"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "scheduler", "val": ": typing.Optional[str] = None"}, {"name": "seed", "val": ": typing.Optional[int] = None"}, {"name": "extra_body", "val": ": typing.Optional[dict[str, typing.Any]] = None"}]</parameters><paramsdesc>- **prompt** (`str`) --
  The prompt to generate an image from.
- **negative_prompt** (`str`, *optional*) --
  One prompt to guide what NOT to include in image generation.
- **height** (`int`, *optional*) --
  The height in pixels of the output image
- **width** (`int`, *optional*) --
  The width in pixels of the output image
- **num_inference_steps** (`int`, *optional*) --
  The number of denoising steps. More denoising steps usually lead to a higher quality image at the
  expense of slower inference.
- **guidance_scale** (`float`, *optional*) --
  A higher guidance scale value encourages the model to generate images closely linked to the text
  prompt, but values too high may cause saturation and other artifacts.
- **model** (`str`, *optional*) --
  The model to use for inference. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. If not provided, the default recommended text-to-image model will be used.
  Defaults to None.
- **scheduler** (`str`, *optional*) --
  Override the scheduler with a compatible one.
- **seed** (`int`, *optional*) --
  Seed for the random number generator.
- **extra_body** (`dict[str, Any]`, *optional*) --
  Additional provider-specific parameters to pass to the model. Refer to the provider's documentation
  for supported parameters.</paramsdesc><paramgroups>0</paramgroups><rettype>`Image`</rettype><retdesc>The generated image.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Generate an image based on a given text using a specified model.

> [!WARNING]
> You must have `PIL` installed if you want to work with images (`pip install Pillow`).

> [!TIP]
> You can pass provider-specific parameters to the model by using the `extra_body` argument.











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.text_to_image.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()

>>> image = client.text_to_image("An astronaut riding a horse on the moon.")
>>> image.save("astronaut.png")

>>> image = client.text_to_image(
...     "An astronaut riding a horse on the moon.",
...     negative_prompt="low resolution, blurry",
...     model="stabilityai/stable-diffusion-2-1",
... )
>>> image.save("better_astronaut.png")
```

</ExampleCodeBlock>
Example using a third-party provider directly. Usage will be billed on your fal.ai account.
<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.text_to_image.example-2">

```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient(
...     provider="fal-ai",  # Use fal.ai provider
...     api_key="fal-ai-api-key",  # Pass your fal.ai API key
... )
>>> image = client.text_to_image(
...     "A majestic lion in a fantasy forest",
...     model="black-forest-labs/FLUX.1-schnell",
... )
>>> image.save("lion.png")
```

</ExampleCodeBlock>

Example using a third-party provider through Hugging Face Routing. Usage will be billed on your Hugging Face account.
<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.text_to_image.example-3">

```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient(
...     provider="replicate",  # Use replicate provider
...     api_key="hf_...",  # Pass your HF token
... )
>>> image = client.text_to_image(
...     "An astronaut riding a horse on the moon.",
...     model="black-forest-labs/FLUX.1-dev",
... )
>>> image.save("astronaut.png")
```

</ExampleCodeBlock>

Example using Replicate provider with extra parameters
<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.text_to_image.example-4">

```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient(
...     provider="replicate",  # Use replicate provider
...     api_key="hf_...",  # Pass your HF token
... )
>>> image = client.text_to_image(
...     "An astronaut riding a horse on the moon.",
...     model="black-forest-labs/FLUX.1-schnell",
...     extra_body={"output_quality": 100},
... )
>>> image.save("astronaut.png")
```

</ExampleCodeBlock>



</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>text_to_speech</name><anchor>huggingface_hub.InferenceClient.text_to_speech</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L2666</source><parameters>[{"name": "text", "val": ": str"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "do_sample", "val": ": typing.Optional[bool] = None"}, {"name": "early_stopping", "val": ": typing.Union[bool, ForwardRef('TextToSpeechEarlyStoppingEnum'), NoneType] = None"}, {"name": "epsilon_cutoff", "val": ": typing.Optional[float] = None"}, {"name": "eta_cutoff", "val": ": typing.Optional[float] = None"}, {"name": "max_length", "val": ": typing.Optional[int] = None"}, {"name": "max_new_tokens", "val": ": typing.Optional[int] = None"}, {"name": "min_length", "val": ": typing.Optional[int] = None"}, {"name": "min_new_tokens", "val": ": typing.Optional[int] = None"}, {"name": "num_beam_groups", "val": ": typing.Optional[int] = None"}, {"name": "num_beams", "val": ": typing.Optional[int] = None"}, {"name": "penalty_alpha", "val": ": typing.Optional[float] = None"}, {"name": "temperature", "val": ": typing.Optional[float] = None"}, {"name": "top_k", "val": ": typing.Optional[int] = None"}, {"name": "top_p", "val": ": typing.Optional[float] = None"}, {"name": "typical_p", "val": ": typing.Optional[float] = None"}, {"name": "use_cache", "val": ": typing.Optional[bool] = None"}, {"name": "extra_body", "val": ": typing.Optional[dict[str, typing.Any]] = None"}]</parameters><paramsdesc>- **text** (`str`) --
  The text to synthesize.
- **model** (`str`, *optional*) --
  The model to use for inference. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. If not provided, the default recommended text-to-speech model will be used.
  Defaults to None.
- **do_sample** (`bool`, *optional*) --
  Whether to use sampling instead of greedy decoding when generating new tokens.
- **early_stopping** (`Union[bool, "TextToSpeechEarlyStoppingEnum"]`, *optional*) --
  Controls the stopping condition for beam-based methods.
- **epsilon_cutoff** (`float`, *optional*) --
  If set to float strictly between 0 and 1, only tokens with a conditional probability greater than
  epsilon_cutoff will be sampled. In the paper, suggested values range from 3e-4 to 9e-4, depending on
  the size of the model. See [Truncation Sampling as Language Model
  Desmoothing](https://hf.co/papers/2210.15191) for more details.
- **eta_cutoff** (`float`, *optional*) --
  Eta sampling is a hybrid of locally typical sampling and epsilon sampling. If set to float strictly
  between 0 and 1, a token is only considered if it is greater than either eta_cutoff or sqrt(eta_cutoff)
  * exp(-entropy(softmax(next_token_logits))). The latter term is intuitively the expected next token
  probability, scaled by sqrt(eta_cutoff). In the paper, suggested values range from 3e-4 to 2e-3,
  depending on the size of the model. See [Truncation Sampling as Language Model
  Desmoothing](https://hf.co/papers/2210.15191) for more details.
- **max_length** (`int`, *optional*) --
  The maximum length (in tokens) of the generated text, including the input.
- **max_new_tokens** (`int`, *optional*) --
  The maximum number of tokens to generate. Takes precedence over max_length.
- **min_length** (`int`, *optional*) --
  The minimum length (in tokens) of the generated text, including the input.
- **min_new_tokens** (`int`, *optional*) --
  The minimum number of tokens to generate. Takes precedence over min_length.
- **num_beam_groups** (`int`, *optional*) --
  Number of groups to divide num_beams into in order to ensure diversity among different groups of beams.
  See [this paper](https://hf.co/papers/1610.02424) for more details.
- **num_beams** (`int`, *optional*) --
  Number of beams to use for beam search.
- **penalty_alpha** (`float`, *optional*) --
  The value balances the model confidence and the degeneration penalty in contrastive search decoding.
- **temperature** (`float`, *optional*) --
  The value used to modulate the next token probabilities.
- **top_k** (`int`, *optional*) --
  The number of highest probability vocabulary tokens to keep for top-k-filtering.
- **top_p** (`float`, *optional*) --
  If set to float < 1, only the smallest set of most probable tokens with probabilities that add up to
  top_p or higher are kept for generation.
- **typical_p** (`float`, *optional*) --
  Local typicality measures how similar the conditional probability of predicting a target token next is
  to the expected conditional probability of predicting a random token next, given the partial text
  already generated. If set to float < 1, the smallest set of the most locally typical tokens with
  probabilities that add up to typical_p or higher are kept for generation. See [this
  paper](https://hf.co/papers/2202.00666) for more details.
- **use_cache** (`bool`, *optional*) --
  Whether the model should use the past last key/values attentions to speed up decoding
- **extra_body** (`dict[str, Any]`, *optional*) --
  Additional provider-specific parameters to pass to the model. Refer to the provider's documentation
  for supported parameters.</paramsdesc><paramgroups>0</paramgroups><rettype>`bytes`</rettype><retdesc>The generated audio.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Synthesize an audio of a voice pronouncing a given text.

> [!TIP]
> You can pass provider-specific parameters to the model by using the `extra_body` argument.











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.text_to_speech.example">

Example:
```py
>>> from pathlib import Path
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()

>>> audio = client.text_to_speech("Hello world")
>>> Path("hello_world.flac").write_bytes(audio)
```

</ExampleCodeBlock>

Example using a third-party provider directly. Usage will be billed on your Replicate account.
<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.text_to_speech.example-2">

```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient(
...     provider="replicate",
...     api_key="your-replicate-api-key",  # Pass your Replicate API key directly
... )
>>> audio = client.text_to_speech(
...     text="Hello world",
...     model="OuteAI/OuteTTS-0.3-500M",
... )
>>> Path("hello_world.flac").write_bytes(audio)
```

</ExampleCodeBlock>

Example using a third-party provider through Hugging Face Routing. Usage will be billed on your Hugging Face account.
<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.text_to_speech.example-3">

```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient(
...     provider="replicate",
...     api_key="hf_...",  # Pass your HF token
... )
>>> audio =client.text_to_speech(
...     text="Hello world",
...     model="OuteAI/OuteTTS-0.3-500M",
... )
>>> Path("hello_world.flac").write_bytes(audio)
```

</ExampleCodeBlock>
Example using Replicate provider with extra parameters
<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.text_to_speech.example-4">

```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient(
...     provider="replicate",  # Use replicate provider
...     api_key="hf_...",  # Pass your HF token
... )
>>> audio = client.text_to_speech(
...     "Hello, my name is Kororo, an awesome text-to-speech model.",
...     model="hexgrad/Kokoro-82M",
...     extra_body={"voice": "af_nicole"},
... )
>>> Path("hello.flac").write_bytes(audio)
```

</ExampleCodeBlock>

Example music-gen using "YuE-s1-7B-anneal-en-cot" on fal.ai
<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.text_to_speech.example-5">

```py
>>> from huggingface_hub import InferenceClient
>>> lyrics = '''
... [verse]
... In the town where I was born
... Lived a man who sailed to sea
... And he told us of his life
... In the land of submarines
... So we sailed on to the sun
... 'Til we found a sea of green
... And we lived beneath the waves
... In our yellow submarine

... [chorus]
... We all live in a yellow submarine
... Yellow submarine, yellow submarine
... We all live in a yellow submarine
... Yellow submarine, yellow submarine
... '''
>>> genres = "pavarotti-style tenor voice"
>>> client = InferenceClient(
...     provider="fal-ai",
...     model="m-a-p/YuE-s1-7B-anneal-en-cot",
...     api_key=...,
... )
>>> audio = client.text_to_speech(lyrics, extra_body={"genres": genres})
>>> with open("output.mp3", "wb") as f:
...     f.write(audio)
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>text_to_video</name><anchor>huggingface_hub.InferenceClient.text_to_video</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L2569</source><parameters>[{"name": "prompt", "val": ": str"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "guidance_scale", "val": ": typing.Optional[float] = None"}, {"name": "negative_prompt", "val": ": typing.Optional[list[str]] = None"}, {"name": "num_frames", "val": ": typing.Optional[float] = None"}, {"name": "num_inference_steps", "val": ": typing.Optional[int] = None"}, {"name": "seed", "val": ": typing.Optional[int] = None"}, {"name": "extra_body", "val": ": typing.Optional[dict[str, typing.Any]] = None"}]</parameters><paramsdesc>- **prompt** (`str`) --
  The prompt to generate a video from.
- **model** (`str`, *optional*) --
  The model to use for inference. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. If not provided, the default recommended text-to-video model will be used.
  Defaults to None.
- **guidance_scale** (`float`, *optional*) --
  A higher guidance scale value encourages the model to generate videos closely linked to the text
  prompt, but values too high may cause saturation and other artifacts.
- **negative_prompt** (`list[str]`, *optional*) --
  One or several prompt to guide what NOT to include in video generation.
- **num_frames** (`float`, *optional*) --
  The num_frames parameter determines how many video frames are generated.
- **num_inference_steps** (`int`, *optional*) --
  The number of denoising steps. More denoising steps usually lead to a higher quality video at the
  expense of slower inference.
- **seed** (`int`, *optional*) --
  Seed for the random number generator.
- **extra_body** (`dict[str, Any]`, *optional*) --
  Additional provider-specific parameters to pass to the model. Refer to the provider's documentation
  for supported parameters.</paramsdesc><paramgroups>0</paramgroups><rettype>`bytes`</rettype><retdesc>The generated video.</retdesc></docstring>

Generate a video based on a given text.

> [!TIP]
> You can pass provider-specific parameters to the model by using the `extra_body` argument.







Example:

Example using a third-party provider directly. Usage will be billed on your fal.ai account.
<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.text_to_video.example">

```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient(
...     provider="fal-ai",  # Using fal.ai provider
...     api_key="fal-ai-api-key",  # Pass your fal.ai API key
... )
>>> video = client.text_to_video(
...     "A majestic lion running in a fantasy forest",
...     model="tencent/HunyuanVideo",
... )
>>> with open("lion.mp4", "wb") as file:
...     file.write(video)
```

</ExampleCodeBlock>

Example using a third-party provider through Hugging Face Routing. Usage will be billed on your Hugging Face account.
<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.text_to_video.example-2">

```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient(
...     provider="replicate",  # Using replicate provider
...     api_key="hf_...",  # Pass your HF token
... )
>>> video = client.text_to_video(
...     "A cat running in a park",
...     model="genmo/mochi-1-preview",
... )
>>> with open("cat.mp4", "wb") as file:
...     file.write(video)
```

</ExampleCodeBlock>



</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>token_classification</name><anchor>huggingface_hub.InferenceClient.token_classification</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L2874</source><parameters>[{"name": "text", "val": ": str"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "aggregation_strategy", "val": ": typing.Optional[ForwardRef('TokenClassificationAggregationStrategy')] = None"}, {"name": "ignore_labels", "val": ": typing.Optional[list[str]] = None"}, {"name": "stride", "val": ": typing.Optional[int] = None"}]</parameters><paramsdesc>- **text** (`str`) --
  A string to be classified.
- **model** (`str`, *optional*) --
  The model to use for the token classification task. Can be a model ID hosted on the Hugging Face Hub or a URL to
  a deployed Inference Endpoint. If not provided, the default recommended token classification model will be used.
  Defaults to None.
- **aggregation_strategy** (`"TokenClassificationAggregationStrategy"`, *optional*) --
  The strategy used to fuse tokens based on model predictions
- **ignore_labels** (`list[str`, *optional*) --
  A list of labels to ignore
- **stride** (`int`, *optional*) --
  The number of overlapping tokens between chunks when splitting the input text.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[TokenClassificationOutputElement]`</rettype><retdesc>List of [TokenClassificationOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.TokenClassificationOutputElement) items containing the entity group, confidence score, word, start and end index.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Perform token classification on the given text.
Usually used for sentence parsing, either grammatical, or Named Entity Recognition (NER) to understand keywords contained within text.











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.token_classification.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()
>>> client.token_classification("My name is Sarah Jessica Parker but you can call me Jessica")
[
    TokenClassificationOutputElement(
        entity_group='PER',
        score=0.9971321225166321,
        word='Sarah Jessica Parker',
        start=11,
        end=31,
    ),
    TokenClassificationOutputElement(
        entity_group='PER',
        score=0.9773476123809814,
        word='Jessica',
        start=52,
        end=59,
    )
]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>translation</name><anchor>huggingface_hub.InferenceClient.translation</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L2949</source><parameters>[{"name": "text", "val": ": str"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "src_lang", "val": ": typing.Optional[str] = None"}, {"name": "tgt_lang", "val": ": typing.Optional[str] = None"}, {"name": "clean_up_tokenization_spaces", "val": ": typing.Optional[bool] = None"}, {"name": "truncation", "val": ": typing.Optional[ForwardRef('TranslationTruncationStrategy')] = None"}, {"name": "generate_parameters", "val": ": typing.Optional[dict[str, typing.Any]] = None"}]</parameters><paramsdesc>- **text** (`str`) --
  A string to be translated.
- **model** (`str`, *optional*) --
  The model to use for the translation task. Can be a model ID hosted on the Hugging Face Hub or a URL to
  a deployed Inference Endpoint. If not provided, the default recommended translation model will be used.
  Defaults to None.
- **src_lang** (`str`, *optional*) --
  The source language of the text. Required for models that can translate from multiple languages.
- **tgt_lang** (`str`, *optional*) --
  Target language to translate to. Required for models that can translate to multiple languages.
- **clean_up_tokenization_spaces** (`bool`, *optional*) --
  Whether to clean up the potential extra spaces in the text output.
- **truncation** (`"TranslationTruncationStrategy"`, *optional*) --
  The truncation strategy to use.
- **generate_parameters** (`dict[str, Any]`, *optional*) --
  Additional parametrization of the text generation algorithm.</paramsdesc><paramgroups>0</paramgroups><rettype>[TranslationOutput](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.TranslationOutput)</rettype><retdesc>The generated translated text.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.
- ``ValueError`` -- 
  If only one of the `src_lang` and `tgt_lang` arguments are provided.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError` or ``ValueError``</raisederrors></docstring>

Convert text from one language to another.

Check out https://huggingface.co/tasks/translation for more information on how to choose the best model for
your specific use case. Source and target languages usually depend on the model.
However, it is possible to specify source and target languages for certain models. If you are working with one of these models,
you can use `src_lang` and `tgt_lang` arguments to pass the relevant information.











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.translation.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()
>>> client.translation("My name is Wolfgang and I live in Berlin")
'Mein Name ist Wolfgang und ich lebe in Berlin.'
>>> client.translation("My name is Wolfgang and I live in Berlin", model="Helsinki-NLP/opus-mt-en-fr")
TranslationOutput(translation_text='Je m'appelle Wolfgang et je vis à Berlin.')
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.translation.example-2">

Specifying languages:
```py
>>> client.translation("My name is Sarah Jessica Parker but you can call me Jessica", model="facebook/mbart-large-50-many-to-many-mmt", src_lang="en_XX", tgt_lang="fr_XX")
"Mon nom est Sarah Jessica Parker mais vous pouvez m'appeler Jessica"
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>visual_question_answering</name><anchor>huggingface_hub.InferenceClient.visual_question_answering</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L3038</source><parameters>[{"name": "image", "val": ": typing.Union[bytes, typing.BinaryIO, str, pathlib.Path, ForwardRef('Image'), bytearray, memoryview]"}, {"name": "question", "val": ": str"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "top_k", "val": ": typing.Optional[int] = None"}]</parameters><paramsdesc>- **image** (`Union[str, Path, bytes, BinaryIO, PIL.Image.Image]`) --
  The input image for the context. It can be raw bytes, an image file, a URL to an online image, or a PIL Image.
- **question** (`str`) --
  Question to be answered.
- **model** (`str`, *optional*) --
  The model to use for the visual question answering task. Can be a model ID hosted on the Hugging Face Hub or a URL to
  a deployed Inference Endpoint. If not provided, the default recommended visual question answering model will be used.
  Defaults to None.
- **top_k** (`int`, *optional*) --
  The number of answers to return (will be chosen by order of likelihood). Note that we return less than
  topk answers if there are not enough options available within the context.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[VisualQuestionAnsweringOutputElement]`</rettype><retdesc>a list of [VisualQuestionAnsweringOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.VisualQuestionAnsweringOutputElement) items containing the predicted label and associated probability.</retdesc><raises>- ``InferenceTimeoutError`` -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>``InferenceTimeoutError`` or `HfHubHTTPError`</raisederrors></docstring>

Answering open-ended questions based on an image.











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.visual_question_answering.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()
>>> client.visual_question_answering(
...     image="https://huggingface.co/datasets/mishig/sample_images/resolve/main/tiger.jpg",
...     question="What is the animal doing?"
... )
[
    VisualQuestionAnsweringOutputElement(score=0.778609573841095, answer='laying down'),
    VisualQuestionAnsweringOutputElement(score=0.6957435607910156, answer='sitting'),
]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>zero_shot_classification</name><anchor>huggingface_hub.InferenceClient.zero_shot_classification</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L3097</source><parameters>[{"name": "text", "val": ": str"}, {"name": "candidate_labels", "val": ": list"}, {"name": "multi_label", "val": ": typing.Optional[bool] = False"}, {"name": "hypothesis_template", "val": ": typing.Optional[str] = None"}, {"name": "model", "val": ": typing.Optional[str] = None"}]</parameters><paramsdesc>- **text** (`str`) --
  The input text to classify.
- **candidate_labels** (`list[str]`) --
  The set of possible class labels to classify the text into.
- **labels** (`list[str]`, *optional*) --
  (deprecated) List of strings. Each string is the verbalization of a possible label for the input text.
- **multi_label** (`bool`, *optional*) --
  Whether multiple candidate labels can be true. If false, the scores are normalized such that the sum of
  the label likelihoods for each sequence is 1. If true, the labels are considered independent and
  probabilities are normalized for each candidate.
- **hypothesis_template** (`str`, *optional*) --
  The sentence used in conjunction with `candidate_labels` to attempt the text classification by
  replacing the placeholder with the candidate labels.
- **model** (`str`, *optional*) --
  The model to use for inference. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. This parameter overrides the model defined at the instance level. If not provided, the default recommended zero-shot classification model will be used.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[ZeroShotClassificationOutputElement]`</rettype><retdesc>List of [ZeroShotClassificationOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ZeroShotClassificationOutputElement) items containing the predicted labels and their confidence.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Provide as input a text and a set of candidate labels to classify the input text.











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.zero_shot_classification.example">

Example with `multi_label=False`:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()
>>> text = (
...     "A new model offers an explanation for how the Galilean satellites formed around the solar system's"
...     "largest world. Konstantin Batygin did not set out to solve one of the solar system's most puzzling"
...     " mysteries when he went for a run up a hill in Nice, France."
... )
>>> labels = ["space & cosmos", "scientific discovery", "microbiology", "robots", "archeology"]
>>> client.zero_shot_classification(text, labels)
[
    ZeroShotClassificationOutputElement(label='scientific discovery', score=0.7961668968200684),
    ZeroShotClassificationOutputElement(label='space & cosmos', score=0.18570658564567566),
    ZeroShotClassificationOutputElement(label='microbiology', score=0.00730885099619627),
    ZeroShotClassificationOutputElement(label='archeology', score=0.006258360575884581),
    ZeroShotClassificationOutputElement(label='robots', score=0.004559356719255447),
]
>>> client.zero_shot_classification(text, labels, multi_label=True)
[
    ZeroShotClassificationOutputElement(label='scientific discovery', score=0.9829297661781311),
    ZeroShotClassificationOutputElement(label='space & cosmos', score=0.755190908908844),
    ZeroShotClassificationOutputElement(label='microbiology', score=0.0005462635890580714),
    ZeroShotClassificationOutputElement(label='archeology', score=0.00047131875180639327),
    ZeroShotClassificationOutputElement(label='robots', score=0.00030448526376858354),
]
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.zero_shot_classification.example-2">

Example with `multi_label=True` and a custom `hypothesis_template`:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()
>>> client.zero_shot_classification(
...    text="I really like our dinner and I'm very happy. I don't like the weather though.",
...    labels=["positive", "negative", "pessimistic", "optimistic"],
...    multi_label=True,
...    hypothesis_template="This text is {} towards the weather"
... )
[
    ZeroShotClassificationOutputElement(label='negative', score=0.9231801629066467),
    ZeroShotClassificationOutputElement(label='pessimistic', score=0.8760990500450134),
    ZeroShotClassificationOutputElement(label='optimistic', score=0.0008674879791215062),
    ZeroShotClassificationOutputElement(label='positive', score=0.0005250611575320363)
]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>zero_shot_image_classification</name><anchor>huggingface_hub.InferenceClient.zero_shot_image_classification</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_client.py#L3203</source><parameters>[{"name": "image", "val": ": typing.Union[bytes, typing.BinaryIO, str, pathlib.Path, ForwardRef('Image'), bytearray, memoryview]"}, {"name": "candidate_labels", "val": ": list"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "hypothesis_template", "val": ": typing.Optional[str] = None"}, {"name": "labels", "val": ": list = None"}]</parameters><paramsdesc>- **image** (`Union[str, Path, bytes, BinaryIO, PIL.Image.Image]`) --
  The input image to caption. It can be raw bytes, an image file, a URL to an online image, or a PIL Image.
- **candidate_labels** (`list[str]`) --
  The candidate labels for this image
- **labels** (`list[str]`, *optional*) --
  (deprecated) List of string possible labels. There must be at least 2 labels.
- **model** (`str`, *optional*) --
  The model to use for inference. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. This parameter overrides the model defined at the instance level. If not provided, the default recommended zero-shot image classification model will be used.
- **hypothesis_template** (`str`, *optional*) --
  The sentence used in conjunction with `candidate_labels` to attempt the image classification by
  replacing the placeholder with the candidate labels.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[ZeroShotImageClassificationOutputElement]`</rettype><retdesc>List of [ZeroShotImageClassificationOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ZeroShotImageClassificationOutputElement) items containing the predicted labels and their confidence.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Provide input image and text labels to predict text labels for the image.











<ExampleCodeBlock anchor="huggingface_hub.InferenceClient.zero_shot_image_classification.example">

Example:
```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient()

>>> client.zero_shot_image_classification(
...     "https://upload.wikimedia.org/wikipedia/commons/thumb/4/43/Cute_dog.jpg/320px-Cute_dog.jpg",
...     labels=["dog", "cat", "horse"],
... )
[ZeroShotImageClassificationOutputElement(label='dog', score=0.956),...]
```

</ExampleCodeBlock>


</div></div>

## Async Inference Client[[huggingface_hub.AsyncInferenceClient]]

An async version of the client is also provided, based on `asyncio` and `httpx`.

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.AsyncInferenceClient</name><anchor>huggingface_hub.AsyncInferenceClient</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L114</source><parameters>[{"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "provider", "val": ": typing.Union[typing.Literal['black-forest-labs', 'cerebras', 'clarifai', 'cohere', 'fal-ai', 'featherless-ai', 'fireworks-ai', 'groq', 'hf-inference', 'hyperbolic', 'nebius', 'novita', 'nscale', 'openai', 'publicai', 'replicate', 'sambanova', 'scaleway', 'together', 'wavespeed', 'zai-org'], typing.Literal['auto'], NoneType] = None"}, {"name": "token", "val": ": typing.Optional[str] = None"}, {"name": "timeout", "val": ": typing.Optional[float] = None"}, {"name": "headers", "val": ": typing.Optional[dict[str, str]] = None"}, {"name": "cookies", "val": ": typing.Optional[dict[str, str]] = None"}, {"name": "bill_to", "val": ": typing.Optional[str] = None"}, {"name": "base_url", "val": ": typing.Optional[str] = None"}, {"name": "api_key", "val": ": typing.Optional[str] = None"}]</parameters><paramsdesc>- **model** (`str`, `optional`) --
  The model to run inference with. Can be a model id hosted on the Hugging Face Hub, e.g. `meta-llama/Meta-Llama-3-8B-Instruct`
  or a URL to a deployed Inference Endpoint. Defaults to None, in which case a recommended model is
  automatically selected for the task.
  Note: for better compatibility with OpenAI's client, `model` has been aliased as `base_url`. Those 2
  arguments are mutually exclusive. If a URL is passed as `model` or `base_url` for chat completion, the `(/v1)/chat/completions` suffix path will be appended to the URL.
- **provider** (`str`, *optional*) --
  Name of the provider to use for inference. Can be `"black-forest-labs"`, `"cerebras"`, `"clarifai"`, `"cohere"`, `"fal-ai"`, `"featherless-ai"`, `"fireworks-ai"`, `"groq"`, `"hf-inference"`, `"hyperbolic"`, `"nebius"`, `"novita"`, `"nscale"`, `"openai"`, `"publicai"`, `"replicate"`, `"sambanova"`, `"scaleway"`, `"together"`, `"wavespeed"` or `"zai-org"`.
  Defaults to "auto" i.e. the first of the providers available for the model, sorted by the user's order in https://hf.co/settings/inference-providers.
  If model is a URL or `base_url` is passed, then `provider` is not used.
- **token** (`str`, *optional*) --
  Hugging Face token. Will default to the locally saved token if not provided.
  Note: for better compatibility with OpenAI's client, `token` has been aliased as `api_key`. Those 2
  arguments are mutually exclusive and have the exact same behavior.
- **timeout** (`float`, `optional`) --
  The maximum number of seconds to wait for a response from the server. Defaults to None, meaning it will loop until the server is available.
- **headers** (`dict[str, str]`, `optional`) --
  Additional headers to send to the server. By default only the authorization and user-agent headers are sent.
  Values in this dictionary will override the default values.
- **bill_to** (`str`, `optional`) --
  The billing account to use for the requests. By default the requests are billed on the user's account.
  Requests can only be billed to an organization the user is a member of, and which has subscribed to Enterprise Hub.
- **cookies** (`dict[str, str]`, `optional`) --
  Additional cookies to send to the server.
- **base_url** (`str`, `optional`) --
  Base URL to run inference. This is a duplicated argument from `model` to make [InferenceClient](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceClient)
  follow the same pattern as `openai.OpenAI` client. Cannot be used if `model` is set. Defaults to None.
- **api_key** (`str`, `optional`) --
  Token to use for authentication. This is a duplicated argument from `token` to make [InferenceClient](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceClient)
  follow the same pattern as `openai.OpenAI` client. Cannot be used if `token` is set. Defaults to None.</paramsdesc><paramgroups>0</paramgroups></docstring>

Initialize a new Inference Client.

[InferenceClient](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceClient) aims to provide a unified experience to perform inference. The client can be used
seamlessly with either the (free) Inference API, self-hosted Inference Endpoints, or third-party Inference Providers.





<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>audio_classification</name><anchor>huggingface_hub.AsyncInferenceClient.audio_classification</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L317</source><parameters>[{"name": "audio", "val": ": typing.Union[bytes, typing.BinaryIO, str, pathlib.Path, ForwardRef('Image'), bytearray, memoryview]"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "top_k", "val": ": typing.Optional[int] = None"}, {"name": "function_to_apply", "val": ": typing.Optional[ForwardRef('AudioClassificationOutputTransform')] = None"}]</parameters><paramsdesc>- **audio** (Union[str, Path, bytes, BinaryIO]) --
  The audio content to classify. It can be raw audio bytes, a local audio file, or a URL pointing to an
  audio file.
- **model** (`str`, *optional*) --
  The model to use for audio classification. Can be a model ID hosted on the Hugging Face Hub
  or a URL to a deployed Inference Endpoint. If not provided, the default recommended model for
  audio classification will be used.
- **top_k** (`int`, *optional*) --
  When specified, limits the output to the top K most probable classes.
- **function_to_apply** (`"AudioClassificationOutputTransform"`, *optional*) --
  The function to apply to the model outputs in order to retrieve the scores.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[AudioClassificationOutputElement]`</rettype><retdesc>List of [AudioClassificationOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.AudioClassificationOutputElement) items containing the predicted labels and their confidence.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Perform audio classification on the provided audio content.











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.audio_classification.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()
>>> await client.audio_classification("audio.flac")
[
    AudioClassificationOutputElement(score=0.4976358711719513, label='hap'),
    AudioClassificationOutputElement(score=0.3677836060523987, label='neu'),
    ...
]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>audio_to_audio</name><anchor>huggingface_hub.AsyncInferenceClient.audio_to_audio</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L375</source><parameters>[{"name": "audio", "val": ": typing.Union[bytes, typing.BinaryIO, str, pathlib.Path, ForwardRef('Image'), bytearray, memoryview]"}, {"name": "model", "val": ": typing.Optional[str] = None"}]</parameters><paramsdesc>- **audio** (Union[str, Path, bytes, BinaryIO]) --
  The audio content for the model. It can be raw audio bytes, a local audio file, or a URL pointing to an
  audio file.
- **model** (`str`, *optional*) --
  The model can be any model which takes an audio file and returns another audio file. Can be a model ID hosted on the Hugging Face Hub
  or a URL to a deployed Inference Endpoint. If not provided, the default recommended model for
  audio_to_audio will be used.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[AudioToAudioOutputElement]`</rettype><retdesc>A list of [AudioToAudioOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.AudioToAudioOutputElement) items containing audios label, content-type, and audio content in blob.</retdesc><raises>- ``InferenceTimeoutError`` -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>``InferenceTimeoutError`` or `HfHubHTTPError`</raisederrors></docstring>

Performs multiple tasks related to audio-to-audio depending on the model (eg: speech enhancement, source separation).











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.audio_to_audio.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()
>>> audio_output = await client.audio_to_audio("audio.flac")
>>> async for i, item in enumerate(audio_output):
>>>     with open(f"output_{i}.flac", "wb") as f:
            f.write(item.blob)
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>automatic_speech_recognition</name><anchor>huggingface_hub.AsyncInferenceClient.automatic_speech_recognition</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L428</source><parameters>[{"name": "audio", "val": ": typing.Union[bytes, typing.BinaryIO, str, pathlib.Path, ForwardRef('Image'), bytearray, memoryview]"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "extra_body", "val": ": typing.Optional[dict] = None"}]</parameters><paramsdesc>- **audio** (Union[str, Path, bytes, BinaryIO]) --
  The content to transcribe. It can be raw audio bytes, local audio file, or a URL to an audio file.
- **model** (`str`, *optional*) --
  The model to use for ASR. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. If not provided, the default recommended model for ASR will be used.
- **extra_body** (`dict`, *optional*) --
  Additional provider-specific parameters to pass to the model. Refer to the provider's documentation
  for supported parameters.</paramsdesc><paramgroups>0</paramgroups><rettype>[AutomaticSpeechRecognitionOutput](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.AutomaticSpeechRecognitionOutput)</rettype><retdesc>An item containing the transcribed text and optionally the timestamp chunks.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Perform automatic speech recognition (ASR or audio-to-text) on the given audio content.











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.automatic_speech_recognition.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()
>>> await client.automatic_speech_recognition("hello_world.flac").text
"hello world"
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>chat_completion</name><anchor>huggingface_hub.AsyncInferenceClient.chat_completion</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L555</source><parameters>[{"name": "messages", "val": ": list"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "stream", "val": ": bool = False"}, {"name": "frequency_penalty", "val": ": typing.Optional[float] = None"}, {"name": "logit_bias", "val": ": typing.Optional[list[float]] = None"}, {"name": "logprobs", "val": ": typing.Optional[bool] = None"}, {"name": "max_tokens", "val": ": typing.Optional[int] = None"}, {"name": "n", "val": ": typing.Optional[int] = None"}, {"name": "presence_penalty", "val": ": typing.Optional[float] = None"}, {"name": "response_format", "val": ": typing.Union[huggingface_hub.inference._generated.types.chat_completion.ChatCompletionInputResponseFormatText, huggingface_hub.inference._generated.types.chat_completion.ChatCompletionInputResponseFormatJSONSchema, huggingface_hub.inference._generated.types.chat_completion.ChatCompletionInputResponseFormatJSONObject, NoneType] = None"}, {"name": "seed", "val": ": typing.Optional[int] = None"}, {"name": "stop", "val": ": typing.Optional[list[str]] = None"}, {"name": "stream_options", "val": ": typing.Optional[huggingface_hub.inference._generated.types.chat_completion.ChatCompletionInputStreamOptions] = None"}, {"name": "temperature", "val": ": typing.Optional[float] = None"}, {"name": "tool_choice", "val": ": typing.Union[huggingface_hub.inference._generated.types.chat_completion.ChatCompletionInputToolChoiceClass, ForwardRef('ChatCompletionInputToolChoiceEnum'), NoneType] = None"}, {"name": "tool_prompt", "val": ": typing.Optional[str] = None"}, {"name": "tools", "val": ": typing.Optional[list[huggingface_hub.inference._generated.types.chat_completion.ChatCompletionInputTool]] = None"}, {"name": "top_logprobs", "val": ": typing.Optional[int] = None"}, {"name": "top_p", "val": ": typing.Optional[float] = None"}, {"name": "extra_body", "val": ": typing.Optional[dict] = None"}]</parameters><paramsdesc>- **messages** (List of [ChatCompletionInputMessage](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ChatCompletionInputMessage)) --
  Conversation history consisting of roles and content pairs.
- **model** (`str`, *optional*) --
  The model to use for chat-completion. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. If not provided, the default recommended model for chat-based text-generation will be used.
  See https://huggingface.co/tasks/text-generation for more details.
  If `model` is a model ID, it is passed to the server as the `model` parameter. If you want to define a
  custom URL while setting `model` in the request payload, you must set `base_url` when initializing [InferenceClient](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceClient).
- **frequency_penalty** (`float`, *optional*) --
  Penalizes new tokens based on their existing frequency
  in the text so far. Range: [-2.0, 2.0]. Defaults to 0.0.
- **logit_bias** (`list[float]`, *optional*) --
  Adjusts the likelihood of specific tokens appearing in the generated output.
- **logprobs** (`bool`, *optional*) --
  Whether to return log probabilities of the output tokens or not. If true, returns the log
  probabilities of each output token returned in the content of message.
- **max_tokens** (`int`, *optional*) --
  Maximum number of tokens allowed in the response. Defaults to 100.
- **n** (`int`, *optional*) --
  The number of completions to generate for each prompt.
- **presence_penalty** (`float`, *optional*) --
  Number between -2.0 and 2.0. Positive values penalize new tokens based on whether they appear in the
  text so far, increasing the model's likelihood to talk about new topics.
- **response_format** (`ChatCompletionInputGrammarType()`, *optional*) --
  Grammar constraints. Can be either a JSONSchema or a regex.
- **seed** (Optional`int`, *optional*) --
  Seed for reproducible control flow. Defaults to None.
- **stop** (`list[str]`, *optional*) --
  Up to four strings which trigger the end of the response.
  Defaults to None.
- **stream** (`bool`, *optional*) --
  Enable realtime streaming of responses. Defaults to False.
- **stream_options** ([ChatCompletionInputStreamOptions](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ChatCompletionInputStreamOptions), *optional*) --
  Options for streaming completions.
- **temperature** (`float`, *optional*) --
  Controls randomness of the generations. Lower values ensure
  less random completions. Range: [0, 2]. Defaults to 1.0.
- **top_logprobs** (`int`, *optional*) --
  An integer between 0 and 5 specifying the number of most likely tokens to return at each token
  position, each with an associated log probability. logprobs must be set to true if this parameter is
  used.
- **top_p** (`float`, *optional*) --
  Fraction of the most likely next words to sample from.
  Must be between 0 and 1. Defaults to 1.0.
- **tool_choice** ([ChatCompletionInputToolChoiceClass](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ChatCompletionInputToolChoiceClass) or `ChatCompletionInputToolChoiceEnum()`, *optional*) --
  The tool to use for the completion. Defaults to "auto".
- **tool_prompt** (`str`, *optional*) --
  A prompt to be appended before the tools.
- **tools** (List of [ChatCompletionInputTool](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ChatCompletionInputTool), *optional*) --
  A list of tools the model may call. Currently, only functions are supported as a tool. Use this to
  provide a list of functions the model may generate JSON inputs for.
- **extra_body** (`dict`, *optional*) --
  Additional provider-specific parameters to pass to the model. Refer to the provider's documentation
  for supported parameters.</paramsdesc><paramgroups>0</paramgroups><rettype>[ChatCompletionOutput](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ChatCompletionOutput) or Iterable of [ChatCompletionStreamOutput](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ChatCompletionStreamOutput)</rettype><retdesc>Generated text returned from the server:
- if `stream=False`, the generated text is returned as a [ChatCompletionOutput](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ChatCompletionOutput) (default).
- if `stream=True`, the generated text is returned token by token as a sequence of [ChatCompletionStreamOutput](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ChatCompletionStreamOutput).</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

A method for completing conversations using a specified language model.

> [!TIP]
> The `client.chat_completion` method is aliased as `client.chat.completions.create` for compatibility with OpenAI's client.
> Inputs and outputs are strictly the same and using either syntax will yield the same results.
> Check out the [Inference guide](https://huggingface.co/docs/huggingface_hub/guides/inference#openai-compatibility)
> for more details about OpenAI's compatibility.

> [!TIP]
> You can pass provider-specific parameters to the model by using the `extra_body` argument.











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.chat_completion.example">

Example:

```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> messages = [{"role": "user", "content": "What is the capital of France?"}]
>>> client = AsyncInferenceClient("meta-llama/Meta-Llama-3-8B-Instruct")
>>> await client.chat_completion(messages, max_tokens=100)
ChatCompletionOutput(
    choices=[
        ChatCompletionOutputComplete(
            finish_reason='eos_token',
            index=0,
            message=ChatCompletionOutputMessage(
                role='assistant',
                content='The capital of France is Paris.',
                name=None,
                tool_calls=None
            ),
            logprobs=None
        )
    ],
    created=1719907176,
    id='',
    model='meta-llama/Meta-Llama-3-8B-Instruct',
    object='text_completion',
    system_fingerprint='2.0.4-sha-f426a33',
    usage=ChatCompletionOutputUsage(
        completion_tokens=8,
        prompt_tokens=17,
        total_tokens=25
    )
)
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.chat_completion.example-2">

Example using streaming:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> messages = [{"role": "user", "content": "What is the capital of France?"}]
>>> client = AsyncInferenceClient("meta-llama/Meta-Llama-3-8B-Instruct")
>>> async for token in await client.chat_completion(messages, max_tokens=10, stream=True):
...     print(token)
ChatCompletionStreamOutput(choices=[ChatCompletionStreamOutputChoice(delta=ChatCompletionStreamOutputDelta(content='The', role='assistant'), index=0, finish_reason=None)], created=1710498504)
ChatCompletionStreamOutput(choices=[ChatCompletionStreamOutputChoice(delta=ChatCompletionStreamOutputDelta(content=' capital', role='assistant'), index=0, finish_reason=None)], created=1710498504)
(...)
ChatCompletionStreamOutput(choices=[ChatCompletionStreamOutputChoice(delta=ChatCompletionStreamOutputDelta(content=' may', role='assistant'), index=0, finish_reason=None)], created=1710498504)
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.chat_completion.example-3">

Example using OpenAI's syntax:
```py
# Must be run in an async context
# instead of `from openai import OpenAI`
from huggingface_hub import AsyncInferenceClient

# instead of `client = OpenAI(...)`
client = AsyncInferenceClient(
    base_url=...,
    api_key=...,
)

output = await client.chat.completions.create(
    model="meta-llama/Meta-Llama-3-8B-Instruct",
    messages=[
        {"role": "system", "content": "You are a helpful assistant."},
        {"role": "user", "content": "Count to 10"},
    ],
    stream=True,
    max_tokens=1024,
)

for chunk in output:
    print(chunk.choices[0].delta.content)
```

</ExampleCodeBlock>

Example using a third-party provider directly with extra (provider-specific) parameters. Usage will be billed on your Together AI account.
<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.chat_completion.example-4">

```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient(
...     provider="together",  # Use Together AI provider
...     api_key="<together_api_key>",  # Pass your Together API key directly
... )
>>> client.chat_completion(
...     model="meta-llama/Meta-Llama-3-8B-Instruct",
...     messages=[{"role": "user", "content": "What is the capital of France?"}],
...     extra_body={"safety_model": "Meta-Llama/Llama-Guard-7b"},
... )
```

</ExampleCodeBlock>

Example using a third-party provider through Hugging Face Routing. Usage will be billed on your Hugging Face account.
<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.chat_completion.example-5">

```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient(
...     provider="sambanova",  # Use Sambanova provider
...     api_key="hf_...",  # Pass your HF token
... )
>>> client.chat_completion(
...     model="meta-llama/Meta-Llama-3-8B-Instruct",
...     messages=[{"role": "user", "content": "What is the capital of France?"}],
... )
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.chat_completion.example-6">

Example using Image + Text as input:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient

# provide a remote URL
>>> image_url ="https://cdn.britannica.com/61/93061-050-99147DCE/Statue-of-Liberty-Island-New-York-Bay.jpg"
# or a base64-encoded image
>>> image_path = "/path/to/image.jpeg"
>>> with open(image_path, "rb") as f:
...     base64_image = base64.b64encode(f.read()).decode("utf-8")
>>> image_url = f"data:image/jpeg;base64,{base64_image}"

>>> client = AsyncInferenceClient("meta-llama/Llama-3.2-11B-Vision-Instruct")
>>> output = await client.chat.completions.create(
...     messages=[
...         {
...             "role": "user",
...             "content": [
...                 {
...                     "type": "image_url",
...                     "image_url": {"url": image_url},
...                 },
...                 {
...                     "type": "text",
...                     "text": "Describe this image in one sentence.",
...                 },
...             ],
...         },
...     ],
... )
>>> output
The image depicts the iconic Statue of Liberty situated in New York Harbor, New York, on a clear day.
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.chat_completion.example-7">

Example using tools:
```py
# Must be run in an async context
>>> client = AsyncInferenceClient("meta-llama/Meta-Llama-3-70B-Instruct")
>>> messages = [
...     {
...         "role": "system",
...         "content": "Don't make assumptions about what values to plug into functions. Ask for clarification if a user request is ambiguous.",
...     },
...     {
...         "role": "user",
...         "content": "What's the weather like the next 3 days in San Francisco, CA?",
...     },
... ]
>>> tools = [
...     {
...         "type": "function",
...         "function": {
...             "name": "get_current_weather",
...             "description": "Get the current weather",
...             "parameters": {
...                 "type": "object",
...                 "properties": {
...                     "location": {
...                         "type": "string",
...                         "description": "The city and state, e.g. San Francisco, CA",
...                     },
...                     "format": {
...                         "type": "string",
...                         "enum": ["celsius", "fahrenheit"],
...                         "description": "The temperature unit to use. Infer this from the users location.",
...                     },
...                 },
...                 "required": ["location", "format"],
...             },
...         },
...     },
...     {
...         "type": "function",
...         "function": {
...             "name": "get_n_day_weather_forecast",
...             "description": "Get an N-day weather forecast",
...             "parameters": {
...                 "type": "object",
...                 "properties": {
...                     "location": {
...                         "type": "string",
...                         "description": "The city and state, e.g. San Francisco, CA",
...                     },
...                     "format": {
...                         "type": "string",
...                         "enum": ["celsius", "fahrenheit"],
...                         "description": "The temperature unit to use. Infer this from the users location.",
...                     },
...                     "num_days": {
...                         "type": "integer",
...                         "description": "The number of days to forecast",
...                     },
...                 },
...                 "required": ["location", "format", "num_days"],
...             },
...         },
...     },
... ]

>>> response = await client.chat_completion(
...     model="meta-llama/Meta-Llama-3-70B-Instruct",
...     messages=messages,
...     tools=tools,
...     tool_choice="auto",
...     max_tokens=500,
... )
>>> response.choices[0].message.tool_calls[0].function
ChatCompletionOutputFunctionDefinition(
    arguments={
        'location': 'San Francisco, CA',
        'format': 'fahrenheit',
        'num_days': 3
    },
    name='get_n_day_weather_forecast',
    description=None
)
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.chat_completion.example-8">

Example using response_format:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient("meta-llama/Meta-Llama-3-70B-Instruct")
>>> messages = [
...     {
...         "role": "user",
...         "content": "I saw a puppy a cat and a raccoon during my bike ride in the park. What did I see and when?",
...     },
... ]
>>> response_format = {
...     "type": "json",
...     "value": {
...         "properties": {
...             "location": {"type": "string"},
...             "activity": {"type": "string"},
...             "animals_seen": {"type": "integer", "minimum": 1, "maximum": 5},
...             "animals": {"type": "array", "items": {"type": "string"}},
...         },
...         "required": ["location", "activity", "animals_seen", "animals"],
...     },
... }
>>> response = await client.chat_completion(
...     messages=messages,
...     response_format=response_format,
...     max_tokens=500,
... )
>>> response.choices[0].message.content
'{

y": "bike ride",
": ["puppy", "cat", "raccoon"],
_seen": 3,
n": "park"}'
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>close</name><anchor>huggingface_hub.AsyncInferenceClient.close</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L238</source><parameters>[]</parameters></docstring>
Close the client.

This method is automatically called when using the client as a context manager.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>document_question_answering</name><anchor>huggingface_hub.AsyncInferenceClient.document_question_answering</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L963</source><parameters>[{"name": "image", "val": ": typing.Union[bytes, typing.BinaryIO, str, pathlib.Path, ForwardRef('Image'), bytearray, memoryview]"}, {"name": "question", "val": ": str"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "doc_stride", "val": ": typing.Optional[int] = None"}, {"name": "handle_impossible_answer", "val": ": typing.Optional[bool] = None"}, {"name": "lang", "val": ": typing.Optional[str] = None"}, {"name": "max_answer_len", "val": ": typing.Optional[int] = None"}, {"name": "max_question_len", "val": ": typing.Optional[int] = None"}, {"name": "max_seq_len", "val": ": typing.Optional[int] = None"}, {"name": "top_k", "val": ": typing.Optional[int] = None"}, {"name": "word_boxes", "val": ": typing.Optional[list[typing.Union[list[float], str]]] = None"}]</parameters><paramsdesc>- **image** (`Union[str, Path, bytes, BinaryIO]`) --
  The input image for the context. It can be raw bytes, an image file, or a URL to an online image.
- **question** (`str`) --
  Question to be answered.
- **model** (`str`, *optional*) --
  The model to use for the document question answering task. Can be a model ID hosted on the Hugging Face Hub or a URL to
  a deployed Inference Endpoint. If not provided, the default recommended document question answering model will be used.
  Defaults to None.
- **doc_stride** (`int`, *optional*) --
  If the words in the document are too long to fit with the question for the model, it will be split in
  several chunks with some overlap. This argument controls the size of that overlap.
- **handle_impossible_answer** (`bool`, *optional*) --
  Whether to accept impossible as an answer
- **lang** (`str`, *optional*) --
  Language to use while running OCR. Defaults to english.
- **max_answer_len** (`int`, *optional*) --
  The maximum length of predicted answers (e.g., only answers with a shorter length are considered).
- **max_question_len** (`int`, *optional*) --
  The maximum length of the question after tokenization. It will be truncated if needed.
- **max_seq_len** (`int`, *optional*) --
  The maximum length of the total sentence (context + question) in tokens of each chunk passed to the
  model. The context will be split in several chunks (using doc_stride as overlap) if needed.
- **top_k** (`int`, *optional*) --
  The number of answers to return (will be chosen by order of likelihood). Can return less than top_k
  answers if there are not enough options available within the context.
- **word_boxes** (`list[Union[list[float], str`, *optional*) --
  A list of words and bounding boxes (normalized 0->1000). If provided, the inference will skip the OCR
  step and use the provided bounding boxes instead.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[DocumentQuestionAnsweringOutputElement]`</rettype><retdesc>a list of [DocumentQuestionAnsweringOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.DocumentQuestionAnsweringOutputElement) items containing the predicted label, associated probability, word ids, and page number.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Answer questions on document images.











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.document_question_answering.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()
>>> await client.document_question_answering(image="https://huggingface.co/spaces/impira/docquery/resolve/2359223c1837a7587402bda0f2643382a6eefeab/invoice.png", question="What is the invoice number?")
[DocumentQuestionAnsweringOutputElement(answer='us-001', end=16, score=0.9999666213989258, start=16)]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>feature_extraction</name><anchor>huggingface_hub.AsyncInferenceClient.feature_extraction</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L1051</source><parameters>[{"name": "text", "val": ": str"}, {"name": "normalize", "val": ": typing.Optional[bool] = None"}, {"name": "prompt_name", "val": ": typing.Optional[str] = None"}, {"name": "truncate", "val": ": typing.Optional[bool] = None"}, {"name": "truncation_direction", "val": ": typing.Optional[typing.Literal['Left', 'Right']] = None"}, {"name": "model", "val": ": typing.Optional[str] = None"}]</parameters><paramsdesc>- **text** (*str*) --
  The text to embed.
- **model** (*str*, *optional*) --
  The model to use for the feature extraction task. Can be a model ID hosted on the Hugging Face Hub or a URL to
  a deployed Inference Endpoint. If not provided, the default recommended feature extraction model will be used.
  Defaults to None.
- **normalize** (*bool*, *optional*) --
  Whether to normalize the embeddings or not.
  Only available on server powered by Text-Embedding-Inference.
- **prompt_name** (*str*, *optional*) --
  The name of the prompt that should be used by for encoding. If not set, no prompt will be applied.
  Must be a key in the *Sentence Transformers* configuration *prompts* dictionary.
  For example if `prompt_name` is "query" and the `prompts` is &amp;lcub;"query": "query: ",...},
  then the sentence "What is the capital of France?" will be encoded as "query: What is the capital of France?"
  because the prompt text will be prepended before any text to encode.
- **truncate** (*bool*, *optional*) --
  Whether to truncate the embeddings or not.
  Only available on server powered by Text-Embedding-Inference.
- **truncation_direction** (*Literal["Left", "Right"]*, *optional*) --
  Which side of the input should be truncated when *truncate=True* is passed.</paramsdesc><paramgroups>0</paramgroups><rettype>*np.ndarray*</rettype><retdesc>The embedding representing the input text as a float32 numpy array.</retdesc><raises>- [*InferenceTimeoutError*] -- 
  If the model is unavailable or the request times out.
- [*HfHubHTTPError*] -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[*InferenceTimeoutError*] or [*HfHubHTTPError*]</raisederrors></docstring>

Generate embeddings for a given text.











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.feature_extraction.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()
>>> await client.feature_extraction("Hi, who are you?")
array([[ 2.424802  ,  2.93384   ,  1.1750331 , ...,  1.240499, -0.13776633, -0.7889173 ],
[-0.42943227, -0.6364878 , -1.693462  , ...,  0.41978157, -2.4336355 ,  0.6162071 ],
...,
[ 0.28552425, -0.928395  , -1.2077185 , ...,  0.76810825, -2.1069427 ,  0.6236161 ]], dtype=float32)
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>fill_mask</name><anchor>huggingface_hub.AsyncInferenceClient.fill_mask</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L1125</source><parameters>[{"name": "text", "val": ": str"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "targets", "val": ": typing.Optional[list[str]] = None"}, {"name": "top_k", "val": ": typing.Optional[int] = None"}]</parameters><paramsdesc>- **text** (`str`) --
  a string to be filled from, must contain the [MASK] token (check model card for exact name of the mask).
- **model** (`str`, *optional*) --
  The model to use for the fill mask task. Can be a model ID hosted on the Hugging Face Hub or a URL to
  a deployed Inference Endpoint. If not provided, the default recommended fill mask model will be used.
- **targets** (`list[str`, *optional*) --
  When passed, the model will limit the scores to the passed targets instead of looking up in the whole
  vocabulary. If the provided targets are not in the model vocab, they will be tokenized and the first
  resulting token will be used (with a warning, and that might be slower).
- **top_k** (`int`, *optional*) --
  When passed, overrides the number of predictions to return.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[FillMaskOutputElement]`</rettype><retdesc>a list of [FillMaskOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.FillMaskOutputElement) items containing the predicted label, associated
probability, token reference, and completed text.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Fill in a hole with a missing word (token to be precise).











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.fill_mask.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()
>>> await client.fill_mask("The goal of life is <mask>.")
[
    FillMaskOutputElement(score=0.06897063553333282, token=11098, token_str=' happiness', sequence='The goal of life is happiness.'),
    FillMaskOutputElement(score=0.06554922461509705, token=45075, token_str=' immortality', sequence='The goal of life is immortality.')
]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_endpoint_info</name><anchor>huggingface_hub.AsyncInferenceClient.get_endpoint_info</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L3321</source><parameters>[{"name": "model", "val": ": typing.Optional[str] = None"}]</parameters><paramsdesc>- **model** (`str`, *optional*) --
  The model to use for inference. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. This parameter overrides the model defined at the instance level. Defaults to None.</paramsdesc><paramgroups>0</paramgroups><rettype>`dict[str, Any]`</rettype><retdesc>Information about the endpoint.</retdesc></docstring>

Get information about the deployed endpoint.

This endpoint is only available on endpoints powered by Text-Generation-Inference (TGI) or Text-Embedding-Inference (TEI).
Endpoints powered by `transformers` return an empty payload.







<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.get_endpoint_info.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient("meta-llama/Meta-Llama-3-70B-Instruct")
>>> await client.get_endpoint_info()
{
    'model_id': 'meta-llama/Meta-Llama-3-70B-Instruct',
    'model_sha': None,
    'model_dtype': 'torch.float16',
    'model_device_type': 'cuda',
    'model_pipeline_tag': None,
    'max_concurrent_requests': 128,
    'max_best_of': 2,
    'max_stop_sequences': 4,
    'max_input_length': 8191,
    'max_total_tokens': 8192,
    'waiting_served_ratio': 0.3,
    'max_batch_total_tokens': 1259392,
    'max_waiting_tokens': 20,
    'max_batch_size': None,
    'validation_workers': 32,
    'max_client_batch_size': 4,
    'version': '2.0.2',
    'sha': 'dccab72549635c7eb5ddb17f43f0b7cdff07c214',
    'docker_label': 'sha-dccab72'
}
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>health_check</name><anchor>huggingface_hub.AsyncInferenceClient.health_check</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L3381</source><parameters>[{"name": "model", "val": ": typing.Optional[str] = None"}]</parameters><paramsdesc>- **model** (`str`, *optional*) --
  URL of the Inference Endpoint. This parameter overrides the model defined at the instance level. Defaults to None.</paramsdesc><paramgroups>0</paramgroups><rettype>`bool`</rettype><retdesc>True if everything is working fine.</retdesc></docstring>

Check the health of the deployed endpoint.

Health check is only available with Inference Endpoints powered by Text-Generation-Inference (TGI) or Text-Embedding-Inference (TEI).







<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.health_check.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient("https://jzgu0buei5.us-east-1.aws.endpoints.huggingface.cloud")
>>> await client.health_check()
True
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>image_classification</name><anchor>huggingface_hub.AsyncInferenceClient.image_classification</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L1182</source><parameters>[{"name": "image", "val": ": typing.Union[bytes, typing.BinaryIO, str, pathlib.Path, ForwardRef('Image'), bytearray, memoryview]"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "function_to_apply", "val": ": typing.Optional[ForwardRef('ImageClassificationOutputTransform')] = None"}, {"name": "top_k", "val": ": typing.Optional[int] = None"}]</parameters><paramsdesc>- **image** (`Union[str, Path, bytes, BinaryIO, PIL.Image.Image]`) --
  The image to classify. It can be raw bytes, an image file, a URL to an online image, or a PIL Image.
- **model** (`str`, *optional*) --
  The model to use for image classification. Can be a model ID hosted on the Hugging Face Hub or a URL to a
  deployed Inference Endpoint. If not provided, the default recommended model for image classification will be used.
- **function_to_apply** (`"ImageClassificationOutputTransform"`, *optional*) --
  The function to apply to the model outputs in order to retrieve the scores.
- **top_k** (`int`, *optional*) --
  When specified, limits the output to the top K most probable classes.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[ImageClassificationOutputElement]`</rettype><retdesc>a list of [ImageClassificationOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ImageClassificationOutputElement) items containing the predicted label and associated probability.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Perform image classification on the given image using the specified model.











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.image_classification.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()
>>> await client.image_classification("https://upload.wikimedia.org/wikipedia/commons/thumb/4/43/Cute_dog.jpg/320px-Cute_dog.jpg")
[ImageClassificationOutputElement(label='Blenheim spaniel', score=0.9779096841812134), ...]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>image_segmentation</name><anchor>huggingface_hub.AsyncInferenceClient.image_segmentation</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L1233</source><parameters>[{"name": "image", "val": ": typing.Union[bytes, typing.BinaryIO, str, pathlib.Path, ForwardRef('Image'), bytearray, memoryview]"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "mask_threshold", "val": ": typing.Optional[float] = None"}, {"name": "overlap_mask_area_threshold", "val": ": typing.Optional[float] = None"}, {"name": "subtask", "val": ": typing.Optional[ForwardRef('ImageSegmentationSubtask')] = None"}, {"name": "threshold", "val": ": typing.Optional[float] = None"}]</parameters><paramsdesc>- **image** (`Union[str, Path, bytes, BinaryIO, PIL.Image.Image]`) --
  The image to segment. It can be raw bytes, an image file, a URL to an online image, or a PIL Image.
- **model** (`str`, *optional*) --
  The model to use for image segmentation. Can be a model ID hosted on the Hugging Face Hub or a URL to a
  deployed Inference Endpoint. If not provided, the default recommended model for image segmentation will be used.
- **mask_threshold** (`float`, *optional*) --
  Threshold to use when turning the predicted masks into binary values.
- **overlap_mask_area_threshold** (`float`, *optional*) --
  Mask overlap threshold to eliminate small, disconnected segments.
- **subtask** (`"ImageSegmentationSubtask"`, *optional*) --
  Segmentation task to be performed, depending on model capabilities.
- **threshold** (`float`, *optional*) --
  Probability threshold to filter out predicted masks.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[ImageSegmentationOutputElement]`</rettype><retdesc>A list of [ImageSegmentationOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ImageSegmentationOutputElement) items containing the segmented masks and associated attributes.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Perform image segmentation on the given image using the specified model.

> [!WARNING]
> You must have `PIL` installed if you want to work with images (`pip install Pillow`).











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.image_segmentation.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()
>>> await client.image_segmentation("cat.jpg")
[ImageSegmentationOutputElement(score=0.989008, label='LABEL_184', mask=<PIL.PngImagePlugin.PngImageFile image mode=L size=400x300 at 0x7FDD2B129CC0>), ...]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>image_to_image</name><anchor>huggingface_hub.AsyncInferenceClient.image_to_image</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L1302</source><parameters>[{"name": "image", "val": ": typing.Union[bytes, typing.BinaryIO, str, pathlib.Path, ForwardRef('Image'), bytearray, memoryview]"}, {"name": "prompt", "val": ": typing.Optional[str] = None"}, {"name": "negative_prompt", "val": ": typing.Optional[str] = None"}, {"name": "num_inference_steps", "val": ": typing.Optional[int] = None"}, {"name": "guidance_scale", "val": ": typing.Optional[float] = None"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "target_size", "val": ": typing.Optional[huggingface_hub.inference._generated.types.image_to_image.ImageToImageTargetSize] = None"}, {"name": "**kwargs", "val": ""}]</parameters><paramsdesc>- **image** (`Union[str, Path, bytes, BinaryIO, PIL.Image.Image]`) --
  The input image for translation. It can be raw bytes, an image file, a URL to an online image, or a PIL Image.
- **prompt** (`str`, *optional*) --
  The text prompt to guide the image generation.
- **negative_prompt** (`str`, *optional*) --
  One prompt to guide what NOT to include in image generation.
- **num_inference_steps** (`int`, *optional*) --
  For diffusion models. The number of denoising steps. More denoising steps usually lead to a higher
  quality image at the expense of slower inference.
- **guidance_scale** (`float`, *optional*) --
  For diffusion models. A higher guidance scale value encourages the model to generate images closely
  linked to the text prompt at the expense of lower image quality.
- **model** (`str`, *optional*) --
  The model to use for inference. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. This parameter overrides the model defined at the instance level. Defaults to None.
- **target_size** (`ImageToImageTargetSize`, *optional*) --
  The size in pixels of the output image. This parameter is only supported by some providers and for
  specific models. It will be ignored when unsupported.</paramsdesc><paramgroups>0</paramgroups><rettype>`Image`</rettype><retdesc>The translated image.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Perform image-to-image translation using a specified model.

> [!WARNING]
> You must have `PIL` installed if you want to work with images (`pip install Pillow`).











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.image_to_image.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()
>>> image = await client.image_to_image("cat.jpg", prompt="turn the cat into a tiger")
>>> image.save("tiger.jpg")
```

</ExampleCodeBlock>



</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>image_to_text</name><anchor>huggingface_hub.AsyncInferenceClient.image_to_text</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L1459</source><parameters>[{"name": "image", "val": ": typing.Union[bytes, typing.BinaryIO, str, pathlib.Path, ForwardRef('Image'), bytearray, memoryview]"}, {"name": "model", "val": ": typing.Optional[str] = None"}]</parameters><paramsdesc>- **image** (`Union[str, Path, bytes, BinaryIO, PIL.Image.Image]`) --
  The input image to caption. It can be raw bytes, an image file, a URL to an online image, or a PIL Image.
- **model** (`str`, *optional*) --
  The model to use for inference. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. This parameter overrides the model defined at the instance level. Defaults to None.</paramsdesc><paramgroups>0</paramgroups><rettype>[ImageToTextOutput](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ImageToTextOutput)</rettype><retdesc>The generated text.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Takes an input image and return text.

Models can have very different outputs depending on your use case (image captioning, optical character recognition
(OCR), Pix2Struct, etc.). Please have a look to the model card to learn more about a model's specificities.











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.image_to_text.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()
>>> await client.image_to_text("cat.jpg")
'a cat standing in a grassy field '
>>> await client.image_to_text("https://upload.wikimedia.org/wikipedia/commons/thumb/4/43/Cute_dog.jpg/320px-Cute_dog.jpg")
'a dog laying on the grass next to a flower pot '
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>image_to_video</name><anchor>huggingface_hub.AsyncInferenceClient.image_to_video</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L1379</source><parameters>[{"name": "image", "val": ": typing.Union[bytes, typing.BinaryIO, str, pathlib.Path, ForwardRef('Image'), bytearray, memoryview]"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "prompt", "val": ": typing.Optional[str] = None"}, {"name": "negative_prompt", "val": ": typing.Optional[str] = None"}, {"name": "num_frames", "val": ": typing.Optional[float] = None"}, {"name": "num_inference_steps", "val": ": typing.Optional[int] = None"}, {"name": "guidance_scale", "val": ": typing.Optional[float] = None"}, {"name": "seed", "val": ": typing.Optional[int] = None"}, {"name": "target_size", "val": ": typing.Optional[huggingface_hub.inference._generated.types.image_to_video.ImageToVideoTargetSize] = None"}, {"name": "**kwargs", "val": ""}]</parameters><paramsdesc>- **image** (`Union[str, Path, bytes, BinaryIO, PIL.Image.Image]`) --
  The input image to generate a video from. It can be raw bytes, an image file, a URL to an online image, or a PIL Image.
- **model** (`str`, *optional*) --
  The model to use for inference. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. This parameter overrides the model defined at the instance level. Defaults to None.
- **prompt** (`str`, *optional*) --
  The text prompt to guide the video generation.
- **negative_prompt** (`str`, *optional*) --
  One prompt to guide what NOT to include in video generation.
- **num_frames** (`float`, *optional*) --
  The num_frames parameter determines how many video frames are generated.
- **num_inference_steps** (`int`, *optional*) --
  For diffusion models. The number of denoising steps. More denoising steps usually lead to a higher
  quality image at the expense of slower inference.
- **guidance_scale** (`float`, *optional*) --
  For diffusion models. A higher guidance scale value encourages the model to generate videos closely
  linked to the text prompt at the expense of lower image quality.
- **seed** (`int`, *optional*) --
  The seed to use for the video generation.
- **target_size** (`ImageToVideoTargetSize`, *optional*) --
  The size in pixel of the output video frames.
- **num_inference_steps** (`int`, *optional*) --
  The number of denoising steps. More denoising steps usually lead to a higher quality video at the
  expense of slower inference.
- **seed** (`int`, *optional*) --
  Seed for the random number generator.</paramsdesc><paramgroups>0</paramgroups><rettype>`bytes`</rettype><retdesc>The generated video.</retdesc></docstring>

Generate a video from an input image.







<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.image_to_video.example">

Examples:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()
>>> video = await client.image_to_video("cat.jpg", model="Wan-AI/Wan2.2-I2V-A14B", prompt="turn the cat into a tiger")
>>> with open("tiger.mp4", "wb") as f:
...     f.write(video)
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>object_detection</name><anchor>huggingface_hub.AsyncInferenceClient.object_detection</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L1506</source><parameters>[{"name": "image", "val": ": typing.Union[bytes, typing.BinaryIO, str, pathlib.Path, ForwardRef('Image'), bytearray, memoryview]"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "threshold", "val": ": typing.Optional[float] = None"}]</parameters><paramsdesc>- **image** (`Union[str, Path, bytes, BinaryIO, PIL.Image.Image]`) --
  The image to detect objects on. It can be raw bytes, an image file, a URL to an online image, or a PIL Image.
- **model** (`str`, *optional*) --
  The model to use for object detection. Can be a model ID hosted on the Hugging Face Hub or a URL to a
  deployed Inference Endpoint. If not provided, the default recommended model for object detection (DETR) will be used.
- **threshold** (`float`, *optional*) --
  The probability necessary to make a prediction.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[ObjectDetectionOutputElement]`</rettype><retdesc>A list of [ObjectDetectionOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ObjectDetectionOutputElement) items containing the bounding boxes and associated attributes.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.
- ``ValueError`` -- 
  If the request output is not a List.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError` or ``ValueError``</raisederrors></docstring>

Perform object detection on the given image using the specified model.

> [!WARNING]
> You must have `PIL` installed if you want to work with images (`pip install Pillow`).











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.object_detection.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()
>>> await client.object_detection("people.jpg")
[ObjectDetectionOutputElement(score=0.9486683011054993, label='person', box=ObjectDetectionBoundingBox(xmin=59, ymin=39, xmax=420, ymax=510)), ...]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>question_answering</name><anchor>huggingface_hub.AsyncInferenceClient.question_answering</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L1555</source><parameters>[{"name": "question", "val": ": str"}, {"name": "context", "val": ": str"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "align_to_words", "val": ": typing.Optional[bool] = None"}, {"name": "doc_stride", "val": ": typing.Optional[int] = None"}, {"name": "handle_impossible_answer", "val": ": typing.Optional[bool] = None"}, {"name": "max_answer_len", "val": ": typing.Optional[int] = None"}, {"name": "max_question_len", "val": ": typing.Optional[int] = None"}, {"name": "max_seq_len", "val": ": typing.Optional[int] = None"}, {"name": "top_k", "val": ": typing.Optional[int] = None"}]</parameters><paramsdesc>- **question** (`str`) --
  Question to be answered.
- **context** (`str`) --
  The context of the question.
- **model** (`str`) --
  The model to use for the question answering task. Can be a model ID hosted on the Hugging Face Hub or a URL to
  a deployed Inference Endpoint.
- **align_to_words** (`bool`, *optional*) --
  Attempts to align the answer to real words. Improves quality on space separated languages. Might hurt
  on non-space-separated languages (like Japanese or Chinese)
- **doc_stride** (`int`, *optional*) --
  If the context is too long to fit with the question for the model, it will be split in several chunks
  with some overlap. This argument controls the size of that overlap.
- **handle_impossible_answer** (`bool`, *optional*) --
  Whether to accept impossible as an answer.
- **max_answer_len** (`int`, *optional*) --
  The maximum length of predicted answers (e.g., only answers with a shorter length are considered).
- **max_question_len** (`int`, *optional*) --
  The maximum length of the question after tokenization. It will be truncated if needed.
- **max_seq_len** (`int`, *optional*) --
  The maximum length of the total sentence (context + question) in tokens of each chunk passed to the
  model. The context will be split in several chunks (using docStride as overlap) if needed.
- **top_k** (`int`, *optional*) --
  The number of answers to return (will be chosen by order of likelihood). Note that we return less than
  topk answers if there are not enough options available within the context.</paramsdesc><paramgroups>0</paramgroups><rettype>Union[`QuestionAnsweringOutputElement`, list[QuestionAnsweringOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.QuestionAnsweringOutputElement)]</rettype><retdesc>When top_k is 1 or not provided, it returns a single `QuestionAnsweringOutputElement`.
When top_k is greater than 1, it returns a list of `QuestionAnsweringOutputElement`.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Retrieve the answer to a question from a given text.











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.question_answering.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()
>>> await client.question_answering(question="What's my name?", context="My name is Clara and I live in Berkeley.")
QuestionAnsweringOutputElement(answer='Clara', end=16, score=0.9326565265655518, start=11)
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>sentence_similarity</name><anchor>huggingface_hub.AsyncInferenceClient.sentence_similarity</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L1640</source><parameters>[{"name": "sentence", "val": ": str"}, {"name": "other_sentences", "val": ": list"}, {"name": "model", "val": ": typing.Optional[str] = None"}]</parameters><paramsdesc>- **sentence** (`str`) --
  The main sentence to compare to others.
- **other_sentences** (`list[str]`) --
  The list of sentences to compare to.
- **model** (`str`, *optional*) --
  The model to use for the sentence similarity task. Can be a model ID hosted on the Hugging Face Hub or a URL to
  a deployed Inference Endpoint. If not provided, the default recommended sentence similarity model will be used.
  Defaults to None.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[float]`</rettype><retdesc>The embedding representing the input text.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Compute the semantic similarity between a sentence and a list of other sentences by comparing their embeddings.











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.sentence_similarity.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()
>>> await client.sentence_similarity(
...     "Machine learning is so easy.",
...     other_sentences=[
...         "Deep learning is so straightforward.",
...         "This is so difficult, like rocket science.",
...         "I can't believe how much I struggled with this.",
...     ],
... )
[0.7785726189613342, 0.45876261591911316, 0.2906220555305481]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>summarization</name><anchor>huggingface_hub.AsyncInferenceClient.summarization</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L1694</source><parameters>[{"name": "text", "val": ": str"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "clean_up_tokenization_spaces", "val": ": typing.Optional[bool] = None"}, {"name": "generate_parameters", "val": ": typing.Optional[dict[str, typing.Any]] = None"}, {"name": "truncation", "val": ": typing.Optional[ForwardRef('SummarizationTruncationStrategy')] = None"}]</parameters><paramsdesc>- **text** (`str`) --
  The input text to summarize.
- **model** (`str`, *optional*) --
  The model to use for inference. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. If not provided, the default recommended model for summarization will be used.
- **clean_up_tokenization_spaces** (`bool`, *optional*) --
  Whether to clean up the potential extra spaces in the text output.
- **generate_parameters** (`dict[str, Any]`, *optional*) --
  Additional parametrization of the text generation algorithm.
- **truncation** (`"SummarizationTruncationStrategy"`, *optional*) --
  The truncation strategy to use.</paramsdesc><paramgroups>0</paramgroups><rettype>[SummarizationOutput](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.SummarizationOutput)</rettype><retdesc>The generated summary text.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Generate a summary of a given text using a specified model.











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.summarization.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()
>>> await client.summarization("The Eiffel tower...")
SummarizationOutput(generated_text="The Eiffel tower is one of the most famous landmarks in the world....")
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>table_question_answering</name><anchor>huggingface_hub.AsyncInferenceClient.table_question_answering</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L1753</source><parameters>[{"name": "table", "val": ": dict"}, {"name": "query", "val": ": str"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "padding", "val": ": typing.Optional[ForwardRef('Padding')] = None"}, {"name": "sequential", "val": ": typing.Optional[bool] = None"}, {"name": "truncation", "val": ": typing.Optional[bool] = None"}]</parameters><paramsdesc>- **table** (`str`) --
  A table of data represented as a dict of lists where entries are headers and the lists are all the
  values, all lists must have the same size.
- **query** (`str`) --
  The query in plain text that you want to ask the table.
- **model** (`str`) --
  The model to use for the table-question-answering task. Can be a model ID hosted on the Hugging Face
  Hub or a URL to a deployed Inference Endpoint.
- **padding** (`"Padding"`, *optional*) --
  Activates and controls padding.
- **sequential** (`bool`, *optional*) --
  Whether to do inference sequentially or as a batch. Batching is faster, but models like SQA require the
  inference to be done sequentially to extract relations within sequences, given their conversational
  nature.
- **truncation** (`bool`, *optional*) --
  Activates and controls truncation.</paramsdesc><paramgroups>0</paramgroups><rettype>[TableQuestionAnsweringOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.TableQuestionAnsweringOutputElement)</rettype><retdesc>a table question answering output containing the answer, coordinates, cells and the aggregator used.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Retrieve the answer to a question from information given in a table.











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.table_question_answering.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()
>>> query = "How many stars does the transformers repository have?"
>>> table = {"Repository": ["Transformers", "Datasets", "Tokenizers"], "Stars": ["36542", "4512", "3934"]}
>>> await client.table_question_answering(table, query, model="google/tapas-base-finetuned-wtq")
TableQuestionAnsweringOutputElement(answer='36542', coordinates=[[0, 1]], cells=['36542'], aggregator='AVERAGE')
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>tabular_classification</name><anchor>huggingface_hub.AsyncInferenceClient.tabular_classification</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L1816</source><parameters>[{"name": "table", "val": ": dict"}, {"name": "model", "val": ": typing.Optional[str] = None"}]</parameters><paramsdesc>- **table** (`dict[str, Any]`) --
  Set of attributes to classify.
- **model** (`str`, *optional*) --
  The model to use for the tabular classification task. Can be a model ID hosted on the Hugging Face Hub or a URL to
  a deployed Inference Endpoint. If not provided, the default recommended tabular classification model will be used.
  Defaults to None.</paramsdesc><paramgroups>0</paramgroups><rettype>`List`</rettype><retdesc>a list of labels, one per row in the initial table.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Classifying a target category (a group) based on a set of attributes.











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.tabular_classification.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()
>>> table = {
...     "fixed_acidity": ["7.4", "7.8", "10.3"],
...     "volatile_acidity": ["0.7", "0.88", "0.32"],
...     "citric_acid": ["0", "0", "0.45"],
...     "residual_sugar": ["1.9", "2.6", "6.4"],
...     "chlorides": ["0.076", "0.098", "0.073"],
...     "free_sulfur_dioxide": ["11", "25", "5"],
...     "total_sulfur_dioxide": ["34", "67", "13"],
...     "density": ["0.9978", "0.9968", "0.9976"],
...     "pH": ["3.51", "3.2", "3.23"],
...     "sulphates": ["0.56", "0.68", "0.82"],
...     "alcohol": ["9.4", "9.8", "12.6"],
... }
>>> await client.tabular_classification(table=table, model="julien-c/wine-quality")
["5", "5", "5"]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>tabular_regression</name><anchor>huggingface_hub.AsyncInferenceClient.tabular_regression</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L1872</source><parameters>[{"name": "table", "val": ": dict"}, {"name": "model", "val": ": typing.Optional[str] = None"}]</parameters><paramsdesc>- **table** (`dict[str, Any]`) --
  Set of attributes stored in a table. The attributes used to predict the target can be both numerical and categorical.
- **model** (`str`, *optional*) --
  The model to use for the tabular regression task. Can be a model ID hosted on the Hugging Face Hub or a URL to
  a deployed Inference Endpoint. If not provided, the default recommended tabular regression model will be used.
  Defaults to None.</paramsdesc><paramgroups>0</paramgroups><rettype>`List`</rettype><retdesc>a list of predicted numerical target values.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Predicting a numerical target value given a set of attributes/features in a table.











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.tabular_regression.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()
>>> table = {
...     "Height": ["11.52", "12.48", "12.3778"],
...     "Length1": ["23.2", "24", "23.9"],
...     "Length2": ["25.4", "26.3", "26.5"],
...     "Length3": ["30", "31.2", "31.1"],
...     "Species": ["Bream", "Bream", "Bream"],
...     "Width": ["4.02", "4.3056", "4.6961"],
... }
>>> await client.tabular_regression(table, model="scikit-learn/Fish-Weight")
[110, 120, 130]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>text_classification</name><anchor>huggingface_hub.AsyncInferenceClient.text_classification</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L1923</source><parameters>[{"name": "text", "val": ": str"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "top_k", "val": ": typing.Optional[int] = None"}, {"name": "function_to_apply", "val": ": typing.Optional[ForwardRef('TextClassificationOutputTransform')] = None"}]</parameters><paramsdesc>- **text** (`str`) --
  A string to be classified.
- **model** (`str`, *optional*) --
  The model to use for the text classification task. Can be a model ID hosted on the Hugging Face Hub or a URL to
  a deployed Inference Endpoint. If not provided, the default recommended text classification model will be used.
  Defaults to None.
- **top_k** (`int`, *optional*) --
  When specified, limits the output to the top K most probable classes.
- **function_to_apply** (`"TextClassificationOutputTransform"`, *optional*) --
  The function to apply to the model outputs in order to retrieve the scores.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[TextClassificationOutputElement]`</rettype><retdesc>a list of [TextClassificationOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.TextClassificationOutputElement) items containing the predicted label and associated probability.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Perform text classification (e.g. sentiment-analysis) on the given text.











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.text_classification.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()
>>> await client.text_classification("I like you")
[
    TextClassificationOutputElement(label='POSITIVE', score=0.9998695850372314),
    TextClassificationOutputElement(label='NEGATIVE', score=0.0001304351753788069),
]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>text_generation</name><anchor>huggingface_hub.AsyncInferenceClient.text_generation</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L2132</source><parameters>[{"name": "prompt", "val": ": str"}, {"name": "details", "val": ": typing.Optional[bool] = None"}, {"name": "stream", "val": ": typing.Optional[bool] = None"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "adapter_id", "val": ": typing.Optional[str] = None"}, {"name": "best_of", "val": ": typing.Optional[int] = None"}, {"name": "decoder_input_details", "val": ": typing.Optional[bool] = None"}, {"name": "do_sample", "val": ": typing.Optional[bool] = None"}, {"name": "frequency_penalty", "val": ": typing.Optional[float] = None"}, {"name": "grammar", "val": ": typing.Optional[huggingface_hub.inference._generated.types.text_generation.TextGenerationInputGrammarType] = None"}, {"name": "max_new_tokens", "val": ": typing.Optional[int] = None"}, {"name": "repetition_penalty", "val": ": typing.Optional[float] = None"}, {"name": "return_full_text", "val": ": typing.Optional[bool] = None"}, {"name": "seed", "val": ": typing.Optional[int] = None"}, {"name": "stop", "val": ": typing.Optional[list[str]] = None"}, {"name": "stop_sequences", "val": ": typing.Optional[list[str]] = None"}, {"name": "temperature", "val": ": typing.Optional[float] = None"}, {"name": "top_k", "val": ": typing.Optional[int] = None"}, {"name": "top_n_tokens", "val": ": typing.Optional[int] = None"}, {"name": "top_p", "val": ": typing.Optional[float] = None"}, {"name": "truncate", "val": ": typing.Optional[int] = None"}, {"name": "typical_p", "val": ": typing.Optional[float] = None"}, {"name": "watermark", "val": ": typing.Optional[bool] = None"}]</parameters><paramsdesc>- **prompt** (`str`) --
  Input text.
- **details** (`bool`, *optional*) --
  By default, text_generation returns a string. Pass `details=True` if you want a detailed output (tokens,
  probabilities, seed, finish reason, etc.). Only available for models running on with the
  `text-generation-inference` backend.
- **stream** (`bool`, *optional*) --
  By default, text_generation returns the full generated text. Pass `stream=True` if you want a stream of
  tokens to be returned. Only available for models running on with the `text-generation-inference`
  backend.
- **model** (`str`, *optional*) --
  The model to use for inference. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. This parameter overrides the model defined at the instance level. Defaults to None.
- **adapter_id** (`str`, *optional*) --
  Lora adapter id.
- **best_of** (`int`, *optional*) --
  Generate best_of sequences and return the one if the highest token logprobs.
- **decoder_input_details** (`bool`, *optional*) --
  Return the decoder input token logprobs and ids. You must set `details=True` as well for it to be taken
  into account. Defaults to `False`.
- **do_sample** (`bool`, *optional*) --
  Activate logits sampling
- **frequency_penalty** (`float`, *optional*) --
  Number between -2.0 and 2.0. Positive values penalize new tokens based on their existing frequency in
  the text so far, decreasing the model's likelihood to repeat the same line verbatim.
- **grammar** ([TextGenerationInputGrammarType](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.TextGenerationInputGrammarType), *optional*) --
  Grammar constraints. Can be either a JSONSchema or a regex.
- **max_new_tokens** (`int`, *optional*) --
  Maximum number of generated tokens. Defaults to 100.
- **repetition_penalty** (`float`, *optional*) --
  The parameter for repetition penalty. 1.0 means no penalty. See [this
  paper](https://arxiv.org/pdf/1909.05858.pdf) for more details.
- **return_full_text** (`bool`, *optional*) --
  Whether to prepend the prompt to the generated text
- **seed** (`int`, *optional*) --
  Random sampling seed
- **stop** (`list[str]`, *optional*) --
  Stop generating tokens if a member of `stop` is generated.
- **stop_sequences** (`list[str]`, *optional*) --
  Deprecated argument. Use `stop` instead.
- **temperature** (`float`, *optional*) --
  The value used to module the logits distribution.
- **top_n_tokens** (`int`, *optional*) --
  Return information about the `top_n_tokens` most likely tokens at each generation step, instead of
  just the sampled token.
- **top_k** (`int`, *optional`) --
  The number of highest probability vocabulary tokens to keep for top-k-filtering.
- **top_p** (`float`, *optional`) --
  If set to < 1, only the smallest set of most probable tokens with probabilities that add up to `top_p` or
  higher are kept for generation.
- **truncate** (`int`, *optional`) --
  Truncate inputs tokens to the given size.
- **typical_p** (`float`, *optional`) --
  Typical Decoding mass
  See [Typical Decoding for Natural Language Generation](https://arxiv.org/abs/2202.00666) for more information
- **watermark** (`bool`, *optional*) --
  Watermarking with [A Watermark for Large Language Models](https://arxiv.org/abs/2301.10226)</paramsdesc><paramgroups>0</paramgroups><rettype>`Union[str, TextGenerationOutput, AsyncIterable[str], AsyncIterable[TextGenerationStreamOutput]]`</rettype><retdesc>Generated text returned from the server:
- if `stream=False` and `details=False`, the generated text is returned as a `str` (default)
- if `stream=True` and `details=False`, the generated text is returned token by token as a `AsyncIterable[str]`
- if `stream=False` and `details=True`, the generated text is returned with more details as a [TextGenerationOutput](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.TextGenerationOutput)
- if `details=True` and `stream=True`, the generated text is returned token by token as a iterable of [TextGenerationStreamOutput](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.TextGenerationStreamOutput)</retdesc><raises>- ``ValidationError`` -- 
  If input values are not valid. No HTTP call is made to the server.
- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>``ValidationError`` or [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Given a prompt, generate the following text.

> [!TIP]
> If you want to generate a response from chat messages, you should use the [InferenceClient.chat_completion()](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceClient.chat_completion) method.
> It accepts a list of messages instead of a single text prompt and handles the chat templating for you.











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.text_generation.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()

# Case 1: generate text
>>> await client.text_generation("The huggingface_hub library is ", max_new_tokens=12)
'100% open source and built to be easy to use.'

# Case 2: iterate over the generated tokens. Useful for large generation.
>>> async for token in await client.text_generation("The huggingface_hub library is ", max_new_tokens=12, stream=True):
...     print(token)
100
%
open
source
and
built
to
be
easy
to
use
.

# Case 3: get more details about the generation process.
>>> await client.text_generation("The huggingface_hub library is ", max_new_tokens=12, details=True)
TextGenerationOutput(
    generated_text='100% open source and built to be easy to use.',
    details=TextGenerationDetails(
        finish_reason='length',
        generated_tokens=12,
        seed=None,
        prefill=[
            TextGenerationPrefillOutputToken(id=487, text='The', logprob=None),
            TextGenerationPrefillOutputToken(id=53789, text=' hugging', logprob=-13.171875),
            (...)
            TextGenerationPrefillOutputToken(id=204, text=' ', logprob=-7.0390625)
        ],
        tokens=[
            TokenElement(id=1425, text='100', logprob=-1.0175781, special=False),
            TokenElement(id=16, text='%', logprob=-0.0463562, special=False),
            (...)
            TokenElement(id=25, text='.', logprob=-0.5703125, special=False)
        ],
        best_of_sequences=None
    )
)

# Case 4: iterate over the generated tokens with more details.
# Last object is more complete, containing the full generated text and the finish reason.
>>> async for details in await client.text_generation("The huggingface_hub library is ", max_new_tokens=12, details=True, stream=True):
...     print(details)
...
TextGenerationStreamOutput(token=TokenElement(id=1425, text='100', logprob=-1.0175781, special=False), generated_text=None, details=None)
TextGenerationStreamOutput(token=TokenElement(id=16, text='%', logprob=-0.0463562, special=False), generated_text=None, details=None)
TextGenerationStreamOutput(token=TokenElement(id=1314, text=' open', logprob=-1.3359375, special=False), generated_text=None, details=None)
TextGenerationStreamOutput(token=TokenElement(id=3178, text=' source', logprob=-0.28100586, special=False), generated_text=None, details=None)
TextGenerationStreamOutput(token=TokenElement(id=273, text=' and', logprob=-0.5961914, special=False), generated_text=None, details=None)
TextGenerationStreamOutput(token=TokenElement(id=3426, text=' built', logprob=-1.9423828, special=False), generated_text=None, details=None)
TextGenerationStreamOutput(token=TokenElement(id=271, text=' to', logprob=-1.4121094, special=False), generated_text=None, details=None)
TextGenerationStreamOutput(token=TokenElement(id=314, text=' be', logprob=-1.5224609, special=False), generated_text=None, details=None)
TextGenerationStreamOutput(token=TokenElement(id=1833, text=' easy', logprob=-2.1132812, special=False), generated_text=None, details=None)
TextGenerationStreamOutput(token=TokenElement(id=271, text=' to', logprob=-0.08520508, special=False), generated_text=None, details=None)
TextGenerationStreamOutput(token=TokenElement(id=745, text=' use', logprob=-0.39453125, special=False), generated_text=None, details=None)
TextGenerationStreamOutput(token=TokenElement(
    id=25,
    text='.',
    logprob=-0.5703125,
    special=False),
    generated_text='100% open source and built to be easy to use.',
    details=TextGenerationStreamOutputStreamDetails(finish_reason='length', generated_tokens=12, seed=None)
)

# Case 5: generate constrained output using grammar
>>> response = await client.text_generation(
...     prompt="I saw a puppy a cat and a raccoon during my bike ride in the park",
...     model="HuggingFaceH4/zephyr-orpo-141b-A35b-v0.1",
...     max_new_tokens=100,
...     repetition_penalty=1.3,
...     grammar={
...         "type": "json",
...         "value": {
...             "properties": {
...                 "location": {"type": "string"},
...                 "activity": {"type": "string"},
...                 "animals_seen": {"type": "integer", "minimum": 1, "maximum": 5},
...                 "animals": {"type": "array", "items": {"type": "string"}},
...             },
...             "required": ["location", "activity", "animals_seen", "animals"],
...         },
...     },
... )
>>> json.loads(response)
{
    "activity": "bike riding",
    "animals": ["puppy", "cat", "raccoon"],
    "animals_seen": 3,
    "location": "park"
}
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>text_to_image</name><anchor>huggingface_hub.AsyncInferenceClient.text_to_image</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L2472</source><parameters>[{"name": "prompt", "val": ": str"}, {"name": "negative_prompt", "val": ": typing.Optional[str] = None"}, {"name": "height", "val": ": typing.Optional[int] = None"}, {"name": "width", "val": ": typing.Optional[int] = None"}, {"name": "num_inference_steps", "val": ": typing.Optional[int] = None"}, {"name": "guidance_scale", "val": ": typing.Optional[float] = None"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "scheduler", "val": ": typing.Optional[str] = None"}, {"name": "seed", "val": ": typing.Optional[int] = None"}, {"name": "extra_body", "val": ": typing.Optional[dict[str, typing.Any]] = None"}]</parameters><paramsdesc>- **prompt** (`str`) --
  The prompt to generate an image from.
- **negative_prompt** (`str`, *optional*) --
  One prompt to guide what NOT to include in image generation.
- **height** (`int`, *optional*) --
  The height in pixels of the output image
- **width** (`int`, *optional*) --
  The width in pixels of the output image
- **num_inference_steps** (`int`, *optional*) --
  The number of denoising steps. More denoising steps usually lead to a higher quality image at the
  expense of slower inference.
- **guidance_scale** (`float`, *optional*) --
  A higher guidance scale value encourages the model to generate images closely linked to the text
  prompt, but values too high may cause saturation and other artifacts.
- **model** (`str`, *optional*) --
  The model to use for inference. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. If not provided, the default recommended text-to-image model will be used.
  Defaults to None.
- **scheduler** (`str`, *optional*) --
  Override the scheduler with a compatible one.
- **seed** (`int`, *optional*) --
  Seed for the random number generator.
- **extra_body** (`dict[str, Any]`, *optional*) --
  Additional provider-specific parameters to pass to the model. Refer to the provider's documentation
  for supported parameters.</paramsdesc><paramgroups>0</paramgroups><rettype>`Image`</rettype><retdesc>The generated image.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Generate an image based on a given text using a specified model.

> [!WARNING]
> You must have `PIL` installed if you want to work with images (`pip install Pillow`).

> [!TIP]
> You can pass provider-specific parameters to the model by using the `extra_body` argument.











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.text_to_image.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()

>>> image = await client.text_to_image("An astronaut riding a horse on the moon.")
>>> image.save("astronaut.png")

>>> image = await client.text_to_image(
...     "An astronaut riding a horse on the moon.",
...     negative_prompt="low resolution, blurry",
...     model="stabilityai/stable-diffusion-2-1",
... )
>>> image.save("better_astronaut.png")
```

</ExampleCodeBlock>
Example using a third-party provider directly. Usage will be billed on your fal.ai account.
<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.text_to_image.example-2">

```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient(
...     provider="fal-ai",  # Use fal.ai provider
...     api_key="fal-ai-api-key",  # Pass your fal.ai API key
... )
>>> image = client.text_to_image(
...     "A majestic lion in a fantasy forest",
...     model="black-forest-labs/FLUX.1-schnell",
... )
>>> image.save("lion.png")
```

</ExampleCodeBlock>

Example using a third-party provider through Hugging Face Routing. Usage will be billed on your Hugging Face account.
<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.text_to_image.example-3">

```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient(
...     provider="replicate",  # Use replicate provider
...     api_key="hf_...",  # Pass your HF token
... )
>>> image = client.text_to_image(
...     "An astronaut riding a horse on the moon.",
...     model="black-forest-labs/FLUX.1-dev",
... )
>>> image.save("astronaut.png")
```

</ExampleCodeBlock>

Example using Replicate provider with extra parameters
<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.text_to_image.example-4">

```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient(
...     provider="replicate",  # Use replicate provider
...     api_key="hf_...",  # Pass your HF token
... )
>>> image = client.text_to_image(
...     "An astronaut riding a horse on the moon.",
...     model="black-forest-labs/FLUX.1-schnell",
...     extra_body={"output_quality": 100},
... )
>>> image.save("astronaut.png")
```

</ExampleCodeBlock>



</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>text_to_speech</name><anchor>huggingface_hub.AsyncInferenceClient.text_to_speech</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L2710</source><parameters>[{"name": "text", "val": ": str"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "do_sample", "val": ": typing.Optional[bool] = None"}, {"name": "early_stopping", "val": ": typing.Union[bool, ForwardRef('TextToSpeechEarlyStoppingEnum'), NoneType] = None"}, {"name": "epsilon_cutoff", "val": ": typing.Optional[float] = None"}, {"name": "eta_cutoff", "val": ": typing.Optional[float] = None"}, {"name": "max_length", "val": ": typing.Optional[int] = None"}, {"name": "max_new_tokens", "val": ": typing.Optional[int] = None"}, {"name": "min_length", "val": ": typing.Optional[int] = None"}, {"name": "min_new_tokens", "val": ": typing.Optional[int] = None"}, {"name": "num_beam_groups", "val": ": typing.Optional[int] = None"}, {"name": "num_beams", "val": ": typing.Optional[int] = None"}, {"name": "penalty_alpha", "val": ": typing.Optional[float] = None"}, {"name": "temperature", "val": ": typing.Optional[float] = None"}, {"name": "top_k", "val": ": typing.Optional[int] = None"}, {"name": "top_p", "val": ": typing.Optional[float] = None"}, {"name": "typical_p", "val": ": typing.Optional[float] = None"}, {"name": "use_cache", "val": ": typing.Optional[bool] = None"}, {"name": "extra_body", "val": ": typing.Optional[dict[str, typing.Any]] = None"}]</parameters><paramsdesc>- **text** (`str`) --
  The text to synthesize.
- **model** (`str`, *optional*) --
  The model to use for inference. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. If not provided, the default recommended text-to-speech model will be used.
  Defaults to None.
- **do_sample** (`bool`, *optional*) --
  Whether to use sampling instead of greedy decoding when generating new tokens.
- **early_stopping** (`Union[bool, "TextToSpeechEarlyStoppingEnum"]`, *optional*) --
  Controls the stopping condition for beam-based methods.
- **epsilon_cutoff** (`float`, *optional*) --
  If set to float strictly between 0 and 1, only tokens with a conditional probability greater than
  epsilon_cutoff will be sampled. In the paper, suggested values range from 3e-4 to 9e-4, depending on
  the size of the model. See [Truncation Sampling as Language Model
  Desmoothing](https://hf.co/papers/2210.15191) for more details.
- **eta_cutoff** (`float`, *optional*) --
  Eta sampling is a hybrid of locally typical sampling and epsilon sampling. If set to float strictly
  between 0 and 1, a token is only considered if it is greater than either eta_cutoff or sqrt(eta_cutoff)
  * exp(-entropy(softmax(next_token_logits))). The latter term is intuitively the expected next token
  probability, scaled by sqrt(eta_cutoff). In the paper, suggested values range from 3e-4 to 2e-3,
  depending on the size of the model. See [Truncation Sampling as Language Model
  Desmoothing](https://hf.co/papers/2210.15191) for more details.
- **max_length** (`int`, *optional*) --
  The maximum length (in tokens) of the generated text, including the input.
- **max_new_tokens** (`int`, *optional*) --
  The maximum number of tokens to generate. Takes precedence over max_length.
- **min_length** (`int`, *optional*) --
  The minimum length (in tokens) of the generated text, including the input.
- **min_new_tokens** (`int`, *optional*) --
  The minimum number of tokens to generate. Takes precedence over min_length.
- **num_beam_groups** (`int`, *optional*) --
  Number of groups to divide num_beams into in order to ensure diversity among different groups of beams.
  See [this paper](https://hf.co/papers/1610.02424) for more details.
- **num_beams** (`int`, *optional*) --
  Number of beams to use for beam search.
- **penalty_alpha** (`float`, *optional*) --
  The value balances the model confidence and the degeneration penalty in contrastive search decoding.
- **temperature** (`float`, *optional*) --
  The value used to modulate the next token probabilities.
- **top_k** (`int`, *optional*) --
  The number of highest probability vocabulary tokens to keep for top-k-filtering.
- **top_p** (`float`, *optional*) --
  If set to float < 1, only the smallest set of most probable tokens with probabilities that add up to
  top_p or higher are kept for generation.
- **typical_p** (`float`, *optional*) --
  Local typicality measures how similar the conditional probability of predicting a target token next is
  to the expected conditional probability of predicting a random token next, given the partial text
  already generated. If set to float < 1, the smallest set of the most locally typical tokens with
  probabilities that add up to typical_p or higher are kept for generation. See [this
  paper](https://hf.co/papers/2202.00666) for more details.
- **use_cache** (`bool`, *optional*) --
  Whether the model should use the past last key/values attentions to speed up decoding
- **extra_body** (`dict[str, Any]`, *optional*) --
  Additional provider-specific parameters to pass to the model. Refer to the provider's documentation
  for supported parameters.</paramsdesc><paramgroups>0</paramgroups><rettype>`bytes`</rettype><retdesc>The generated audio.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Synthesize an audio of a voice pronouncing a given text.

> [!TIP]
> You can pass provider-specific parameters to the model by using the `extra_body` argument.











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.text_to_speech.example">

Example:
```py
# Must be run in an async context
>>> from pathlib import Path
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()

>>> audio = await client.text_to_speech("Hello world")
>>> Path("hello_world.flac").write_bytes(audio)
```

</ExampleCodeBlock>

Example using a third-party provider directly. Usage will be billed on your Replicate account.
<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.text_to_speech.example-2">

```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient(
...     provider="replicate",
...     api_key="your-replicate-api-key",  # Pass your Replicate API key directly
... )
>>> audio = client.text_to_speech(
...     text="Hello world",
...     model="OuteAI/OuteTTS-0.3-500M",
... )
>>> Path("hello_world.flac").write_bytes(audio)
```

</ExampleCodeBlock>

Example using a third-party provider through Hugging Face Routing. Usage will be billed on your Hugging Face account.
<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.text_to_speech.example-3">

```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient(
...     provider="replicate",
...     api_key="hf_...",  # Pass your HF token
... )
>>> audio =client.text_to_speech(
...     text="Hello world",
...     model="OuteAI/OuteTTS-0.3-500M",
... )
>>> Path("hello_world.flac").write_bytes(audio)
```

</ExampleCodeBlock>
Example using Replicate provider with extra parameters
<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.text_to_speech.example-4">

```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient(
...     provider="replicate",  # Use replicate provider
...     api_key="hf_...",  # Pass your HF token
... )
>>> audio = client.text_to_speech(
...     "Hello, my name is Kororo, an awesome text-to-speech model.",
...     model="hexgrad/Kokoro-82M",
...     extra_body={"voice": "af_nicole"},
... )
>>> Path("hello.flac").write_bytes(audio)
```

</ExampleCodeBlock>

Example music-gen using "YuE-s1-7B-anneal-en-cot" on fal.ai
<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.text_to_speech.example-5">

```py
>>> from huggingface_hub import InferenceClient
>>> lyrics = '''
... [verse]
... In the town where I was born
... Lived a man who sailed to sea
... And he told us of his life
... In the land of submarines
... So we sailed on to the sun
... 'Til we found a sea of green
... And we lived beneath the waves
... In our yellow submarine

... [chorus]
... We all live in a yellow submarine
... Yellow submarine, yellow submarine
... We all live in a yellow submarine
... Yellow submarine, yellow submarine
... '''
>>> genres = "pavarotti-style tenor voice"
>>> client = InferenceClient(
...     provider="fal-ai",
...     model="m-a-p/YuE-s1-7B-anneal-en-cot",
...     api_key=...,
... )
>>> audio = client.text_to_speech(lyrics, extra_body={"genres": genres})
>>> with open("output.mp3", "wb") as f:
...     f.write(audio)
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>text_to_video</name><anchor>huggingface_hub.AsyncInferenceClient.text_to_video</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L2613</source><parameters>[{"name": "prompt", "val": ": str"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "guidance_scale", "val": ": typing.Optional[float] = None"}, {"name": "negative_prompt", "val": ": typing.Optional[list[str]] = None"}, {"name": "num_frames", "val": ": typing.Optional[float] = None"}, {"name": "num_inference_steps", "val": ": typing.Optional[int] = None"}, {"name": "seed", "val": ": typing.Optional[int] = None"}, {"name": "extra_body", "val": ": typing.Optional[dict[str, typing.Any]] = None"}]</parameters><paramsdesc>- **prompt** (`str`) --
  The prompt to generate a video from.
- **model** (`str`, *optional*) --
  The model to use for inference. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. If not provided, the default recommended text-to-video model will be used.
  Defaults to None.
- **guidance_scale** (`float`, *optional*) --
  A higher guidance scale value encourages the model to generate videos closely linked to the text
  prompt, but values too high may cause saturation and other artifacts.
- **negative_prompt** (`list[str]`, *optional*) --
  One or several prompt to guide what NOT to include in video generation.
- **num_frames** (`float`, *optional*) --
  The num_frames parameter determines how many video frames are generated.
- **num_inference_steps** (`int`, *optional*) --
  The number of denoising steps. More denoising steps usually lead to a higher quality video at the
  expense of slower inference.
- **seed** (`int`, *optional*) --
  Seed for the random number generator.
- **extra_body** (`dict[str, Any]`, *optional*) --
  Additional provider-specific parameters to pass to the model. Refer to the provider's documentation
  for supported parameters.</paramsdesc><paramgroups>0</paramgroups><rettype>`bytes`</rettype><retdesc>The generated video.</retdesc></docstring>

Generate a video based on a given text.

> [!TIP]
> You can pass provider-specific parameters to the model by using the `extra_body` argument.







Example:

Example using a third-party provider directly. Usage will be billed on your fal.ai account.
<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.text_to_video.example">

```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient(
...     provider="fal-ai",  # Using fal.ai provider
...     api_key="fal-ai-api-key",  # Pass your fal.ai API key
... )
>>> video = client.text_to_video(
...     "A majestic lion running in a fantasy forest",
...     model="tencent/HunyuanVideo",
... )
>>> with open("lion.mp4", "wb") as file:
...     file.write(video)
```

</ExampleCodeBlock>

Example using a third-party provider through Hugging Face Routing. Usage will be billed on your Hugging Face account.
<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.text_to_video.example-2">

```py
>>> from huggingface_hub import InferenceClient
>>> client = InferenceClient(
...     provider="replicate",  # Using replicate provider
...     api_key="hf_...",  # Pass your HF token
... )
>>> video = client.text_to_video(
...     "A cat running in a park",
...     model="genmo/mochi-1-preview",
... )
>>> with open("cat.mp4", "wb") as file:
...     file.write(video)
```

</ExampleCodeBlock>



</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>token_classification</name><anchor>huggingface_hub.AsyncInferenceClient.token_classification</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L2919</source><parameters>[{"name": "text", "val": ": str"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "aggregation_strategy", "val": ": typing.Optional[ForwardRef('TokenClassificationAggregationStrategy')] = None"}, {"name": "ignore_labels", "val": ": typing.Optional[list[str]] = None"}, {"name": "stride", "val": ": typing.Optional[int] = None"}]</parameters><paramsdesc>- **text** (`str`) --
  A string to be classified.
- **model** (`str`, *optional*) --
  The model to use for the token classification task. Can be a model ID hosted on the Hugging Face Hub or a URL to
  a deployed Inference Endpoint. If not provided, the default recommended token classification model will be used.
  Defaults to None.
- **aggregation_strategy** (`"TokenClassificationAggregationStrategy"`, *optional*) --
  The strategy used to fuse tokens based on model predictions
- **ignore_labels** (`list[str`, *optional*) --
  A list of labels to ignore
- **stride** (`int`, *optional*) --
  The number of overlapping tokens between chunks when splitting the input text.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[TokenClassificationOutputElement]`</rettype><retdesc>List of [TokenClassificationOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.TokenClassificationOutputElement) items containing the entity group, confidence score, word, start and end index.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Perform token classification on the given text.
Usually used for sentence parsing, either grammatical, or Named Entity Recognition (NER) to understand keywords contained within text.











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.token_classification.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()
>>> await client.token_classification("My name is Sarah Jessica Parker but you can call me Jessica")
[
    TokenClassificationOutputElement(
        entity_group='PER',
        score=0.9971321225166321,
        word='Sarah Jessica Parker',
        start=11,
        end=31,
    ),
    TokenClassificationOutputElement(
        entity_group='PER',
        score=0.9773476123809814,
        word='Jessica',
        start=52,
        end=59,
    )
]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>translation</name><anchor>huggingface_hub.AsyncInferenceClient.translation</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L2995</source><parameters>[{"name": "text", "val": ": str"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "src_lang", "val": ": typing.Optional[str] = None"}, {"name": "tgt_lang", "val": ": typing.Optional[str] = None"}, {"name": "clean_up_tokenization_spaces", "val": ": typing.Optional[bool] = None"}, {"name": "truncation", "val": ": typing.Optional[ForwardRef('TranslationTruncationStrategy')] = None"}, {"name": "generate_parameters", "val": ": typing.Optional[dict[str, typing.Any]] = None"}]</parameters><paramsdesc>- **text** (`str`) --
  A string to be translated.
- **model** (`str`, *optional*) --
  The model to use for the translation task. Can be a model ID hosted on the Hugging Face Hub or a URL to
  a deployed Inference Endpoint. If not provided, the default recommended translation model will be used.
  Defaults to None.
- **src_lang** (`str`, *optional*) --
  The source language of the text. Required for models that can translate from multiple languages.
- **tgt_lang** (`str`, *optional*) --
  Target language to translate to. Required for models that can translate to multiple languages.
- **clean_up_tokenization_spaces** (`bool`, *optional*) --
  Whether to clean up the potential extra spaces in the text output.
- **truncation** (`"TranslationTruncationStrategy"`, *optional*) --
  The truncation strategy to use.
- **generate_parameters** (`dict[str, Any]`, *optional*) --
  Additional parametrization of the text generation algorithm.</paramsdesc><paramgroups>0</paramgroups><rettype>[TranslationOutput](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.TranslationOutput)</rettype><retdesc>The generated translated text.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.
- ``ValueError`` -- 
  If only one of the `src_lang` and `tgt_lang` arguments are provided.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError` or ``ValueError``</raisederrors></docstring>

Convert text from one language to another.

Check out https://huggingface.co/tasks/translation for more information on how to choose the best model for
your specific use case. Source and target languages usually depend on the model.
However, it is possible to specify source and target languages for certain models. If you are working with one of these models,
you can use `src_lang` and `tgt_lang` arguments to pass the relevant information.











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.translation.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()
>>> await client.translation("My name is Wolfgang and I live in Berlin")
'Mein Name ist Wolfgang und ich lebe in Berlin.'
>>> await client.translation("My name is Wolfgang and I live in Berlin", model="Helsinki-NLP/opus-mt-en-fr")
TranslationOutput(translation_text='Je m'appelle Wolfgang et je vis à Berlin.')
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.translation.example-2">

Specifying languages:
```py
>>> client.translation("My name is Sarah Jessica Parker but you can call me Jessica", model="facebook/mbart-large-50-many-to-many-mmt", src_lang="en_XX", tgt_lang="fr_XX")
"Mon nom est Sarah Jessica Parker mais vous pouvez m'appeler Jessica"
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>visual_question_answering</name><anchor>huggingface_hub.AsyncInferenceClient.visual_question_answering</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L3085</source><parameters>[{"name": "image", "val": ": typing.Union[bytes, typing.BinaryIO, str, pathlib.Path, ForwardRef('Image'), bytearray, memoryview]"}, {"name": "question", "val": ": str"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "top_k", "val": ": typing.Optional[int] = None"}]</parameters><paramsdesc>- **image** (`Union[str, Path, bytes, BinaryIO, PIL.Image.Image]`) --
  The input image for the context. It can be raw bytes, an image file, a URL to an online image, or a PIL Image.
- **question** (`str`) --
  Question to be answered.
- **model** (`str`, *optional*) --
  The model to use for the visual question answering task. Can be a model ID hosted on the Hugging Face Hub or a URL to
  a deployed Inference Endpoint. If not provided, the default recommended visual question answering model will be used.
  Defaults to None.
- **top_k** (`int`, *optional*) --
  The number of answers to return (will be chosen by order of likelihood). Note that we return less than
  topk answers if there are not enough options available within the context.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[VisualQuestionAnsweringOutputElement]`</rettype><retdesc>a list of [VisualQuestionAnsweringOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.VisualQuestionAnsweringOutputElement) items containing the predicted label and associated probability.</retdesc><raises>- ``InferenceTimeoutError`` -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>``InferenceTimeoutError`` or `HfHubHTTPError`</raisederrors></docstring>

Answering open-ended questions based on an image.











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.visual_question_answering.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()
>>> await client.visual_question_answering(
...     image="https://huggingface.co/datasets/mishig/sample_images/resolve/main/tiger.jpg",
...     question="What is the animal doing?"
... )
[
    VisualQuestionAnsweringOutputElement(score=0.778609573841095, answer='laying down'),
    VisualQuestionAnsweringOutputElement(score=0.6957435607910156, answer='sitting'),
]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>zero_shot_classification</name><anchor>huggingface_hub.AsyncInferenceClient.zero_shot_classification</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L3145</source><parameters>[{"name": "text", "val": ": str"}, {"name": "candidate_labels", "val": ": list"}, {"name": "multi_label", "val": ": typing.Optional[bool] = False"}, {"name": "hypothesis_template", "val": ": typing.Optional[str] = None"}, {"name": "model", "val": ": typing.Optional[str] = None"}]</parameters><paramsdesc>- **text** (`str`) --
  The input text to classify.
- **candidate_labels** (`list[str]`) --
  The set of possible class labels to classify the text into.
- **labels** (`list[str]`, *optional*) --
  (deprecated) List of strings. Each string is the verbalization of a possible label for the input text.
- **multi_label** (`bool`, *optional*) --
  Whether multiple candidate labels can be true. If false, the scores are normalized such that the sum of
  the label likelihoods for each sequence is 1. If true, the labels are considered independent and
  probabilities are normalized for each candidate.
- **hypothesis_template** (`str`, *optional*) --
  The sentence used in conjunction with `candidate_labels` to attempt the text classification by
  replacing the placeholder with the candidate labels.
- **model** (`str`, *optional*) --
  The model to use for inference. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. This parameter overrides the model defined at the instance level. If not provided, the default recommended zero-shot classification model will be used.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[ZeroShotClassificationOutputElement]`</rettype><retdesc>List of [ZeroShotClassificationOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ZeroShotClassificationOutputElement) items containing the predicted labels and their confidence.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Provide as input a text and a set of candidate labels to classify the input text.











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.zero_shot_classification.example">

Example with `multi_label=False`:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()
>>> text = (
...     "A new model offers an explanation for how the Galilean satellites formed around the solar system's"
...     "largest world. Konstantin Batygin did not set out to solve one of the solar system's most puzzling"
...     " mysteries when he went for a run up a hill in Nice, France."
... )
>>> labels = ["space & cosmos", "scientific discovery", "microbiology", "robots", "archeology"]
>>> await client.zero_shot_classification(text, labels)
[
    ZeroShotClassificationOutputElement(label='scientific discovery', score=0.7961668968200684),
    ZeroShotClassificationOutputElement(label='space & cosmos', score=0.18570658564567566),
    ZeroShotClassificationOutputElement(label='microbiology', score=0.00730885099619627),
    ZeroShotClassificationOutputElement(label='archeology', score=0.006258360575884581),
    ZeroShotClassificationOutputElement(label='robots', score=0.004559356719255447),
]
>>> await client.zero_shot_classification(text, labels, multi_label=True)
[
    ZeroShotClassificationOutputElement(label='scientific discovery', score=0.9829297661781311),
    ZeroShotClassificationOutputElement(label='space & cosmos', score=0.755190908908844),
    ZeroShotClassificationOutputElement(label='microbiology', score=0.0005462635890580714),
    ZeroShotClassificationOutputElement(label='archeology', score=0.00047131875180639327),
    ZeroShotClassificationOutputElement(label='robots', score=0.00030448526376858354),
]
```

</ExampleCodeBlock>

<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.zero_shot_classification.example-2">

Example with `multi_label=True` and a custom `hypothesis_template`:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()
>>> await client.zero_shot_classification(
...    text="I really like our dinner and I'm very happy. I don't like the weather though.",
...    labels=["positive", "negative", "pessimistic", "optimistic"],
...    multi_label=True,
...    hypothesis_template="This text is {} towards the weather"
... )
[
    ZeroShotClassificationOutputElement(label='negative', score=0.9231801629066467),
    ZeroShotClassificationOutputElement(label='pessimistic', score=0.8760990500450134),
    ZeroShotClassificationOutputElement(label='optimistic', score=0.0008674879791215062),
    ZeroShotClassificationOutputElement(label='positive', score=0.0005250611575320363)
]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>zero_shot_image_classification</name><anchor>huggingface_hub.AsyncInferenceClient.zero_shot_image_classification</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/inference/_generated/_async_client.py#L3253</source><parameters>[{"name": "image", "val": ": typing.Union[bytes, typing.BinaryIO, str, pathlib.Path, ForwardRef('Image'), bytearray, memoryview]"}, {"name": "candidate_labels", "val": ": list"}, {"name": "model", "val": ": typing.Optional[str] = None"}, {"name": "hypothesis_template", "val": ": typing.Optional[str] = None"}, {"name": "labels", "val": ": list = None"}]</parameters><paramsdesc>- **image** (`Union[str, Path, bytes, BinaryIO, PIL.Image.Image]`) --
  The input image to caption. It can be raw bytes, an image file, a URL to an online image, or a PIL Image.
- **candidate_labels** (`list[str]`) --
  The candidate labels for this image
- **labels** (`list[str]`, *optional*) --
  (deprecated) List of string possible labels. There must be at least 2 labels.
- **model** (`str`, *optional*) --
  The model to use for inference. Can be a model ID hosted on the Hugging Face Hub or a URL to a deployed
  Inference Endpoint. This parameter overrides the model defined at the instance level. If not provided, the default recommended zero-shot image classification model will be used.
- **hypothesis_template** (`str`, *optional*) --
  The sentence used in conjunction with `candidate_labels` to attempt the image classification by
  replacing the placeholder with the candidate labels.</paramsdesc><paramgroups>0</paramgroups><rettype>`list[ZeroShotImageClassificationOutputElement]`</rettype><retdesc>List of [ZeroShotImageClassificationOutputElement](/docs/huggingface_hub/main/en/package_reference/inference_types#huggingface_hub.ZeroShotImageClassificationOutputElement) items containing the predicted labels and their confidence.</retdesc><raises>- [InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) -- 
  If the model is unavailable or the request times out.
- `HfHubHTTPError` -- 
  If the request fails with an HTTP error status code other than HTTP 503.</raises><raisederrors>[InferenceTimeoutError](/docs/huggingface_hub/main/en/package_reference/inference_client#huggingface_hub.InferenceTimeoutError) or `HfHubHTTPError`</raisederrors></docstring>

Provide input image and text labels to predict text labels for the image.











<ExampleCodeBlock anchor="huggingface_hub.AsyncInferenceClient.zero_shot_image_classification.example">

Example:
```py
# Must be run in an async context
>>> from huggingface_hub import AsyncInferenceClient
>>> client = AsyncInferenceClient()

>>> await client.zero_shot_image_classification(
...     "https://upload.wikimedia.org/wikipedia/commons/thumb/4/43/Cute_dog.jpg/320px-Cute_dog.jpg",
...     labels=["dog", "cat", "horse"],
... )
[ZeroShotImageClassificationOutputElement(label='dog', score=0.956),...]
```

</ExampleCodeBlock>


</div></div>

## InferenceTimeoutError[[huggingface_hub.InferenceTimeoutError]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class huggingface_hub.InferenceTimeoutError</name><anchor>huggingface_hub.InferenceTimeoutError</anchor><source>https://github.com/huggingface/huggingface_hub/blob/main/src/huggingface_hub/errors.py#L95</source><parameters>[{"name": "message", "val": ": str"}]</parameters></docstring>
Error raised when a model is unavailable or the request times out.

</div>

<EditOnGithub source="https://github.com/huggingface/huggingface_hub/blob/main/docs/source/en/package_reference/inference_client.md" />