# Optimization

🤗 Optimum Intel provides an `openvino` package that enables you to apply a variety of model quantization methods on many models hosted on the 🤗 hub using the [NNCF](https://docs.openvino.ai/2024/openvino-workflow/model-optimization.html) framework.


Quantization is a technique to reduce the computational and memory costs of running inference by representing the weights and / or the activations with lower precision data types like 8-bit or 4-bit.

## Optimization Support Matrix

Click on a ✅ to copy the command/code for the corresponding optimization case.

<div id="copyMsg" style="
    display: none;
    position: fixed;
    top: 20px;
    left: 50%;
    transform: translateX(-50%);
    background: rgba(60, 179, 113, 0.95);
    color: white;
    padding: 10px 20px;
    border-radius: 6px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    z-index: 1000;
    font-family: 'Segoe UI', sans-serif;
    font-size: 14px;
    font-weight: 500;
    letter-spacing: 0.3px;
    white-space: nowrap;
">
    Command copied to clipboard
</div>
<table>
    <thead>
        <tr>
            <th rowspan="3">Task<br>(OV Model Class)</th>
            <th colspan="4">Weight-only Quantization</th>
            <th colspan="2" rowspan="2">Hybrid Quantization</th>
            <th colspan="2" rowspan="2">Full Quantization</th>
            <th colspan="2" rowspan="2">Mixed Quantization</th>
        </tr>
        <tr>
            <th colspan="2">Data-free</th>
            <th colspan="2">Data-aware</th>
        </tr>
        <tr>
            <th>CLI</th>
            <th>Python</th>
            <th>CLI</th>
            <th>Python</th>
            <th>CLI</th>
            <th>Python</th>
            <th>CLI</th>
            <th>Python</th>
            <th>CLI</th>
            <th>Python</th>
        </tr>
    </thead>
    <tbody>
        <tr>
            <td style="text-align: center; vertical-align: middle;">text-generation<br>(OVModelForCausalLM)</td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino -m TinyLlama/TinyLlama-1.1B-Chat-v1.0 --weight-format int8 ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVModelForCausalLM.from_pretrained(\'TinyLlama/TinyLlama-1.1B-Chat-v1.0\', quantization_config=OVWeightQuantizationConfig(bits=8)).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino -m TinyLlama/TinyLlama-1.1B-Chat-v1.0 --weight-format int4 --dataset wikitext2 ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVModelForCausalLM.from_pretrained(\'TinyLlama/TinyLlama-1.1B-Chat-v1.0\', quantization_config=OVWeightQuantizationConfig(bits=4, dataset=\'wikitext2\')).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">-</td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino -m TinyLlama/TinyLlama-1.1B-Chat-v1.0 --quant-mode int8 --dataset wikitext2 ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVModelForCausalLM.from_pretrained(\'TinyLlama/TinyLlama-1.1B-Chat-v1.0\', quantization_config=OVQuantizationConfig(bits=8, dataset=\'wikitext2\')).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino -m TinyLlama/TinyLlama-1.1B-Chat-v1.0 --quant-mode nf4_f8e4m3 --dataset wikitext2 ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVModelForCausalLM.from_pretrained(\'TinyLlama/TinyLlama-1.1B-Chat-v1.0\', quantization_config=OVMixedQuantizationConfig(OVWeightQuantizationConfig(bits=4, dtype=\'nf4\'), OVQuantizationConfig(dtype=\'f8e4m3\', dataset=\'wikitext2\'))).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
        </tr>
        <tr>
            <td style="text-align: center; vertical-align: middle;">image-text-to-text<br>(OVModelForVisualCausalLM)</td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino --task image-text-to-text -m OpenGVLab/InternVL2-1B --trust-remote-code --weight-format int4 ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVModelForVisualCausalLM.from_pretrained(\'OpenGVLab/InternVL2-1B\', trust_remote_code=True, quantization_config=OVWeightQuantizationConfig(bits=4)).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino --task image-text-to-text -m OpenGVLab/InternVL2-1B --trust-remote-code --weight-format int4 --dataset contextual ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVModelForVisualCausalLM.from_pretrained(\'OpenGVLab/InternVL2-1B\', trust_remote_code=True, quantization_config=OVWeightQuantizationConfig(bits=4, dataset=\'contextual\', trust_remote_code=True)).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino --task image-text-to-text -m OpenGVLab/InternVL2-1B --trust-remote-code --quant-mode int8 --dataset contextual ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVModelForVisualCausalLM.from_pretrained(\'OpenGVLab/InternVL2-1B\', trust_remote_code=True, quantization_config=OVQuantizationConfig(bits=8, dataset=\'contextual\', trust_remote_code=True)).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">–</td>
        </tr>
        <tr>
            <td style="text-align: center; vertical-align: middle;">text-to-image, text-to-video<br>(OVDiffusionPipeline)</td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino -m dreamlike-art/dreamlike-anime-1.0 --weight-format int8 ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVDiffusionPipeline.from_pretrained(\'dreamlike-art/dreamlike-anime-1.0\', quantization_config=OVWeightQuantizationConfig(bits=8)).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino -m dreamlike-art/dreamlike-anime-1.0 --weight-format int8 --dataset conceptual_captions ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVDiffusionPipeline.from_pretrained(\'dreamlike-art/dreamlike-anime-1.0\', quantization_config=OVWeightQuantizationConfig(bits=8, quant_method=\'hybrid\', dataset=\'conceptual_captions\')).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino -m dreamlike-art/dreamlike-anime-1.0 --quant-mode int8 --dataset conceptual_captions ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVDiffusionPipeline.from_pretrained(\'dreamlike-art/dreamlike-anime-1.0\', quantization_config=OVQuantizationConfig(bits=8, dataset=\'conceptual_captions\')).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">–</td>
        </tr>
        <tr>
            <td style="text-align: center; vertical-align: middle;">automatic-speech-recognition<br>(OVModelForSpeechSeq2Seq)</td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino -m openai/whisper-large-v3-turbo --disable-stateful --quant-mode int8 --dataset librispeech --num-samples 10 ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVModelForSpeechSeq2Seq.from_pretrained(\'openai/whisper-large-v3-turbo\', stateful=False, quantization_config=OVQuantizationConfig(bits=8, dataset=\'librispeech\', num_samples=10)).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">–</td>
        </tr>
        <tr>
            <td style="text-align: center; vertical-align: middle;">feature-extraction<br>(OVModelForFeatureExtraction)</td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino -m microsoft/codebert-base --weight-format int8 ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVModelForFeatureExtraction.from_pretrained(\'microsoft/codebert-base\', quantization_config=OVWeightQuantizationConfig(bits=8)).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino -m microsoft/codebert-base --weight-format int4 --dataset wikitext2 ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVModelForFeatureExtraction.from_pretrained(\'microsoft/codebert-base\', quantization_config=OVWeightQuantizationConfig(bits=4, dataset=\'wikitext2\')).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">-</td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino -m microsoft/codebert-base --quant-mode int8 --dataset wikitext2 ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVModelForFeatureExtraction.from_pretrained(\'microsoft/codebert-base\', quantization_config=OVQuantizationConfig(bits=8, dataset=\'wikitext2\')).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino -m microsoft/codebert-base --quant-mode nf4_f8e4m3 --dataset wikitext2 ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVModelForFeatureExtraction.from_pretrained(\'microsoft/codebert-base\', quantization_config=OVMixedQuantizationConfig(OVWeightQuantizationConfig(bits=4, dtype=\'nf4\'), OVQuantizationConfig(dtype=\'f8e4m3\', dataset=\'wikitext2\'))).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
        </tr>
        <tr>
            <td style="text-align: center; vertical-align: middle;">feature-extraction<br>(OVSentenceTransformer)</td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino --library sentence_transformers -m sentence-transformers/all-mpnet-base-v2 --weight-format int8 ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVSentenceTransformer.from_pretrained(\'sentence-transformers/all-mpnet-base-v2\', quantization_config=OVWeightQuantizationConfig(bits=8)).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino --library sentence_transformers -m sentence-transformers/all-mpnet-base-v2 --weight-format int4 --dataset wikitext2 ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVSentenceTransformer.from_pretrained(\'sentence-transformers/all-mpnet-base-v2\', quantization_config=OVWeightQuantizationConfig(bits=4, dataset=\'wikitext2\')).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">-</td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino --library sentence_transformers -m sentence-transformers/all-mpnet-base-v2 --quant-mode int8 --dataset wikitext2 ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVSentenceTransformer.from_pretrained(\'sentence-transformers/all-mpnet-base-v2\', quantization_config=OVQuantizationConfig(bits=8, dataset=\'wikitext2\')).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino --library sentence_transformers -m sentence-transformers/all-mpnet-base-v2 --quant-mode nf4_f8e4m3 --dataset wikitext2 ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVSentenceTransformer.from_pretrained(\'sentence-transformers/all-mpnet-base-v2\', quantization_config=OVMixedQuantizationConfig(OVWeightQuantizationConfig(bits=4, dtype=\'nf4\'), OVQuantizationConfig(dtype=\'f8e4m3\', dataset=\'wikitext2\'))).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
        </tr>
        <tr>
            <td style="text-align: center; vertical-align: middle;">fill-mask<br>(OVModelForMaskedLM)</td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino -m FacebookAI/roberta-base --weight-format int8 ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVModelForMaskedLM.from_pretrained(\'FacebookAI/roberta-base\', quantization_config=OVWeightQuantizationConfig(bits=8)).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino -m FacebookAI/roberta-base --weight-format int4 --dataset wikitext2 ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVModelForMaskedLM.from_pretrained(\'FacebookAI/roberta-base\', quantization_config=OVWeightQuantizationConfig(bits=4, dataset=\'wikitext2\')).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">-</td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino -m FacebookAI/roberta-base --quant-mode int8 --dataset wikitext2 ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVModelForMaskedLM.from_pretrained(\'FacebookAI/roberta-base\', quantization_config=OVQuantizationConfig(bits=8, dataset=\'wikitext2\')).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino -m FacebookAI/roberta-base --quant-mode nf4_f8e4m3 --dataset wikitext2 ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVModelForMaskedLM.from_pretrained(\'FacebookAI/roberta-base\', quantization_config=OVMixedQuantizationConfig(OVWeightQuantizationConfig(bits=4, dtype=\'nf4\'), OVQuantizationConfig(dtype=\'f8e4m3\', dataset=\'wikitext2\'))).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
        </tr>
        <tr>
            <td style="text-align: center; vertical-align: middle;">text2text-generation<br>(OVModelForSeq2SeqLM)</td>
            <td style="text-align: center; vertical-align: middle;">
                <button
                    onclick="navigator.clipboard.writeText('optimum-cli export openvino -m google-t5/t5-small --weight-format int8 ./save_dir')">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button
                    onclick="navigator.clipboard.writeText('OVModelForSeq2SeqLM.from_pretrained(\'google-t5/t5-small\', quantization_config=OVWeightQuantizationConfig(bits=8)).save_pretrained(\'save_dir\')')">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button
                    onclick="navigator.clipboard.writeText('optimum-cli export openvino -m google-t5/t5-small --weight-format int4 --dataset wikitext2 ./save_dir')">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button
                    onclick="navigator.clipboard.writeText('OVModelForSeq2SeqLM.from_pretrained(\'google-t5/t5-small\', quantization_config=OVWeightQuantizationConfig(bits=4, dataset=\'wikitext2\')).save_pretrained(\'save_dir\')')">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">-</td>
            <td style="text-align: center; vertical-align: middle;">
                <button
                    onclick="navigator.clipboard.writeText('optimum-cli export openvino -m google-t5/t5-small --quant-mode int8 --dataset wikitext2 --smooth-quant-alpha -1 ./save_dir')">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button
                    onclick="navigator.clipboard.writeText('OVModelForSeq2SeqLM.from_pretrained(\'google-t5/t5-small\', quantization_config=OVQuantizationConfig(bits=8, dataset=\'wikitext2\', smooth_quant_alpha=-1)).save_pretrained(\'save_dir\')')">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button
                    onclick="navigator.clipboard.writeText('optimum-cli export openvino -m google-t5/t5-small --quant-mode nf4_f8e4m3 --dataset wikitext2 --smooth-quant-alpha -1 ./save_dir')">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button
                    onclick="navigator.clipboard.writeText('OVModelForSeq2SeqLM.from_pretrained(\'google-t5/t5-small\', quantization_config=OVMixedQuantizationConfig(OVWeightQuantizationConfig(bits=4, dtype=\'nf4\'), OVQuantizationConfig(dtype=\'f8e4m3\', dataset=\'wikitext2\', smooth_quant_alpha=-1))).save_pretrained(\'save_dir\')')">
                    ✅
                </button>
            </td>
        </tr>
        <tr>
            <td style="text-align: center; vertical-align: middle;">zero-shot-image-classification<br>(OVModelForZeroShotImageClassification)</td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino -m openai/clip-vit-base-patch16 --weight-format int8 ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVModelForZeroShotImageClassification.from_pretrained(\'openai/clip-vit-base-patch16\', quantization_config=OVWeightQuantizationConfig(bits=8)).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino -m openai/clip-vit-base-patch16 --weight-format int4 --dataset conceptual_captions ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVModelForZeroShotImageClassification.from_pretrained(\'openai/clip-vit-base-patch16\', quantization_config=OVWeightQuantizationConfig(bits=4, dataset=\'conceptual_captions\')).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">-</td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino -m openai/clip-vit-base-patch16 --quant-mode int8 --dataset conceptual_captions ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVModelForZeroShotImageClassification.from_pretrained(\'openai/clip-vit-base-patch16\', quantization_config=OVQuantizationConfig(bits=8, dataset=\'conceptual_captions\')).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino -m openai/clip-vit-base-patch16 --quant-mode nf4_f8e4m3 --dataset conceptual_captions ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVModelForZeroShotImageClassification.from_pretrained(\'openai/clip-vit-base-patch16\', quantization_config=OVMixedQuantizationConfig(OVWeightQuantizationConfig(bits=4, dtype=\'nf4\'), OVQuantizationConfig(dtype=\'f8e4m3\', dataset=\'conceptual_captions\'))).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
        </tr>
        <tr>
            <td style="text-align: center; vertical-align: middle;">feature-extraction<br>(OVSamModel)</td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVSamModel.from_pretrained(\'facebook/sam-vit-base\', quantization_config=OVPipelineQuantizationConfig(quantization_configs=dict(vision_encoder=OVWeightQuantizationConfig(bits=8)))).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVSamModel.from_pretrained(\'facebook/sam-vit-base\', quantization_config=OVPipelineQuantizationConfig(quantization_configs=dict(vision_encoder=OVWeightQuantizationConfig(bits=4, dataset=\'coco\')))).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">-</td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('optimum-cli export openvino -m facebook/sam-vit-base --quant-mode int8 --dataset coco ./save_dir');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVSamModel.from_pretrained(\'facebook/sam-vit-base\', quantization_config=OVPipelineQuantizationConfig(quantization_configs=dict(vision_encoder=OVQuantizationConfig(bits=8, dataset=\'coco\')))).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">–</td>
        </tr>
        <tr>
            <td style="text-align: center; vertical-align: middle;">text-to-audio<br>(OVModelForTextToSpeechSeq2Seq)</td>
            <td style="text-align: center; vertical-align: middle;">✅</td>
            <td style="text-align: center; vertical-align: middle;">
                <button onclick="
                    navigator.clipboard.writeText('OVModelForTextToSpeechSeq2Seq.from_pretrained(\'microsoft/speecht5_tts\', vocoder=\'microsoft/speecht5_hifigan\', quantization_config=OVWeightQuantizationConfig(bits=8)).save_pretrained(\'save_dir\')');
                    let m=document.getElementById('copyMsg');
                    m.style.display='block';
                    clearTimeout(window._copyTimeout);
                    window._copyTimeout=setTimeout(()=>m.style.display='none', 2000);
                ">
                    ✅
                </button>
            </td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">–</td>
            <td style="text-align: center; vertical-align: middle;">–</td>
        </tr>
    </tbody>
</table>


## Weight-only Quantization

Quantization can be applied on the model's Linear, Convolutional and Embedding layers, enabling the loading of large models on memory-limited devices. For example, when applying 8-bit quantization, the resulting model will be x4 smaller than its fp32 counterpart. For 4-bit quantization, the reduction in memory could theoretically reach x8, but is closer to x6 in practice.


### 8-bit

For the 8-bit weight quantization you can provide `quantization_config` equal to `OVWeightQuantizationConfig(bits=8)` to load your model's weights in 8-bit:

```python
from optimum.intel import OVModelForCausalLM, OVWeightQuantizationConfig

model_id = "helenai/gpt2-ov"
quantization_config = OVWeightQuantizationConfig(bits=8)
model = OVModelForCausalLM.from_pretrained(model_id, quantization_config=quantization_config)

# Saves the int8 model that will be x4 smaller than its fp32 counterpart
model.save_pretrained(saving_directory)
```

Weights of language models inside vision-language pipelines can be quantized in a similar way:
```python
model = OVModelForVisualCausalLM.from_pretrained(
    "llava-hf/llava-v1.6-mistral-7b-hf",
    quantization_config=quantization_config
)
```

<Tip warning={true}>

If quantization_config is not provided, model will be exported in 8 bits by default when it has more than 1 billion parameters. You can disable it with `load_in_8bit=False`.

</Tip>


### 4-bit

4-bit weight quantization can be achieved in a similar way:

```python
from optimum.intel import OVModelForCausalLM, OVWeightQuantizationConfig

quantization_config = OVWeightQuantizationConfig(bits=4)
model = OVModelForCausalLM.from_pretrained(model_id, quantization_config=quantization_config)
```

Or for vision-language pipelines:
```python
model = OVModelForVisualCausalLM.from_pretrained(
    "llava-hf/llava-v1.6-mistral-7b-hf",
    quantization_config=quantization_config
)
```

You can tune quantization parameters to achieve a better performance accuracy trade-off as follows:

```python
quantization_config = OVWeightQuantizationConfig(
    bits=4,
    sym=False,
    ratio=0.8,
    quant_method="awq",
    dataset="wikitext2"
)
```

Note: `OVWeightQuantizationConfig` also accepts keyword arguments that are not listed in its constructor. In this case such arguments will be passed directly to `nncf.compress_weights()` call. This is useful for passing additional parameters to the quantization algorithm.

By default the quantization scheme will be [asymmetric](https://github.com/openvinotoolkit/nncf/blob/develop/docs/usage/training_time_compression/other_algorithms/LegacyQuantization.md#asymmetric-quantization), to make it [symmetric](https://github.com/openvinotoolkit/nncf/blob/develop/docs/usage/training_time_compression/other_algorithms/LegacyQuantization.md#symmetric-quantization) you can add `sym=True`.

For 4-bit quantization you can also specify the following arguments in the quantization configuration :
* The `group_size` parameter will define the group size to use for quantization, `-1` it will results in per-column quantization.
* The `ratio` parameter controls the ratio between 4-bit and 8-bit quantization. If set to 0.9, it means that 90% of the layers will be quantized to `int4` while 10% will be quantized to `int8`.

Smaller `group_size` and `ratio` values usually improve accuracy at the sacrifice of the model size and inference latency.

Quality of 4-bit weight compressed model can further be improved by employing one of the following data-dependent methods:
* **AWQ** which stands for Activation Aware Quantization is an algorithm that tunes model weights for more accurate 4-bit compression. It slightly improves generation quality of compressed LLMs, but requires significant additional time and memory for tuning weights on a calibration dataset. Please note that it is possible that there will be no matching patterns in the model to apply AWQ, in such case it will be skipped. There is also a data-free version of AWQ available that relies on per-column magnitudes of weights instead of activations.
* **Scale Estimation** is a method that tunes quantization scales to minimize the `L2` error between the original and compressed layers. Providing a dataset is required to run scale estimation. Using this method also incurs additional time and memory overhead.
* **GPTQ** optimizes compressed weights in a layer-wise fashion to minimize the difference between activations of a compressed and original layer.
* **LoRA Correction** mitigates quantization noise introduced during weight compression by leveraging low-rank adaptation.

Data-aware algorithms can be applied together or separately. For that, provide corresponding arguments to the 4-bit `OVWeightQuantizationConfig` together with a dataset. For example:
```python
quantization_config = OVWeightQuantizationConfig(
    bits=4,
    sym=False,
    ratio=0.8,
    quant_method="awq",
    scale_estimation=True,
    gptq=True,
    dataset="wikitext2"
)
```

Note: GPTQ and LoRA Correction algorithms can't be applied simultaneously.

## Full quantization

When applying post-training full quantization, both the weights and the activations are quantized.
To apply quantization on the activations, an additional calibration step is needed which consists in feeding a `calibration_dataset` to the network in order to estimate the quantization activations parameters.

Here is how to apply full quantization on a fine-tuned DistilBERT given your own `calibration_dataset`:

```python
from transformers import AutoTokenizer
from optimum.intel import OVQuantizer, OVModelForSequenceClassification, OVConfig, OVQuantizationConfig

model_id = "distilbert-base-uncased-finetuned-sst-2-english"
model = OVModelForSequenceClassification.from_pretrained(model_id, export=True)
tokenizer = AutoTokenizer.from_pretrained(model_id)
# The directory where the quantized model will be saved
save_dir = "ptq_model"

quantizer = OVQuantizer.from_pretrained(model)

# Apply full quantization and export the resulting quantized model to OpenVINO IR format
ov_config = OVConfig(quantization_config=OVQuantizationConfig())
quantizer.quantize(ov_config=ov_config, calibration_dataset=calibration_dataset, save_directory=save_dir)
# Save the tokenizer
tokenizer.save_pretrained(save_dir)
```

The calibration dataset can also be created easily using your `OVQuantizer`:

```python
from functools import partial

def preprocess_function(examples, tokenizer):
    return tokenizer(examples["sentence"], padding="max_length", max_length=128, truncation=True)

# Create the calibration dataset used to perform full quantization
calibration_dataset = quantizer.get_calibration_dataset(
    "glue",
    dataset_config_name="sst2",
    preprocess_function=partial(preprocess_function, tokenizer=tokenizer),
    num_samples=300,
    dataset_split="train",
)
```


The `quantize()` method applies post-training quantization and export the resulting quantized model to the OpenVINO Intermediate Representation (IR). The resulting graph is represented with two files: an XML file describing the network topology and a binary file describing the weights. The resulting model can be run on any target Intel device.


### Speech-to-text Models Quantization

The speech-to-text Whisper model can be quantized without the need for preparing a custom calibration dataset. Please see example below.

```python
model_id = "openai/whisper-tiny"
ov_model = OVModelForSpeechSeq2Seq.from_pretrained(
    model_id,
    stateful=False,
    quantization_config=OVQuantizationConfig(
        num_samples=10,
        dataset="librispeech",
        processor=model_id,
        smooth_quant_alpha=0.95,
    )
)
```

With this, encoder, decoder and decoder-with-past models of the Whisper pipeline will be fully quantized, including activations.

##  Hybrid quantization

Traditional optimization methods like post-training 8-bit quantization do not work well for Stable Diffusion (SD) models and can lead to poor generation results. On the other hand, weight compression does not improve performance significantly when applied to Stable Diffusion models, as the size of activations is comparable to weights.
The U-Net component takes up most of the overall execution time of the pipeline. Thus, optimizing just this one component can bring substantial benefits in terms of inference speed while keeping acceptable accuracy without fine-tuning. Quantizing the rest of the diffusion pipeline does not significantly improve inference performance but could potentially lead to substantial accuracy degradation.
Therefore, the proposal is to apply quantization in *hybrid mode* for the U-Net model and weight-only quantization for the rest of the pipeline components :
* U-Net : quantization applied on both the weights and activations 
* The text encoder, VAE encoder / decoder : quantization applied on the weights 

The hybrid mode involves the quantization of weights in MatMul and Embedding layers, and activations of other layers, facilitating accuracy preservation post-optimization while reducing the model size.

The `quantization_config` is utilized to define optimization parameters for optimizing the SD pipeline. To enable hybrid quantization, specify the quantization dataset in the `quantization_config`. If the dataset is not defined, weight-only quantization will be applied on all components.

```python
from optimum.intel import OVStableDiffusionPipeline, OVWeightQuantizationConfig

model = OVStableDiffusionPipeline.from_pretrained(
    model_id,
    export=True,
    quantization_config=OVWeightQuantizationConfig(bits=8, dataset="conceptual_captions"),
)
```


For more details, please refer to the corresponding NNCF [documentation](https://github.com/openvinotoolkit/nncf/blob/develop/docs/usage/post_training_compression/weights_compression/Usage.md).


## Mixed Quantization

Mixed quantization is a technique that combines weight-only quantization with full quantization. During mixed quantization we separately quantize:
1. weights of weighted layers to one precision, and
2. activations (and possibly, weights, if some were skipped at the first step) of other supported layers to another precision.

By default, weights of all weighted layers are quantized in the first step. In the second step activations of weighted and non-weighted layers are quantized. If some layers are instructed to be ignored in the first step with `weight_quantization_config.ignored_scope` parameter, both weights and activations of these layers are quantized to the precision given in the `full_quantization_config`.

When running this kind of optimization through Python API, `OVMixedQuantizationConfig` should be used. In such case the precision for the first step should be provided with `weight_quantization_config` argument and the precision for the second step with `full_quantization_config` argument. For example:

```python
model = OVModelForCausalLM.from_pretrained(
    'TinyLlama/TinyLlama-1.1B-Chat-v1.0',
    quantization_config=OVMixedQuantizationConfig(
        weight_quantization_config=OVWeightQuantizationConfig(bits=4, dtype='nf4'),
        full_quantization_config=OVQuantizationConfig(dtype='f8e4m3', dataset='wikitext2')
    )
)
```

To apply mixed quantization through CLI, the `--quant-mode` argument should be used. For example:

```bash
optimum-cli export openvino -m TinyLlama/TinyLlama-1.1B-Chat-v1.0 --quant-mode nf4_f8e4m3 --dataset wikitext2 ./save_dir
```

Don't forget to provide a dataset since it is required for the calibration procedure during full quantization.


## Pipeline Quantization

There are multimodal pipelines that consist of multiple components, such as Stable Diffusion or Visual Language models. In these cases, there may be a need to apply different quantization methods to different components of the pipeline. For example, you may want to apply int4 data-aware weight-only quantization to a language model in visual-language pipeline, while applying int8 weight-only quantization to other components. In this case you can use the `OVPipelineQuantizationConfig` class to specify the quantization configuration for each component of the pipeline.

For example, the code below quantizes weights and activations of a language model inside InternVL2-1B, compresses weights of a text embedding model and skips any quantization for vision embedding model.

```python
from optimum.intel import OVModelForVisualCausalLM
from optimum.intel import OVPipelineQuantizationConfig, OVQuantizationConfig, OVWeightQuantizationConfig

model_id = "OpenGVLab/InternVL2-1B"
model = OVModelForVisualCausalLM.from_pretrained(
    model_id,
    export=True,
    trust_remote_code=True,
    quantization_config=OVPipelineQuantizationConfig(
        quantization_configs={
            "lm_model": OVQuantizationConfig(bits=8),
            "text_embeddings_model": OVWeightQuantizationConfig(bits=8),
        },
        dataset="contextual",
        trust_remote_code=True,
    )
)
```