# MCP Clients

Now that we have a basic understanding of the Model Context Protocol, we can explore the essential role of MCP Clients in the Model Context Protocol ecosystem.

 In this part of Unit 1, we'll explore the essential role of MCP Clients in the Model Context Protocol ecosystem.

In this section, you will:

* Understand what MCP Clients are and their role in the MCP architecture
* Learn about the key responsibilities of MCP Clients
* Explore the major MCP Client implementations
* Discover how to connect to the Hugging Face MCP Server and built-in tools
* See practical examples of MCP Client usage

> [!TIP]
> In this page we're going to show examples of how to set up MCP Clients in a few different ways using the JSON notation. For now, we will use *examples* like `path/to/server.py` to represent the path to the MCP Server. In the next unit, we'll implement this with real MCP Servers.  
>
> For now, focus on understanding the MCP Client notation. We'll implement the MCP Servers in the next unit.

## Understanding MCP Clients

MCP Clients are crucial components that act as the bridge between AI applications (Hosts) and external capabilities provided by MCP Servers. Think of the Host as your main application (like an AI assistant or IDE) and the Client as a specialized module within that Host responsible for handling MCP communications.

## User Interface Client

Let's start by exploring the user interface clients that are available for the MCP.

### Chat Interface Clients

- Claude Desktop (Anthropic)

### Interactive Development Clients

- VS Code extensions with MCP (e.g., Continue)
- Cursor IDE (built-in MCP client)
- Zed editor

These clients support connecting to multiple MCP servers and real-time tool invocation.

## Quick connect to Hugging Face MCP Server

Hugging Face provides a hosted MCP server with built-in tools for exploring models, datasets, Spaces, and papers.

1. Visit https://huggingface.co/settings/mcp while logged in.
2. Select your MCP-compatible client (e.g., VS Code, Cursor, Zed, Claude Desktop).
3. Copy the generated configuration snippet into your client's MCP config.
4. Restart or reload your client. You should see “Hugging Face” connected.

Tip: Prefer the generated snippet over hand-written config; it’s tailored per client.

## Configuring MCP Clients

Now that we've covered the core of the MCP protocol, let's look at how to configure your MCP servers and clients.

Effective deployment of MCP servers and clients requires proper configuration. 

> [!TIP]
> The MCP specification is still evolving, so the configuration methods are subject to evolution. We'll focus on the current best practices for configuration.

### MCP Configuration Files

MCP hosts use configuration files to manage server connections. These files define which servers are available and how to connect to them.

Fortunately, the configuration files are very simple, easy to understand, and consistent across major MCP hosts.

#### `mcp.json` Structure

The standard configuration file for MCP is named `mcp.json`. Here's the basic structure:

This is the basic structure of the `mcp.json` can be passed to applications like Claude Desktop, Cursor, or VS Code.

```json
{
  "servers": [
    {
      "name": "Server Name",
      "transport": {
        "type": "stdio|sse",
        // Transport-specific configuration
      }
    }
  ]
}
```

In this example, we have a single server with a name and a transport type. The transport type is either `stdio` or `sse`.

#### Configuration for stdio Transport

For local servers using stdio transport, the configuration includes the command and arguments to launch the server process:

```json
{
  "servers": [
    {
      "name": "File Explorer",
      "transport": {
        "type": "stdio",
        "command": "python",
        "args": ["/path/to/file_explorer_server.py"] // This is an example, we'll use a real server in the next unit
      }
    }
  ]
}
```

Here, we have a server called "File Explorer" that is a local script.

#### Configuration for HTTP+SSE Transport

For remote servers using HTTP+SSE transport, the configuration includes the server URL:

```json
{
  "servers": [
    {
      "name": "Remote API Server",
      "transport": {
        "type": "sse",
        "url": "https://example.com/mcp-server"
      }
    }
  ]
}
```

#### Environment Variables in Configuration

Environment variables can be passed to server processes using the `env` field. Here's how to access them in your server code:

<hfoptions id="env-variables">
<hfoption id="python">

In Python, we use the `os` module to access environment variables:

```python
import os

# Access environment variables
github_token = os.environ.get("GITHUB_TOKEN")
if not github_token:
    raise ValueError("GITHUB_TOKEN environment variable is required")

# Use the token in your server code
def make_github_request():
    headers = {"Authorization": f"Bearer {github_token}"}
    # ... rest of your code
```

</hfoption>
<hfoption id="javascript">

In JavaScript, we use the `process.env` object to access environment variables:

```javascript
// Access environment variables
const githubToken = process.env.GITHUB_TOKEN;
if (!githubToken) {
    throw new Error("GITHUB_TOKEN environment variable is required");
}

// Use the token in your server code
function makeGithubRequest() {
    const headers = { "Authorization": `Bearer ${githubToken}` };
    // ... rest of your code
}
```

</hfoption>
</hfoptions>

The corresponding configuration in `mcp.json` would look like this:

```json
{
  "servers": [
    {
      "name": "GitHub API",
      "transport": {
        "type": "stdio",
        "command": "python",
        "args": ["/path/to/github_server.py"], // This is an example, we'll use a real server in the next unit
        "env": {
          "GITHUB_TOKEN": "your_github_token"
        }
      }
    }
  ]
}
```

### Configuration Examples

Let's look at some real-world configuration scenarios:

#### Scenario 1: Local Server Configuration

In this scenario, we have a local server that is a Python script which could be a file explorer or a code editor.

```json
{
  "servers": [
    {
      "name": "File Explorer",
      "transport": {
        "type": "stdio",
        "command": "python",
        "args": ["/path/to/file_explorer_server.py"] // This is an example, we'll use a real server in the next unit
      }
    }
  ]
}
```

#### Scenario 2: Remote Server Configuration

In this scenario, we have a remote server that is a weather API.

```json
{
  "servers": [
    {
      "name": "Weather API",
      "transport": {
        "type": "sse",
        "url": "https://example.com/mcp-server" // This is an example, we'll use a real server in the next unit
      }
    }
  ]
}
```

Proper configuration is essential for successfully deploying MCP integrations. By understanding these aspects, you can create robust and reliable connections between AI applications and external capabilities.

In the next section, we'll explore the ecosystem of MCP servers available on Hugging Face Hub and how to publish your own servers there. 

## Tiny Agents Clients

Now, let's explore how to use MCP Clients within code.

You can also use tiny agents as MCP Clients to connect directly to MCP servers from your code. Tiny agents provide a simple way to create AI agents that can use tools from MCP servers.

Tiny Agent can run MCP servers with a command line environment. To do this, we will need to install `npm` and run the server with `npx`. **We'll need these for both Python and JavaScript.**

Let's install `npx` with `npm`. If you don't have `npm` installed, check out the [npm documentation](https://docs.npmjs.com/downloading-and-installing-node-js-and-npm).

### Setup

First, we will need to install `npx` if you don't have it installed. You can do this with the following command:

```bash
# install npx
npm install -g npx
```

Then, we will need to install the huggingface_hub package with the MCP support. This will allow us to run MCP servers and clients.

```bash
pip install "huggingface_hub[mcp]>=0.32.0"
```

Then, we will need to log in to the Hugging Face Hub to access the MCP servers. You can do this with the `huggingface-cli` command line tool. You will need a [login token](https://huggingface.co/docs/huggingface_hub/v0.32.3/en/quick-start#authentication) to do this.

```bash
huggingface-cli login
```
### Configure Access Token Permissions

After creating your Hugging Face access token and logging in, you need to ensure your token has the proper permissions to work with inference providers.

> [!WARNING]
> **Important:** If you skip this step, you may encounter authentication errors when running tiny agents with hosted models.

1. Go to your [Hugging Face Access Tokens page](https://huggingface.co/settings/tokens)
2. Find your MCP token and click the three dots (⋮) next to it
3. Select **"Edit permissions"**
4. Under the **Inference** section, check the box for:
   - **"Make calls to Inference Providers"**
5. Save your changes

This permission is required because tiny agents need to make API calls to hosted models like `Qwen/Qwen2.5-72B-Instruct` through providers like Nebius.


<hfoptions id="language">
<hfoption id="python">

### Connecting to MCP Servers

Now, let's create an agent configuration file `agent.json`.

```json
{
    "name": "playwright-agent",
    "description": "Agent with Playwright MCP server",
    "model": "Qwen/Qwen2.5-72B-Instruct",
    "provider": "nebius",
    "servers": [
        {
            "type": "stdio",
            "command": "npx",
            "args": ["@playwright/mcp@latest"]
        }
    ]
}
```

In this configuration, we are using the `@playwright/mcp` MCP server. This is an MCP server that can control a browser with Playwright.

Now you can run the agent:

```bash
tiny-agents run agent.json
```
</hfoption>
<hfoption id="javascript">

First, install the tiny agents package with [npm](https://docs.npmjs.com/downloading-and-installing-node-js-and-npm).

```bash
npm install @huggingface/tiny-agents
```

### Connecting to MCP Servers

Make an agent project directory and create an `agent.json` file.

```bash
mkdir my-agent
touch my-agent/agent.json
```

Create an agent configuration file at `my-agent/agent.json`:

```json
{
    "model": "Qwen/Qwen2.5-72B-Instruct",
    "provider": "nebius",
    "servers": [
        {
            "type": "stdio",
            "command": "npx",
            "args": ["@playwright/mcp@latest"]
        }
    ]
}
```

Now you can run the agent:

```bash
npx @huggingface/tiny-agents run ./my-agent
```

</hfoption>
</hfoptions>

In the video below, we run the agent and ask it to open a new tab in the browser.

The following example shows a web-browsing agent configured to use the [Qwen/Qwen2.5-72B-Instruct](https://huggingface.co/Qwen/Qwen2.5-72B-Instruct) model via Nebius inference provider, and it comes equipped with a playwright MCP server, which lets it use a web browser! The agent config is loaded specifying [its path in the `tiny-agents/tiny-agents`](https://huggingface.co/datasets/tiny-agents/tiny-agents/tree/main/celinah/web-browser) Hugging Face dataset.

<video controls autoplay loop>
  <source src="https://huggingface.co/datasets/huggingface/documentation-images/resolve/main/blog/python-tiny-agents/web_browser_agent.mp4" type="video/mp4">
</video>

When you run the agent, you'll see it load, listing the tools it has discovered from its connected MCP servers. Then, it's ready for your prompts!

Prompt used in this demo:

> do a Web Search for HF inference providers on Brave Search and open the first result and then give me the list of the inference providers supported on Hugging Face 

## Next Steps

Now that you understand MCP Clients, you're ready to:
* Explore specific MCP Server implementations
* Learn about creating custom MCP Clients
* Dive into advanced MCP integration patterns

Let's continue our journey into the world of Model Context Protocol!


<EditOnGithub source="https://github.com/huggingface/mcp-course/blob/main/units/en/unit1/mcp-clients.mdx" />