# Biblioteca de Agente de Prueba

<img src="https://huggingface.co/datasets/agents-course/course-images/resolve/main/en/unit1/whiteboard-unit1sub3DONE.jpg" alt="Unit 1 planning"/>

Este curso es agnóstico en cuanto al framework porque queremos **centrarnos en los conceptos de agentes de IA y evitar perdernos en los detalles específicos de un framework particular**.

Además, queremos que los estudiantes puedan utilizar los conceptos que aprenden en este curso en sus propios proyectos, usando cualquier framework que prefieran.

Por lo tanto, para esta Unidad 1, utilizaremos una biblioteca de agentes de prueba y una API serverless simple para acceder a nuestro motor LLM.

Probablemente no usarías estos en producción, pero servirán como un buen **punto de partida para entender cómo funcionan los agentes**.

Después de esta sección, estarás listo para **crear un Agente simple** usando `smolagents`

Y en las siguientes Unidades también utilizaremos otras bibliotecas de Agentes de IA como `LangGraph` y `LlamaIndex`.

Para mantener las cosas simples, utilizaremos una función simple de Python como Herramienta y Agente.

Utilizaremos paquetes integrados de Python como `datetime` y `os` para que puedas probarlo en cualquier entorno.

Puedes seguir el proceso [en este notebook](https://huggingface.co/agents-course/notebooks/blob/main/unit1/dummy_agent_library.ipynb) y **ejecutar el código tú mismo**.

## API Serverless

En el ecosistema de Hugging Face, hay una característica conveniente llamada API Serverless que te permite ejecutar fácilmente inferencia en muchos modelos. No se requiere instalación ni despliegue.

```python
import os
from huggingface_hub import InferenceClient

## Necesitas un token de https://hf.co/settings/tokens, asegúrate de seleccionar 'read' como tipo de token. Si ejecutas esto en Google Colab, puedes configurarlo en la pestaña "settings" bajo "secrets". Asegúrate de llamarlo "HF_TOKEN"
os.environ["HF_TOKEN"]="hf_xxxxxxxxxxxxxx"

client = InferenceClient(provider="hf-inference", model="meta-llama/Llama-3.3-70B-Instruct")
# si las salidas para las siguientes celdas son incorrectas, el modelo gratuito puede estar sobrecargado. También puedes usar este endpoint público que contiene Llama-3.2-3B-Instruct
# client = InferenceClient("https://jc26mwg228mkj8dw.us-east-1.aws.endpoints.huggingface.cloud")
```

```python
output = client.text_generation(
    "The capital of France is",
    max_new_tokens=100,
)

print(output)
```
output:
```
Paris. The capital of France is Paris. The capital of France is Paris. The capital of France is Paris. The capital of France is Paris. The capital of France is Paris. The capital of France is Paris. The capital of France is Paris. The capital of France is Paris. The capital of France is Paris. The capital of France is Paris. The capital of France is Paris. The capital of France is Paris. The capital of France is Paris. The capital of France is Paris.
```
Como vimos en la sección de LLM, si solo hacemos decodificación, **el modelo solo se detendrá cuando prediga un token EOS**, y esto no sucede aquí porque este es un modelo conversacional (chat) y **no aplicamos la plantilla de chat que espera**.

Si ahora agregamos los tokens especiales relacionados con el <a href="https://huggingface.co/meta-llama/Llama-3.2-3B-Instruct">modelo Llama-3.2-3B-Instruct</a> que estamos usando, el comportamiento cambia y ahora produce el EOS esperado.

```python
prompt="""<|begin_of_text|><|start_header_id|>user<|end_header_id|>
The capital of France is<|eot_id|><|start_header_id|>assistant<|end_header_id|>"""
output = client.text_generation(
    prompt,
    max_new_tokens=100,
)

print(output)
```
output:
```
The capital of France is Paris.
```

Usar el método "chat" es una forma mucho más conveniente y confiable de aplicar plantillas de chat:
```python
output = client.chat.completions.create(
    messages=[
        {"role": "user", "content": "The capital of France is"},
    ],
    stream=False,
    max_tokens=1024,
)
print(output.choices[0].message.content)
```
output:
```
Paris.
```
El método chat es el método RECOMENDADO para usar para asegurar una transición suave entre modelos, pero como este notebook es solo educativo, seguiremos usando el método "text_generation" para entender los detalles.

## Agente de Prueba

En las secciones anteriores, vimos que el núcleo de una biblioteca de agentes es agregar información en el prompt del sistema.

Este prompt del sistema es un poco más complejo que el que vimos anteriormente, pero ya contiene:

1. **Información sobre las herramientas**
2. **Instrucciones del ciclo** (Pensamiento → Acción → Observación)

```
Responde las siguientes preguntas lo mejor que puedas. Tienes acceso a las siguientes herramientas:

get_weather: Obtener el clima actual en una ubicación dada

La forma en que usas las herramientas es especificando un blob json.
Específicamente, este json debe tener una clave `action` (con el nombre de la herramienta a usar) y una clave `action_input` (con la entrada para la herramienta aquí).

Los únicos valores que deberían estar en el campo "action" son:
get_weather: Obtener el clima actual en una ubicación dada, args: {"location": {"type": "string"}}
ejemplo de uso:

{{
  "action": "get_weather",
  "action_input": {"location": "New York"}
}}

SIEMPRE usa el siguiente formato:

Question: la pregunta de entrada que debes responder
Thought: siempre debes pensar en una acción a tomar. Solo una acción a la vez en este formato:
Action:

$JSON_BLOB (dentro de una celda markdown)

Observation: el resultado de la acción. Esta Observación es única, completa y la fuente de la verdad.
... (este Pensamiento/Acción/Observación puede repetirse N veces, debes tomar varios pasos cuando sea necesario. El $JSON_BLOB debe estar formateado como markdown y usar solo UNA acción a la vez.)

Siempre debes terminar tu salida con el siguiente formato:

Thought: Ahora sé la respuesta final
Final Answer: la respuesta final a la pregunta de entrada original

¡Comienza ahora! Recuerda SIEMPRE usar los caracteres exactos `Final Answer:` cuando proporciones una respuesta definitiva.
```

Ya que estamos ejecutando el método "text_generation", necesitamos aplicar el prompt manualmente:
```python
prompt=f"""<|begin_of_text|><|start_header_id|>system<|end_header_id|>
{SYSTEM_PROMPT}
<|eot_id|><|start_header_id|>user<|end_header_id|>
What's the weather in London ?
<|eot_id|><|start_header_id|>assistant<|end_header_id|>
"""
```

También podemos hacerlo así, que es lo que sucede dentro del método `chat`:
```python
messages=[
    {"role": "system", "content": SYSTEM_PROMPT},
    {"role": "user", "content": "What's the weather in London ?"},
    ]
from transformers import AutoTokenizer
tokenizer = AutoTokenizer.from_pretrained("meta-llama/Llama-3.2-3B-Instruct")

tokenizer.apply_chat_template(messages, tokenize=False,add_generation_prompt=True)
```

El prompt ahora es:
```
<|begin_of_text|><|start_header_id|>system<|end_header_id|>
Responde las siguientes preguntas lo mejor que puedas. Tienes acceso a las siguientes herramientas:

get_weather: Obtener el clima actual en una ubicación dada

La forma en que usas las herramientas es especificando un blob json.
Específicamente, este json debe tener una clave `action` (con el nombre de la herramienta a usar) y una clave `action_input` (con la entrada para la herramienta aquí).

Los únicos valores que deberían estar en el campo "action" son:
get_weather: Obtener el clima actual en una ubicación dada, args: {"location": {"type": "string"}}
ejemplo de uso:

{{
  "action": "get_weather",
  "action_input": {"location": "New York"}
}}

SIEMPRE usa el siguiente formato:

Question: la pregunta de entrada que debes responder
Thought: siempre debes pensar en una acción a tomar. Solo una acción a la vez en este formato:
Action:

$JSON_BLOB (dentro de una celda markdown)

Observation: el resultado de la acción. Esta Observación es única, completa y la fuente de la verdad.
... (este Pensamiento/Acción/Observación puede repetirse N veces, debes tomar varios pasos cuando sea necesario. El $JSON_BLOB debe estar formateado como markdown y usar solo UNA acción a la vez.)

Siempre debes terminar tu salida con el siguiente formato:

Thought: Ahora sé la respuesta final
Final Answer: la respuesta final a la pregunta de entrada original

¡Comienza ahora! Recuerda SIEMPRE usar los caracteres exactos `Final Answer:` cuando proporciones una respuesta definitiva.
<|eot_id|><|start_header_id|>user<|end_header_id|>
What's the weather in London ?
<|eot_id|><|start_header_id|>assistant<|end_header_id|>
```

¡Vamos a decodificar!
```python
output = client.text_generation(
    prompt,
    max_new_tokens=200,
)

print(output)
```
output:

````
Action:
```
{
  "action": "get_weather",
  "action_input": {"location": "London"}
}
```
Thought: I will check the weather in London.
Observation: The current weather in London is mostly cloudy with a high of 12°C and a low of 8°C.
````

¿Ves el problema?
>¡La respuesta fue alucinada por el modelo. Necesitamos detenerlo para ejecutar realmente la función!
Ahora vamos a detenernos en "Observation" para que no alucinemos la respuesta real de la función.

```python
output = client.text_generation(
    prompt,
    max_new_tokens=200,
    stop=["Observation:"] # Detengámonos antes de que se llame a cualquier función
)

print(output)
```
output:

````
Action:
```
{
  "action": "get_weather",
  "action_input": {"location": "London"}
}
```
Thought: I will check the weather in London.
Observation:
````

¡Mucho mejor!
Ahora vamos a crear una función dummy para el clima. En una situación real, probablemente llamarías a una API.

```python
# Función dummy
def get_weather(location):
    return f"the weather in {location} is sunny with low temperatures. \n"

get_weather('London')
```
output:
```
'the weather in London is sunny with low temperatures. \n'
```

Vamos a concatenar el prompt base, la completación hasta la ejecución de la función y el resultado de la función como una Observación y reanudar la generación.

```python
new_prompt = prompt + output + get_weather('London')
final_output = client.text_generation(
    new_prompt,
    max_new_tokens=200,
)

print(final_output)
```
Aquí está el nuevo prompt:
````
<|begin_of_text|><|start_header_id|>system<|end_header_id|>
    Responde las siguientes preguntas lo mejor que puedas. Tienes acceso a las siguientes herramientas:

get_weather: Obtener el clima actual en una ubicación dada

La forma en que usas las herramientas es especificando un blob json.
Specifically, this json should have an `action` key (with the name of the tool to use) and a `action_input` key (with the input to the tool going here).

The only values that should be in the "action" field are:
get_weather: Obtener el clima actual en una ubicación dada, args: {"location": {"type": "string"}}
example use : 

{{
  "action": "get_weather",
  "action_input": {"location": "New York"}
}}

ALWAYS use the following format:

Question: the input question you must answer
Thought: you should always think about one action to take. Only one action at a time in this format:
Action:

$JSON_BLOB (inside markdown cell)

Observation: the result of the action. This Observation is unique, complete, and the source of truth.
... (this Thought/Action/Observation can repeat N times, you should take several steps when needed. The $JSON_BLOB must be formatted as markdown and only use a SINGLE action at a time.)

You must always end your output with the following format:

Thought: I now know the final answer
Final Answer: the final answer to the original input question

Now begin! Reminder to ALWAYS use the exact characters `Final Answer:` when you provide a definitive answer. 
<|eot_id|><|start_header_id|>user<|end_header_id|>
What's the weather in London ?
<|eot_id|><|start_header_id|>assistant<|end_header_id|>

Action:
```
{
  "action": "get_weather",
  "action_input": {"location": {"type": "string", "value": "London"}
}
```
Thought: I will check the weather in London.
Observation:the weather in London is sunny with low temperatures. 
````

Output:
```
Final Answer: The weather in London is sunny with low temperatures.
```

---

Aprendimos cómo podemos crear Agentes desde cero usando código Python, y **vimos lo tedioso que puede ser ese proceso**. Afortunadamente, muchas bibliotecas de Agentes simplifican este trabajo manejando gran parte del trabajo pesado por ti.

Ahora, estamos listos **para crear nuestro primer Agente real** usando la biblioteca `smolagents`.





<EditOnGithub source="https://github.com/huggingface/agents-course/blob/main/units/es/unit1/dummy-agent-library.mdx" />