# The Tasks Manager

Exporting a model from one framework to some format (also called backend here) involves specifying inputs and outputs information that the export function needs. The way `optimum.exporters` is structured for each backend is as follows:
- Configuration classes containing the information for each model to perform the export.
- Exporting functions using the proper configuration for the model to export.

The role of the [TasksManager](/docs/optimum/main/en/exporters/task_manager#optimum.exporters.tasks.TasksManager) is to be the main entry-point to load a model given a name and a task, and to get the proper configuration for a given (architecture, backend) couple. 
That way, there is a centralized place to register the `task -> model class` and `(architecture, backend) -> configuration` mappings. This allows the export functions to use this, and to rely on the various checks it provides.

## Task names

The tasks supported might depend on the backend, but here are the mappings between a task name and the auto class for PyTorch.

<Tip>

It is possible to know which tasks are supported for a model for a given backend, by doing:

```python
>>> from optimum.exporters.tasks import TasksManager

>>> model_type = "distilbert"
>>> # For instance, for the ONNX export.
>>> backend = "onnx"
>>> distilbert_tasks = list(TasksManager.get_supported_tasks_for_model_type(model_type, backend).keys())

>>> print(distilbert_tasks)
['default', 'fill-mask', 'text-classification', 'multiple-choice', 'token-classification', 'question-answering']
```

</Tip>

### PyTorch

#### Transformers

| Task                             | Auto Class                                                    |
|----------------------------------|---------------------------------------------------------------|
| `audio-classification`           | `AutoModelForAudioClassification`                             |
| `audio-frame-classification`     | `AutoModelForAudioFrameClassification`                        |
| `audio-xvector`                  | `AutoModelForAudioXVector`                                    |
| `automatic-speech-recognition`   | `AutoModelForSpeechSeq2Seq`, `AutoModelForCTC`                |
| `depth-estimation`               | `AutoModelForDepthEstimation`                                 |
| `feature-extraction`             | `AutoModel`                                                   |
| `fill-mask`                      | `AutoModelForMaskedLM`                                        |
| `image-classification`           | `AutoModelForImageClassification`                             |
| `image-to-image`                 | `AutoModelForImageToImage`                                    |
| `image-to-text`                  | `AutoModelForVision2Seq`, `AutoModel`                         |
| `image-text-to-text`             | `AutoModelForImageTextToText`                                 |
| `mask-generation`                | `AutoModel`                                                   |
| `masked-im`                      | `AutoModelForMaskedImageModeling`                             |
| `multiple-choice`                | `AutoModelForMultipleChoice`                                  |
| `object-detection`               | `AutoModelForObjectDetection`                                 |
| `question-answering`             | `AutoModelForQuestionAnswering`                               |
| `reinforcement-learning`         | `AutoModel`                                                   |
| `semantic-segmentation`          | `AutoModelForSemanticSegmentation`                            |
| `text-to-audio`                  | `AutoModelForTextToSpectrogram`, `AutoModelForTextToWaveform` |
| `text-generation`                | `AutoModelForCausalLM`                                        |
| `text2text-generation`           | `AutoModelForSeq2SeqLM`                                       |
| `text-classification`            | `AutoModelForSequenceClassification`                          |
| `token-classification`           | `AutoModelForTokenClassification`                             |
| `visual-question-answering`      | `AutoModelForVisualQuestionAnswering`                         |
| `zero-shot-image-classification` | `AutoModelForZeroShotImageClassification`                     |
| `zero-shot-object-detection`     | `AutoModelForZeroShotObjectDetection`                         |

#### Diffusers

| Task             | Auto Class                   |
|------------------|------------------------------|
| `text-to-image`  | `AutoPipelineForText2Image`  |
| `image-to-image` | `AutoPipelineForImage2Image` |
| `inpainting`     | `AutoPipelineForInpainting`  |

#### Sentence Transformers

| Task                  | Auto Class            |
|-----------------------|-----------------------|
| `feature-extraction`  | `SentenceTransformer` |
| `sentence-similarity` | `SentenceTransformer` |

## Reference[[optimum.exporters.tasks.TasksManager]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class optimum.exporters.tasks.TasksManager</name><anchor>optimum.exporters.tasks.TasksManager</anchor><source>https://github.com/huggingface/optimum/blob/main/optimum/exporters/tasks.py#L118</source><parameters>[]</parameters></docstring>

Handles the `task name -> model class` and `architecture -> configuration` mappings.



<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>create_register</name><anchor>optimum.exporters.tasks.TasksManager.create_register</anchor><source>https://github.com/huggingface/optimum/blob/main/optimum/exporters/tasks.py#L287</source><parameters>[{"name": "backend", "val": ": str"}, {"name": "overwrite_existing", "val": ": bool = False"}]</parameters><paramsdesc>- **backend** (`str`) --
  The name of the backend that the register function will handle.
- **overwrite_existing** (`bool`, defaults to `False`) --
  Whether or not the register function is allowed to overwrite an already existing config.</paramsdesc><paramgroups>0</paramgroups><rettype>`Callable[[str, Tuple[str, ...]], Callable[[Type], Type]]`</rettype><retdesc>A decorator taking the model type and a the
supported tasks.</retdesc></docstring>

Creates a register function for the specified backend.







<ExampleCodeBlock anchor="optimum.exporters.tasks.TasksManager.create_register.example">

Example:
```python
>>> register_for_new_backend = create_register("new-backend")

>>> @register_for_new_backend("bert", "text-classification", "token-classification")
>>> class BertNewBackendConfig(NewBackendConfig):
>>>     pass
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>determine_framework</name><anchor>optimum.exporters.tasks.TasksManager.determine_framework</anchor><source>https://github.com/huggingface/optimum/blob/main/optimum/exporters/tasks.py#L579</source><parameters>[{"name": "model_name_or_path", "val": ": typing.Union[str, pathlib.Path]"}, {"name": "subfolder", "val": ": str = ''"}, {"name": "revision", "val": ": typing.Optional[str] = None"}, {"name": "cache_dir", "val": ": str = '/home/runner/.cache/huggingface/hub'"}, {"name": "token", "val": ": typing.Union[bool, str, NoneType] = None"}]</parameters><paramsdesc>- **model_name_or_path** (`Union[str, Path]`) --
  Can be either the model id of a model repo on the Hugging Face Hub, or a path to a local directory
  containing a model.
- **subfolder** (`str`, *optional*, defaults to `""`) --
  In case the model files are located inside a subfolder of the model directory / repo on the Hugging
  Face Hub, you can specify the subfolder name here.
- **revision** (`Optional[str]`,  defaults to `None`) --
  Revision is the specific model version to use. It can be a branch name, a tag name, or a commit id.
- **cache_dir** (`Optional[str]`, *optional*) --
  Path to a directory in which a downloaded pretrained model weights have been cached if the standard cache should not be used.
- **token** (`Optional[Union[bool,str]]`, defaults to `None`) --
  The token to use as HTTP bearer authorization for remote files. If `True`, will use the token generated
  when running `huggingface-cli login` (stored in `huggingface_hub.constants.HF_TOKEN_PATH`).</paramsdesc><paramgroups>0</paramgroups><rettype>`str`</rettype><retdesc>The framework to use for the export.</retdesc></docstring>

Determines the framework to use for the export.

The priority is in the following order:
1. User input via `framework`.
2. If local checkpoint is provided, use the same framework as the checkpoint.
3. If model repo, try to infer the framework from the cache if available, else from the Hub.
4. If could not infer, use available framework in environment, with priority given to PyTorch.








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_all_tasks</name><anchor>optimum.exporters.tasks.TasksManager.get_all_tasks</anchor><source>https://github.com/huggingface/optimum/blob/main/optimum/exporters/tasks.py#L1059</source><parameters>[]</parameters><rettype>`List`</rettype><retdesc>all the possible tasks.</retdesc></docstring>

Retrieves all the possible tasks.






</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_exporter_config_constructor</name><anchor>optimum.exporters.tasks.TasksManager.get_exporter_config_constructor</anchor><source>https://github.com/huggingface/optimum/blob/main/optimum/exporters/tasks.py#L1232</source><parameters>[{"name": "exporter", "val": ": str"}, {"name": "model", "val": ": typing.Optional[ForwardRef('PreTrainedModel')] = None"}, {"name": "task", "val": ": str = 'feature-extraction'"}, {"name": "model_type", "val": ": typing.Optional[str] = None"}, {"name": "model_name", "val": ": typing.Optional[str] = None"}, {"name": "exporter_config_kwargs", "val": ": typing.Optional[typing.Dict[str, typing.Any]] = None"}, {"name": "library_name", "val": ": typing.Optional[str] = None"}]</parameters><paramsdesc>- **exporter** (`str`) --
  The exporter to use.
- **model** (`Optional[PreTrainedModel]`, defaults to `None`) --
  The instance of the model.
- **task** (`str`, defaults to `"feature-extraction"`) --
  The task to retrieve the config for.
- **model_type** (`Optional[str]`, defaults to `None`) --
  The model type to retrieve the config for.
- **model_name** (`Optional[str]`, defaults to `None`) --
  The name attribute of the model object, only used for the exception message.
- **exporter_config_kwargs** (`Optional[Dict[str, Any]]`, defaults to `None`) --
  Arguments that will be passed to the exporter config class when building the config constructor.
- **library_name** (`Optional[str]`, defaults to `None`) --
  The library name of the model. Can be any of "transformers", "timm", "diffusers", "sentence_transformers".</paramsdesc><paramgroups>0</paramgroups><rettype>`ExportConfigConstructor`</rettype><retdesc>The `ExporterConfig` constructor for the requested backend.</retdesc></docstring>

Gets the `ExportConfigConstructor` for a model (or alternatively for a model type) and task combination.








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_model_class_for_task</name><anchor>optimum.exporters.tasks.TasksManager.get_model_class_for_task</anchor><source>https://github.com/huggingface/optimum/blob/main/optimum/exporters/tasks.py#L445</source><parameters>[{"name": "task", "val": ": str"}, {"name": "framework", "val": ": str = 'pt'"}, {"name": "model_type", "val": ": typing.Optional[str] = None"}, {"name": "model_class_name", "val": ": typing.Optional[str] = None"}, {"name": "library", "val": ": str = 'transformers'"}]</parameters><paramsdesc>- **task** (`str`) --
  The task required.
- **framework** (`str`, defaults to `"pt"`) --
  The framework to use for the export.
- **model_type** (`Optional[str]`, defaults to `None`) --
  The model type to retrieve the model class for. Some architectures need a custom class to be loaded,
  and can not be loaded from auto class.
- **model_class_name** (`Optional[str]`, defaults to `None`) --
  A model class name, allowing to override the default class that would be detected for the task. This
  parameter is useful for example for "automatic-speech-recognition", that may map to
  AutoModelForSpeechSeq2Seq or to AutoModelForCTC.
- **library** (`str`, defaults to `transformers`) --
  The library name of the model. Can be any of "transformers", "timm", "diffusers", "sentence_transformers".</paramsdesc><paramgroups>0</paramgroups><retdesc>The AutoModel class corresponding to the task.</retdesc></docstring>

Attempts to retrieve an AutoModel class from a task name.






</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_model_from_task</name><anchor>optimum.exporters.tasks.TasksManager.get_model_from_task</anchor><source>https://github.com/huggingface/optimum/blob/main/optimum/exporters/tasks.py#L1079</source><parameters>[{"name": "task", "val": ": str"}, {"name": "model_name_or_path", "val": ": typing.Union[str, pathlib.Path]"}, {"name": "subfolder", "val": ": str = ''"}, {"name": "revision", "val": ": typing.Optional[str] = None"}, {"name": "cache_dir", "val": ": str = '/home/runner/.cache/huggingface/hub'"}, {"name": "token", "val": ": typing.Union[bool, str, NoneType] = None"}, {"name": "framework", "val": ": typing.Optional[str] = None"}, {"name": "torch_dtype", "val": ": typing.Optional[ForwardRef('torch.dtype')] = None"}, {"name": "device", "val": ": typing.Union[ForwardRef('torch.device'), str, NoneType] = None"}, {"name": "library_name", "val": ": typing.Optional[str] = None"}, {"name": "**model_kwargs", "val": ""}]</parameters><paramsdesc>- **task** (`str`) --
  The task required.
- **model_name_or_path** (`Union[str, Path]`) --
  Can be either the model id of a model repo on the Hugging Face Hub, or a path to a local directory
  containing a model.
- **subfolder** (`str`, defaults to `""`) --
  In case the model files are located inside a subfolder of the model directory / repo on the Hugging
  Face Hub, you can specify the subfolder name here.
- **revision** (`Optional[str]`, *optional*) --
  Revision is the specific model version to use. It can be a branch name, a tag name, or a commit id.
- **cache_dir** (`Optional[str]`, *optional*) --
  Path to a directory in which a downloaded pretrained model weights have been cached if the standard cache should not be used.
- **token** (`Optional[Union[bool,str]]`, defaults to `None`) --
  The token to use as HTTP bearer authorization for remote files. If `True`, will use the token generated
  when running `huggingface-cli login` (stored in `huggingface_hub.constants.HF_TOKEN_PATH`).
- **framework** (`Optional[str]`, *optional*) --
  The framework to use for the export. See `TasksManager.determine_framework` for the priority should
  none be provided.
- **torch_dtype** (`Optional[torch.dtype]`, defaults to `None`) --
  Data type to load the model on. PyTorch-only argument.
- **device** (`Optional[torch.device]`, defaults to `None`) --
  Device to initialize the model on. PyTorch-only argument. For PyTorch, defaults to "cpu".
- **library_name** (`Optional[str]`, defaults to `None`) --
  The library name of the model. Can be any of "transformers", "timm", "diffusers", "sentence_transformers". See `TasksManager.infer_library_from_model` for the priority should
  none be provided.
- **model_kwargs** (`Dict[str, Any]`, *optional*) --
  Keyword arguments to pass to the model `.from_pretrained()` method.
- **library_name** (`Optional[str]`, defaults to `None`) --
  The library name of the model. Can be any of "transformers", "timm", "diffusers", "sentence_transformers". See `TasksManager.infer_library_from_model` for the priority should
  none be provided.</paramsdesc><paramgroups>0</paramgroups><retdesc>The instance of the model.</retdesc></docstring>

Retrieves a model from its name and the task to be enabled.






</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_supported_model_type_for_task</name><anchor>optimum.exporters.tasks.TasksManager.get_supported_model_type_for_task</anchor><source>https://github.com/huggingface/optimum/blob/main/optimum/exporters/tasks.py#L403</source><parameters>[{"name": "task", "val": ": str"}, {"name": "exporter", "val": ": str"}]</parameters></docstring>

Returns the list of supported architectures by the exporter for a given task. Transformers-specific.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_supported_tasks_for_model_type</name><anchor>optimum.exporters.tasks.TasksManager.get_supported_tasks_for_model_type</anchor><source>https://github.com/huggingface/optimum/blob/main/optimum/exporters/tasks.py#L342</source><parameters>[{"name": "model_type", "val": ": str"}, {"name": "exporter", "val": ": str"}, {"name": "model_name", "val": ": typing.Optional[str] = None"}, {"name": "library_name", "val": ": typing.Optional[str] = None"}]</parameters><paramsdesc>- **model_type** (`str`) --
  The model type to retrieve the supported tasks for.
- **exporter** (`str`) --
  The name of the exporter.
- **model_name** (`Optional[str]`, defaults to `None`) --
  The name attribute of the model object, only used for the exception message.
- **library_name** (`Optional[str]`, defaults to `None`) --
  The library name of the model. Can be any of "transformers", "timm", "diffusers", "sentence_transformers".</paramsdesc><paramgroups>0</paramgroups><rettype>`Dict[str, ExportConfigConstructor]`</rettype><retdesc>The mapping between the supported tasks and the backend config
constructors for the specified model type.</retdesc></docstring>

Retrieves the `task -> exporter backend config constructors` map from the model type.








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>infer_library_from_model</name><anchor>optimum.exporters.tasks.TasksManager.infer_library_from_model</anchor><source>https://github.com/huggingface/optimum/blob/main/optimum/exporters/tasks.py#L962</source><parameters>[{"name": "model", "val": ": typing.Union[str, ForwardRef('PreTrainedModel'), ForwardRef('DiffusionPipeline'), typing.Type]"}, {"name": "subfolder", "val": ": str = ''"}, {"name": "revision", "val": ": typing.Optional[str] = None"}, {"name": "cache_dir", "val": ": str = '/home/runner/.cache/huggingface/hub'"}, {"name": "token", "val": ": typing.Union[bool, str, NoneType] = None"}]</parameters><paramsdesc>- **model** (`Union[str, PreTrainedModel, DiffusionPipeline, Type]`) --
  The model to infer the task from. This can either be the name of a repo on the HuggingFace Hub, an
  instance of a model, or a model class.
- **subfolder** (`str`, defaults to `""`) --
  In case the model files are located inside a subfolder of the model directory / repo on the Hugging
  Face Hub, you can specify the subfolder name here.
- **revision** (`Optional[str]`, *optional*, defaults to `None`) --
  Revision is the specific model version to use. It can be a branch name, a tag name, or a commit id.
- **cache_dir** (`Optional[str]`, *optional*) --
  Path to a directory in which a downloaded pretrained model weights have been cached if the standard cache should not be used.
- **token** (`Optional[Union[bool,str]]`, defaults to `None`) --
  The token to use as HTTP bearer authorization for remote files. If `True`, will use the token generated
  when running `huggingface-cli login` (stored in `huggingface_hub.constants.HF_TOKEN_PATH`).</paramsdesc><paramgroups>0</paramgroups><rettype>`str`</rettype><retdesc>The library name automatically detected from the model repo, model instance, or model class.</retdesc></docstring>

Infers the library from the model repo, model instance, or model class.








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>infer_task_from_model</name><anchor>optimum.exporters.tasks.TasksManager.infer_task_from_model</anchor><source>https://github.com/huggingface/optimum/blob/main/optimum/exporters/tasks.py#L803</source><parameters>[{"name": "model", "val": ": typing.Union[str, ForwardRef('PreTrainedModel'), ForwardRef('DiffusionPipeline'), typing.Type]"}, {"name": "subfolder", "val": ": str = ''"}, {"name": "revision", "val": ": typing.Optional[str] = None"}, {"name": "cache_dir", "val": ": str = '/home/runner/.cache/huggingface/hub'"}, {"name": "token", "val": ": typing.Union[bool, str, NoneType] = None"}, {"name": "library_name", "val": ": typing.Optional[str] = None"}]</parameters><paramsdesc>- **model** (`Union[str, PreTrainedModel, DiffusionPipeline, Type]`) --
  The model to infer the task from. This can either be the name of a repo on the HuggingFace Hub, an
  instance of a model, or a model class.
- **subfolder** (`str`, *optional*, defaults to `""`) --
  In case the model files are located inside a subfolder of the model directory / repo on the Hugging
  Face Hub, you can specify the subfolder name here.
- **revision** (`Optional[str]`,  defaults to `None`) --
  Revision is the specific model version to use. It can be a branch name, a tag name, or a commit id.
- **cache_dir** (`Optional[str]`, *optional*) --
  Path to a directory in which a downloaded pretrained model weights have been cached if the standard cache should not be used.
- **token** (`Optional[Union[bool,str]]`, defaults to `None`) --
  The token to use as HTTP bearer authorization for remote files. If `True`, will use the token generated
  when running `huggingface-cli login` (stored in `huggingface_hub.constants.HF_TOKEN_PATH`).
- **library_name** (`Optional[str]`, defaults to `None`) --
  The library name of the model. Can be any of "transformers", "timm", "diffusers", "sentence_transformers". See `TasksManager.infer_library_from_model` for the priority should
  none be provided.</paramsdesc><paramgroups>0</paramgroups><rettype>`str`</rettype><retdesc>The task name automatically detected from the HF hub repo, model instance, or model class.</retdesc></docstring>

Infers the task from the model repo, model instance, or model class.








</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>standardize_model_attributes</name><anchor>optimum.exporters.tasks.TasksManager.standardize_model_attributes</anchor><source>https://github.com/huggingface/optimum/blob/main/optimum/exporters/tasks.py#L1008</source><parameters>[{"name": "model", "val": ": typing.Union[ForwardRef('PreTrainedModel'), ForwardRef('DiffusionPipeline')]"}, {"name": "library_name", "val": ": typing.Optional[str] = None"}]</parameters><paramsdesc>- **model** (`Union[PreTrainedModel, DiffusionPipeline]`) --
  The instance of the model.</paramsdesc><paramgroups>0</paramgroups></docstring>

Updates the model for export. This function is suitable to make required changes to the models from different
libraries to follow transformers style.




</div></div>