# Stateful Classes

Below are variations of a [singleton class](https://en.wikipedia.org/wiki/Singleton_pattern) in the sense that all
instances share the same state, which is initialized on the first instantiation.

These classes are immutable and store information about certain configurations or 
states.

## PartialState[[accelerate.PartialState]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class accelerate.PartialState</name><anchor>accelerate.PartialState</anchor><source>https://github.com/huggingface/accelerate/blob/main/src/accelerate/state.py#L124</source><parameters>[{"name": "cpu", "val": ": bool = False"}, {"name": "**kwargs", "val": ""}]</parameters><paramsdesc>- **cpu** (`bool`, *optional*) --
  Whether or not to force the script to execute on CPU. Will ignore any accelerators available if set to
  `True` and force the execution on the CPU.
- **kwargs** (additional keyword arguments, *optional*) --
  Additional keyword arguments to pass to the relevant `init_process_group` function. Valid `kwargs` can be
  found in [utils.InitProcessGroupKwargs](/docs/accelerate/main/en/package_reference/utilities#accelerate.InitProcessGroupKwargs). See the example section for detailed usage.</paramsdesc><paramgroups>0</paramgroups></docstring>

Singleton class that has information about the current training environment and functions to help with process
control. Designed to be used when only process control and device execution states are needed. Does *not* need to
be initialized from `Accelerator`.



**Available attributes:**

- **device** (`torch.device`) -- The device to use.
- **distributed_type** ([DistributedType](/docs/accelerate/main/en/package_reference/utilities#accelerate.DistributedType)) -- The type of distributed environment currently
  in use.
- **local_process_index** (`int`) -- The index of the current process on the current server.
- **mixed_precision** (`str`) -- Whether or not the current script will use mixed precision, and if so the type
  of mixed precision being performed. (Choose from 'no','fp16','bf16 or 'fp8').
- **num_processes** (`int`) -- The number of processes currently launched in parallel.
- **process_index** (`int`) -- The index of the current process.
- **is_last_process** (`bool`) -- Whether or not the current process is the last one.
- **is_main_process** (`bool`) -- Whether or not the current process is the main one.
- **is_local_main_process** (`bool`) -- Whether or not the current process is the main one on the local node.
- **debug** (`bool`) -- Whether or not the current script is being run in debug mode.

<ExampleCodeBlock anchor="accelerate.PartialState.example">

Example:
```python
from accelerate.utils import InitProcessGroupKwargs

# To include `InitProcessGroupKwargs`, init then call `.to_kwargs()`
kwargs = InitProcessGroupKwargs(...).to_kwargs()
state = PartialState(**kwargs)
```

</ExampleCodeBlock>



<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>destroy_process_group</name><anchor>accelerate.PartialState.destroy_process_group</anchor><source>https://github.com/huggingface/accelerate/blob/main/src/accelerate/state.py#L840</source><parameters>[{"name": "group", "val": " = None"}]</parameters></docstring>

Destroys the process group. If one is not specified, the default process group is destroyed.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>local_main_process_first</name><anchor>accelerate.PartialState.local_main_process_first</anchor><source>https://github.com/huggingface/accelerate/blob/main/src/accelerate/state.py#L528</source><parameters>[]</parameters></docstring>

Lets the local main process go inside a with block.

The other processes will enter the with block after the main process exits.

<ExampleCodeBlock anchor="accelerate.PartialState.local_main_process_first.example">

Example:

```python
>>> from accelerate.state import PartialState

>>> state = PartialState()
>>> with state.local_main_process_first():
...     # This will be printed first by local process 0 then in a seemingly
...     # random order by the other processes.
...     print(f"This will be printed by process {state.local_process_index}")
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>main_process_first</name><anchor>accelerate.PartialState.main_process_first</anchor><source>https://github.com/huggingface/accelerate/blob/main/src/accelerate/state.py#L507</source><parameters>[]</parameters></docstring>

Lets the main process go first inside a with block.

The other processes will enter the with block after the main process exits.

<ExampleCodeBlock anchor="accelerate.PartialState.main_process_first.example">

Example:

```python
>>> from accelerate import Accelerator

>>> accelerator = Accelerator()
>>> with accelerator.main_process_first():
...     # This will be printed first by process 0 then in a seemingly
...     # random order by the other processes.
...     print(f"This will be printed by process {accelerator.process_index}")
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>on_last_process</name><anchor>accelerate.PartialState.on_last_process</anchor><source>https://github.com/huggingface/accelerate/blob/main/src/accelerate/state.py#L610</source><parameters>[{"name": "function", "val": ": Callable[..., Any]"}]</parameters><paramsdesc>- **function** (`Callable`) -- The function to decorate.</paramsdesc><paramgroups>0</paramgroups></docstring>

Decorator that only runs the decorated function on the last process.



<ExampleCodeBlock anchor="accelerate.PartialState.on_last_process.example">

Example:
```python
# Assume we have 4 processes.
from accelerate.state import PartialState

state = PartialState()


@state.on_last_process
def print_something():
    print(f"Printed on process {state.process_index}")


print_something()
"Printed on process 3"
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>on_local_main_process</name><anchor>accelerate.PartialState.on_local_main_process</anchor><source>https://github.com/huggingface/accelerate/blob/main/src/accelerate/state.py#L579</source><parameters>[{"name": "function", "val": ": Callable[..., Any] | None = None"}]</parameters><paramsdesc>- **function** (`Callable`) -- The function to decorate.</paramsdesc><paramgroups>0</paramgroups></docstring>

Decorator that only runs the decorated function on the local main process.



<ExampleCodeBlock anchor="accelerate.PartialState.on_local_main_process.example">

Example:
```python
# Assume we have 2 servers with 4 processes each.
from accelerate.state import PartialState

state = PartialState()


@state.on_local_main_process
def print_something():
    print("This will be printed by process 0 only on each server.")


print_something()
# On server 1:
"This will be printed by process 0 only"
# On server 2:
"This will be printed by process 0 only"
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>on_local_process</name><anchor>accelerate.PartialState.on_local_process</anchor><source>https://github.com/huggingface/accelerate/blob/main/src/accelerate/state.py#L671</source><parameters>[{"name": "function", "val": ": Callable[..., Any] | None = None"}, {"name": "local_process_index", "val": ": int | None = None"}]</parameters><paramsdesc>- **function** (`Callable`, *optional*) --
  The function to decorate.
- **local_process_index** (`int`, *optional*) --
  The index of the local process on which to run the function.</paramsdesc><paramgroups>0</paramgroups></docstring>

Decorator that only runs the decorated function on the process with the given index on the current node.



<ExampleCodeBlock anchor="accelerate.PartialState.on_local_process.example">

Example:
```python
# Assume we have 2 servers with 4 processes each.
from accelerate import Accelerator

accelerator = Accelerator()


@accelerator.on_local_process(local_process_index=2)
def print_something():
    print(f"Printed on process {accelerator.local_process_index}")


print_something()
# On server 1:
"Printed on process 2"
# On server 2:
"Printed on process 2"
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>on_main_process</name><anchor>accelerate.PartialState.on_main_process</anchor><source>https://github.com/huggingface/accelerate/blob/main/src/accelerate/state.py#L549</source><parameters>[{"name": "function", "val": ": Callable[..., Any] | None = None"}]</parameters><paramsdesc>- **function** (`Callable`) -- The function to decorate.</paramsdesc><paramgroups>0</paramgroups></docstring>

Decorator that only runs the decorated function on the main process.



<ExampleCodeBlock anchor="accelerate.PartialState.on_main_process.example">

Example:

```python
>>> from accelerate.state import PartialState

>>> state = PartialState()


>>> @state.on_main_process
... def print_something():
...     print("This will be printed by process 0 only.")


>>> print_something()
"This will be printed by process 0 only"
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>on_process</name><anchor>accelerate.PartialState.on_process</anchor><source>https://github.com/huggingface/accelerate/blob/main/src/accelerate/state.py#L638</source><parameters>[{"name": "function", "val": ": Callable[..., Any] | None = None"}, {"name": "process_index", "val": ": int | None = None"}]</parameters><paramsdesc>- **function** (`Callable`, `optional`) --
  The function to decorate.
- **process_index** (`int`, `optional`) --
  The index of the process on which to run the function.</paramsdesc><paramgroups>0</paramgroups></docstring>

Decorator that only runs the decorated function on the process with the given index.



<ExampleCodeBlock anchor="accelerate.PartialState.on_process.example">

Example:
```python
# Assume we have 4 processes.
from accelerate.state import PartialState

state = PartialState()


@state.on_process(process_index=2)
def print_something():
    print(f"Printed on process {state.process_index}")


print_something()
"Printed on process 2"
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>set_device</name><anchor>accelerate.PartialState.set_device</anchor><source>https://github.com/huggingface/accelerate/blob/main/src/accelerate/state.py#L814</source><parameters>[]</parameters></docstring>

Sets the device in `self.device` to the current distributed environment.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>split_between_processes</name><anchor>accelerate.PartialState.split_between_processes</anchor><source>https://github.com/huggingface/accelerate/blob/main/src/accelerate/state.py#L416</source><parameters>[{"name": "inputs", "val": ": list | tuple | dict | torch.Tensor"}, {"name": "apply_padding", "val": ": bool = False"}]</parameters><paramsdesc>- **inputs** (`list`, `tuple`, `torch.Tensor`, `dict` of `list`/`tuple`/`torch.Tensor`, or `datasets.Dataset`) --
  The input to split between processes.
- **apply_padding** (`bool`, `optional`, defaults to `False`) --
  Whether to apply padding by repeating the last element of the input so that all processes have the same
  number of elements. Useful when trying to perform actions such as `gather()` on the outputs or passing
  in less inputs than there are processes. If so, just remember to drop the padded elements afterwards.</paramsdesc><paramgroups>0</paramgroups></docstring>

Splits `input` between `self.num_processes` quickly and can be then used on that process. Useful when doing
distributed inference, such as with different prompts.

Note that when using a `dict`, all keys need to have the same number of elements.



<ExampleCodeBlock anchor="accelerate.PartialState.split_between_processes.example">

Example:

```python
# Assume there are two processes
from accelerate import PartialState

state = PartialState()
with state.split_between_processes(["A", "B", "C"]) as inputs:
    print(inputs)
# Process 0
["A", "B"]
# Process 1
["C"]

with state.split_between_processes(["A", "B", "C"], apply_padding=True) as inputs:
    print(inputs)
# Process 0
["A", "B"]
# Process 1
["C", "C"]
```

</ExampleCodeBlock>


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>wait_for_everyone</name><anchor>accelerate.PartialState.wait_for_everyone</anchor><source>https://github.com/huggingface/accelerate/blob/main/src/accelerate/state.py#L369</source><parameters>[]</parameters></docstring>

Will stop the execution of the current process until every other process has reached that point (so this does
nothing when the script is only run in one process). Useful to do before saving a model.

<ExampleCodeBlock anchor="accelerate.PartialState.wait_for_everyone.example">

Example:

```python
>>> # Assuming two GPU processes
>>> import time
>>> from accelerate.state import PartialState

>>> state = PartialState()
>>> if state.is_main_process:
...     time.sleep(2)
>>> else:
...     print("I'm waiting for the main process to finish its sleep...")
>>> state.wait_for_everyone()
>>> # Should print on every process at the same time
>>> print("Everyone is here")
```

</ExampleCodeBlock>


</div></div>

## AcceleratorState[[accelerate.state.AcceleratorState]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class accelerate.state.AcceleratorState</name><anchor>accelerate.state.AcceleratorState</anchor><source>https://github.com/huggingface/accelerate/blob/main/src/accelerate/state.py#L863</source><parameters>[{"name": "mixed_precision", "val": ": str | None = None"}, {"name": "cpu", "val": ": bool = False"}, {"name": "dynamo_plugin", "val": " = None"}, {"name": "deepspeed_plugin", "val": " = None"}, {"name": "fsdp_plugin", "val": " = None"}, {"name": "torch_tp_plugin", "val": " = None"}, {"name": "megatron_lm_plugin", "val": " = None"}, {"name": "parallelism_config", "val": " = None"}, {"name": "_from_accelerator", "val": ": bool = False"}, {"name": "**kwargs", "val": ""}]</parameters></docstring>

Singleton class that has information about the current training environment.

**Available attributes:**

- **device** (`torch.device`) -- The device to use.
- **distributed_type** ([DistributedType](/docs/accelerate/main/en/package_reference/utilities#accelerate.DistributedType)) -- The type of distributed environment currently
  in use.
- **parallelism_config** (`ParallelismConfig`) -- The parallelism configuration for the
  current training environment. This is used to configure the distributed training environment.
- **initialized** (`bool`) -- Whether or not the `AcceleratorState` has been initialized from `Accelerator`.
- **local_process_index** (`int`) -- The index of the current process on the current server.
- **mixed_precision** (`str`) -- Whether or not the current script will use mixed precision, and if so the type
  of mixed precision being performed. (Choose from 'no','fp16','bf16 or 'fp8').
- **num_processes** (`int`) -- The number of processes currently launched in parallel.
- **process_index** (`int`) -- The index of the current process.
- **is_last_process** (`bool`) -- Whether or not the current process is the last one.
- **is_main_process** (`bool`) -- Whether or not the current process is the main one.
- **is_local_main_process** (`bool`) -- Whether or not the current process is the main one on the local node.
- **debug** (`bool`) -- Whether or not the current script is being run in debug mode.



<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>destroy_process_group</name><anchor>accelerate.state.AcceleratorState.destroy_process_group</anchor><source>https://github.com/huggingface/accelerate/blob/main/src/accelerate/state.py#L1083</source><parameters>[{"name": "group", "val": " = None"}]</parameters></docstring>

Destroys the process group. If one is not specified, the default process group is destroyed.

If `self.fork_launched` is `True` and `group` is `None`, nothing happens.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>get_deepspeed_plugin</name><anchor>accelerate.state.AcceleratorState.get_deepspeed_plugin</anchor><source>https://github.com/huggingface/accelerate/blob/main/src/accelerate/state.py#L1200</source><parameters>[{"name": "name", "val": ": str"}]</parameters></docstring>

Returns the DeepSpeedPlugin with the given plugin_key.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>local_main_process_first</name><anchor>accelerate.state.AcceleratorState.local_main_process_first</anchor><source>https://github.com/huggingface/accelerate/blob/main/src/accelerate/state.py#L1176</source><parameters>[]</parameters></docstring>

Lets the local main process go inside a with block.

The other processes will enter the with block after the main process exits.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>main_process_first</name><anchor>accelerate.state.AcceleratorState.main_process_first</anchor><source>https://github.com/huggingface/accelerate/blob/main/src/accelerate/state.py#L1166</source><parameters>[]</parameters></docstring>

Lets the main process go first inside a with block.

The other processes will enter the with block after the main process exits.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>select_deepspeed_plugin</name><anchor>accelerate.state.AcceleratorState.select_deepspeed_plugin</anchor><source>https://github.com/huggingface/accelerate/blob/main/src/accelerate/state.py#L1207</source><parameters>[{"name": "name", "val": ": str | None = None"}]</parameters></docstring>

Activates the DeepSpeedPlugin with the given `name`, and will disable all other plugins.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>split_between_processes</name><anchor>accelerate.state.AcceleratorState.split_between_processes</anchor><source>https://github.com/huggingface/accelerate/blob/main/src/accelerate/state.py#L1124</source><parameters>[{"name": "inputs", "val": ": list | tuple | dict | torch.Tensor"}, {"name": "apply_padding", "val": ": bool = False"}]</parameters><paramsdesc>- **inputs** (`list`, `tuple`, `torch.Tensor`, or `dict` of `list`/`tuple`/`torch.Tensor`) --
  The input to split between processes.
- **apply_padding** (`bool`, `optional`, defaults to `False`) --
  Whether to apply padding by repeating the last element of the input so that all processes have the same
  number of elements. Useful when trying to perform actions such as `gather()` on the outputs or passing
  in less inputs than there are processes. If so, just remember to drop the padded elements afterwards.</paramsdesc><paramgroups>0</paramgroups></docstring>

Splits `input` between `self.num_processes` quickly and can be then used on that process. Useful when doing
distributed inference, such as with different prompts.

Note that when using a `dict`, all keys need to have the same number of elements.



<ExampleCodeBlock anchor="accelerate.state.AcceleratorState.split_between_processes.example">

Example:

```python
# Assume there are two processes
from accelerate.state import AcceleratorState

state = AcceleratorState()
with state.split_between_processes(["A", "B", "C"]) as inputs:
    print(inputs)
# Process 0
["A", "B"]
# Process 1
["C"]

with state.split_between_processes(["A", "B", "C"], apply_padding=True) as inputs:
    print(inputs)
# Process 0
["A", "B"]
# Process 1
["C", "C"]
```

</ExampleCodeBlock>


</div></div>

## GradientState[[accelerate.state.GradientState]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class accelerate.state.GradientState</name><anchor>accelerate.state.GradientState</anchor><source>https://github.com/huggingface/accelerate/blob/main/src/accelerate/state.py#L1233</source><parameters>[{"name": "gradient_accumulation_plugin", "val": ": GradientAccumulationPlugin | None = None"}]</parameters></docstring>

Singleton class that has information related to gradient synchronization for gradient accumulation

**Available attributes:**

- **end_of_dataloader** (`bool`) -- Whether we have reached the end the current dataloader
- **remainder** (`int`) -- The number of extra samples that were added from padding the dataloader
- **sync_gradients** (`bool`) -- Whether the gradients should be synced across all devices
- **active_dataloader** (`Optional[DataLoader]`) -- The dataloader that is currently being iterated over
- **dataloader_references** (`List[Optional[DataLoader]]`) -- A list of references to the dataloaders that are
  being iterated over
- **num_steps** (`int`) -- The number of steps to accumulate over
- **adjust_scheduler** (`bool`) -- Whether the scheduler should be adjusted to account for the gradient
  accumulation
- **sync_with_dataloader** (`bool`) -- Whether the gradients should be synced at the end of the dataloader
  iteration and the number of total steps reset
- **is_xla_gradients_synced** (`bool`) -- Whether the XLA gradients have been synchronized. It is initialized
  as false. Once gradients have been reduced before the optimizer step, this flag is set to true. Subsequently,
  after each step, the flag is reset to false. FSDP will always synchronize the gradients, hence
  is_xla_gradients_synced is always true.


</div>

<EditOnGithub source="https://github.com/huggingface/accelerate/blob/main/docs/source/package_reference/state.md" />