# GaudiTrainer

The [`GaudiTrainer`](https://huggingface.co/docs/optimum/habana/package_reference/trainer#optimum.habana.GaudiTrainer) class provides an extended API for the feature-complete [Transformers Trainer](https://huggingface.co/docs/transformers/main_classes/trainer). It is used in all the [example scripts](/examples).

Before instantiating your [`GaudiTrainer`](https://huggingface.co/docs/optimum/habana/package_reference/trainer#optimum.habana.GaudiTrainer), create a [GaudiTrainingArguments](/docs/optimum.habana/v1.19.0/en/package_reference/trainer#optimum.habana.GaudiTrainingArguments) object to access all the points of customization during training.

<Tip warning={true}>

The [`GaudiTrainer`](https://huggingface.co/docs/optimum/habana/package_reference/trainer#optimum.habana.GaudiTrainer) class is optimized for 🤗 Transformers models running on Intel Gaudi.

</Tip>

Here is an example of how to customize [`GaudiTrainer`](https://huggingface.co/docs/optimum/habana/package_reference/trainer#optimum.habana.GaudiTrainer) to use a weighted loss (useful when you have an unbalanced training set):

```python
from torch import nn
from optimum.habana import GaudiTrainer


class CustomGaudiTrainer(GaudiTrainer):
    def compute_loss(self, model, inputs, return_outputs=False):
        labels = inputs.get("labels")
        # forward pass
        outputs = model(**inputs)
        logits = outputs.get("logits")
        # compute custom loss (suppose one has 3 labels with different weights)
        loss_fct = nn.CrossEntropyLoss(weight=torch.tensor([1.0, 2.0, 3.0]))
        loss = loss_fct(logits.view(-1, self.model.config.num_labels), labels.view(-1))
        return (loss, outputs) if return_outputs else loss
```

Another way to customize the training loop behavior for the PyTorch [`GaudiTrainer`](https://huggingface.co/docs/optimum/habana/package_reference/trainer#optimum.habana.GaudiTrainer) is to use [callbacks](https://huggingface.co/docs/transformers/main_classes/callback) that can inspect the training loop state (for progress reporting, logging on TensorBoard or other ML platforms...) and take decisions (like early stopping).

## GaudiTrainer[[optimum.habana.GaudiTrainer]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class optimum.habana.GaudiTrainer</name><anchor>optimum.habana.GaudiTrainer</anchor><source>https://github.com/huggingface/optimum-habana/blob/v1.19.0/optimum/habana/transformers/trainer.py#L214</source><parameters>[{"name": "model", "val": ": typing.Union[transformers.modeling_utils.PreTrainedModel, torch.nn.modules.module.Module, NoneType] = None"}, {"name": "gaudi_config", "val": ": GaudiConfig = None"}, {"name": "args", "val": ": TrainingArguments = None"}, {"name": "data_collator", "val": ": typing.Optional[transformers.data.data_collator.DataCollator] = None"}, {"name": "train_dataset", "val": ": typing.Union[torch.utils.data.dataset.Dataset, torch.utils.data.dataset.IterableDataset, ForwardRef('datasets.Dataset'), NoneType] = None"}, {"name": "eval_dataset", "val": ": typing.Union[torch.utils.data.dataset.Dataset, dict[str, torch.utils.data.dataset.Dataset], ForwardRef('datasets.Dataset'), NoneType] = None"}, {"name": "processing_class", "val": ": typing.Union[transformers.tokenization_utils_base.PreTrainedTokenizerBase, transformers.image_processing_utils.BaseImageProcessor, transformers.feature_extraction_utils.FeatureExtractionMixin, transformers.processing_utils.ProcessorMixin, NoneType] = None"}, {"name": "model_init", "val": ": typing.Optional[typing.Callable[[], transformers.modeling_utils.PreTrainedModel]] = None"}, {"name": "compute_loss_func", "val": ": typing.Optional[typing.Callable] = None"}, {"name": "compute_metrics", "val": ": typing.Optional[typing.Callable[[transformers.trainer_utils.EvalPrediction], dict]] = None"}, {"name": "callbacks", "val": ": typing.Optional[list[transformers.trainer_callback.TrainerCallback]] = None"}, {"name": "optimizers", "val": ": tuple = (None, None)"}, {"name": "optimizer_cls_and_kwargs", "val": ": typing.Optional[tuple[type[torch.optim.optimizer.Optimizer], dict[str, typing.Any]]] = None"}, {"name": "preprocess_logits_for_metrics", "val": ": typing.Optional[typing.Callable[[torch.Tensor, torch.Tensor], torch.Tensor]] = None"}]</parameters></docstring>

GaudiTrainer is built on top of the tranformers' Trainer to enable
deployment on Habana's Gaudi.



<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>autocast_smart_context_manager</name><anchor>optimum.habana.GaudiTrainer.autocast_smart_context_manager</anchor><source>https://github.com/huggingface/optimum-habana/blob/v1.19.0/optimum/habana/transformers/trainer.py#L1686</source><parameters>[{"name": "cache_enabled", "val": ": typing.Optional[bool] = True"}]</parameters></docstring>

A helper wrapper that creates an appropriate context manager for `autocast` while feeding it the desired
arguments, depending on the situation.
Modified by Habana to enable using `autocast` on Gaudi devices.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>evaluate</name><anchor>optimum.habana.GaudiTrainer.evaluate</anchor><source>https://github.com/huggingface/optimum-habana/blob/v1.19.0/optimum/habana/transformers/trainer.py#L1883</source><parameters>[{"name": "eval_dataset", "val": ": typing.Union[torch.utils.data.dataset.Dataset, dict[str, torch.utils.data.dataset.Dataset], NoneType] = None"}, {"name": "ignore_keys", "val": ": typing.Optional[list[str]] = None"}, {"name": "metric_key_prefix", "val": ": str = 'eval'"}]</parameters></docstring>

From https://github.com/huggingface/transformers/blob/v4.38.2/src/transformers/trainer.py#L3162 with the following modification
1. use throughput_warmup_steps in evaluation throughput calculation


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>evaluation_loop</name><anchor>optimum.habana.GaudiTrainer.evaluation_loop</anchor><source>https://github.com/huggingface/optimum-habana/blob/v1.19.0/optimum/habana/transformers/trainer.py#L2004</source><parameters>[{"name": "dataloader", "val": ": DataLoader"}, {"name": "description", "val": ": str"}, {"name": "prediction_loss_only", "val": ": typing.Optional[bool] = None"}, {"name": "ignore_keys", "val": ": typing.Optional[list[str]] = None"}, {"name": "metric_key_prefix", "val": ": str = 'eval'"}]</parameters></docstring>

Prediction/evaluation loop, shared by `Trainer.evaluate()` and `Trainer.predict()`.
Works both with or without labels.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>predict</name><anchor>optimum.habana.GaudiTrainer.predict</anchor><source>https://github.com/huggingface/optimum-habana/blob/v1.19.0/optimum/habana/transformers/trainer.py#L1959</source><parameters>[{"name": "test_dataset", "val": ": Dataset"}, {"name": "ignore_keys", "val": ": typing.Optional[list[str]] = None"}, {"name": "metric_key_prefix", "val": ": str = 'test'"}]</parameters></docstring>

From https://github.com/huggingface/transformers/blob/v4.45.2/src/transformers/trainer.py#L3904 with the following modification
1. comment out TPU related
2. use throughput_warmup_steps in evaluation throughput calculation


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>prediction_step</name><anchor>optimum.habana.GaudiTrainer.prediction_step</anchor><source>https://github.com/huggingface/optimum-habana/blob/v1.19.0/optimum/habana/transformers/trainer.py#L2271</source><parameters>[{"name": "model", "val": ": Module"}, {"name": "inputs", "val": ": dict"}, {"name": "prediction_loss_only", "val": ": bool"}, {"name": "ignore_keys", "val": ": typing.Optional[list[str]] = None"}]</parameters><paramsdesc>- **model** (`torch.nn.Module`) --
  The model to evaluate.
- **inputs** (`dict[str, Union[torch.Tensor, Any]]`) --
  The inputs and targets of the model.
  The dictionary will be unpacked before being fed to the model. Most models expect the targets under the
  argument `labels`. Check your model's documentation for all accepted arguments.
- **prediction_loss_only** (`bool`) --
  Whether or not to return the loss only.
- **ignore_keys** (`List[str]`, *optional*) --
  A list of keys in the output of your model (if it is a dictionary) that should be ignored when
  gathering predictions.</paramsdesc><paramgroups>0</paramgroups><rettype>tuple[Optional[torch.Tensor], Optional[torch.Tensor], Optional[torch.Tensor]]</rettype><retdesc>A tuple with the loss,
logits and labels (each being optional).</retdesc></docstring>

Perform an evaluation step on `model` using `inputs`.
Subclass and override to inject custom behavior.







</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>save_model</name><anchor>optimum.habana.GaudiTrainer.save_model</anchor><source>https://github.com/huggingface/optimum-habana/blob/v1.19.0/optimum/habana/transformers/trainer.py#L1786</source><parameters>[{"name": "output_dir", "val": ": typing.Optional[str] = None"}, {"name": "_internal_call", "val": ": bool = False"}]</parameters></docstring>

Will save the model, so you can reload it using `from_pretrained()`.
Will only save from the main process.


</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>train</name><anchor>optimum.habana.GaudiTrainer.train</anchor><source>https://github.com/huggingface/optimum-habana/blob/v1.19.0/optimum/habana/transformers/trainer.py#L517</source><parameters>[{"name": "resume_from_checkpoint", "val": ": typing.Union[str, bool, NoneType] = None"}, {"name": "trial", "val": ": typing.Union[ForwardRef('optuna.Trial'), dict[str, typing.Any], NoneType] = None"}, {"name": "ignore_keys_for_eval", "val": ": typing.Optional[list[str]] = None"}, {"name": "**kwargs", "val": ""}]</parameters><paramsdesc>- **resume_from_checkpoint** (`str` or `bool`, *optional*) --
  If a `str`, local path to a saved checkpoint as saved by a previous instance of `Trainer`. If a
  `bool` and equals `True`, load the last checkpoint in *args.output_dir* as saved by a previous instance
  of `Trainer`. If present, training will resume from the model/optimizer/scheduler states loaded here.
- **trial** (`optuna.Trial` or `dict[str, Any]`, *optional*) --
  The trial run or the hyperparameter dictionary for hyperparameter search.
- **ignore_keys_for_eval** (`List[str]`, *optional*) --
  A list of keys in the output of your model (if it is a dictionary) that should be ignored when
  gathering predictions for evaluation during the training.
- **kwargs** (`dict[str, Any]`, *optional*) --
  Additional keyword arguments used to hide deprecated arguments</paramsdesc><paramgroups>0</paramgroups></docstring>

Main training entry point.




</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>training_step</name><anchor>optimum.habana.GaudiTrainer.training_step</anchor><source>https://github.com/huggingface/optimum-habana/blob/v1.19.0/optimum/habana/transformers/trainer.py#L1706</source><parameters>[{"name": "model", "val": ": Module"}, {"name": "inputs", "val": ": dict"}, {"name": "num_items_in_batch", "val": ": typing.Optional[torch.Tensor] = None"}]</parameters><paramsdesc>- **model** (`torch.nn.Module`) --
  The model to train.
- **inputs** (`dict[str, Union[torch.Tensor, Any]]`) --
  The inputs and targets of the model.

  The dictionary will be unpacked before being fed to the model. Most models expect the targets under the
  argument `labels`. Check your model's documentation for all accepted arguments.</paramsdesc><paramgroups>0</paramgroups><rettype>`torch.Tensor`</rettype><retdesc>The tensor with training loss on this batch.</retdesc></docstring>

Perform a training step on a batch of inputs.

Subclass and override to inject custom behavior.








</div></div>

## GaudiSeq2SeqTrainer[[optimum.habana.GaudiSeq2SeqTrainer]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class optimum.habana.GaudiSeq2SeqTrainer</name><anchor>optimum.habana.GaudiSeq2SeqTrainer</anchor><source>https://github.com/huggingface/optimum-habana/blob/v1.19.0/optimum/habana/transformers/trainer_seq2seq.py#L56</source><parameters>[{"name": "model", "val": ": typing.Union[ForwardRef('PreTrainedModel'), torch.nn.modules.module.Module] = None"}, {"name": "gaudi_config", "val": ": GaudiConfig = None"}, {"name": "args", "val": ": GaudiTrainingArguments = None"}, {"name": "data_collator", "val": ": typing.Optional[ForwardRef('DataCollator')] = None"}, {"name": "train_dataset", "val": ": typing.Union[torch.utils.data.dataset.Dataset, ForwardRef('IterableDataset'), ForwardRef('datasets.Dataset'), NoneType] = None"}, {"name": "eval_dataset", "val": ": typing.Union[torch.utils.data.dataset.Dataset, dict[str, torch.utils.data.dataset.Dataset], NoneType] = None"}, {"name": "processing_class", "val": ": typing.Union[ForwardRef('PreTrainedTokenizerBase'), ForwardRef('BaseImageProcessor'), ForwardRef('FeatureExtractionMixin'), ForwardRef('ProcessorMixin'), NoneType] = None"}, {"name": "model_init", "val": ": typing.Optional[typing.Callable[[], ForwardRef('PreTrainedModel')]] = None"}, {"name": "compute_loss_func", "val": ": typing.Optional[typing.Callable] = None"}, {"name": "compute_metrics", "val": ": typing.Optional[typing.Callable[[ForwardRef('EvalPrediction')], dict]] = None"}, {"name": "callbacks", "val": ": typing.Optional[list['TrainerCallback']] = None"}, {"name": "optimizers", "val": ": tuple = (None, None)"}, {"name": "preprocess_logits_for_metrics", "val": ": typing.Optional[typing.Callable[[torch.Tensor, torch.Tensor], torch.Tensor]] = None"}]</parameters></docstring>



<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>evaluate</name><anchor>optimum.habana.GaudiSeq2SeqTrainer.evaluate</anchor><source>https://github.com/huggingface/optimum-habana/blob/v1.19.0/optimum/habana/transformers/trainer_seq2seq.py#L142</source><parameters>[{"name": "eval_dataset", "val": ": typing.Optional[torch.utils.data.dataset.Dataset] = None"}, {"name": "ignore_keys", "val": ": typing.Optional[list[str]] = None"}, {"name": "metric_key_prefix", "val": ": str = 'eval'"}, {"name": "**gen_kwargs", "val": ""}]</parameters><paramsdesc>- **eval_dataset** (`Dataset`, *optional*) --
  Pass a dataset if you wish to override `self.eval_dataset`. If it is an [Dataset](https://huggingface.co/docs/datasets/v4.3.0/en/package_reference/main_classes#datasets.Dataset), columns
  not accepted by the `model.forward()` method are automatically removed. It must implement the `__len__`
  method.
- **ignore_keys** (`List[str]`, *optional*) --
  A list of keys in the output of your model (if it is a dictionary) that should be ignored when
  gathering predictions.
- **metric_key_prefix** (`str`, *optional*, defaults to `"eval"`) --
  An optional prefix to be used as the metrics key prefix. For example the metrics "bleu" will be named
  "eval_bleu" if the prefix is `"eval"` (default)
- **max_length** (`int`, *optional*) --
  The maximum target length to use when predicting with the generate method.
- **num_beams** (`int`, *optional*) --
  Number of beams for beam search that will be used when predicting with the generate method. 1 means no
  beam search.
- **gen_kwargs** --
  Additional `generate` specific kwargs.</paramsdesc><paramgroups>0</paramgroups><retdesc>A dictionary containing the evaluation loss and the potential metrics computed from the predictions. The
dictionary also contains the epoch number which comes from the training state.</retdesc></docstring>

Run evaluation and returns metrics.
The calling script will be responsible for providing a method to compute metrics, as they are task-dependent
(pass it to the init `compute_metrics` argument).
You can also subclass and override this method to inject custom behavior.





</div>
<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>predict</name><anchor>optimum.habana.GaudiSeq2SeqTrainer.predict</anchor><source>https://github.com/huggingface/optimum-habana/blob/v1.19.0/optimum/habana/transformers/trainer_seq2seq.py#L194</source><parameters>[{"name": "test_dataset", "val": ": Dataset"}, {"name": "ignore_keys", "val": ": typing.Optional[list[str]] = None"}, {"name": "metric_key_prefix", "val": ": str = 'test'"}, {"name": "**gen_kwargs", "val": ""}]</parameters><paramsdesc>- **test_dataset** (`Dataset`) --
  Dataset to run the predictions on. If it is a [Dataset](https://huggingface.co/docs/datasets/v4.3.0/en/package_reference/main_classes#datasets.Dataset), columns not accepted by the
  `model.forward()` method are automatically removed. Has to implement the method `__len__`
- **ignore_keys** (`List[str]`, *optional*) --
  A list of keys in the output of your model (if it is a dictionary) that should be ignored when
  gathering predictions.
- **metric_key_prefix** (`str`, *optional*, defaults to `"eval"`) --
  An optional prefix to be used as the metrics key prefix. For example the metrics "bleu" will be named
  "eval_bleu" if the prefix is `"eval"` (default)
- **max_length** (`int`, *optional*) --
  The maximum target length to use when predicting with the generate method.
- **num_beams** (`int`, *optional*) --
  Number of beams for beam search that will be used when predicting with the generate method. 1 means no
  beam search.
- **gen_kwargs** --
  Additional `generate` specific kwargs.</paramsdesc><paramgroups>0</paramgroups></docstring>

Run prediction and returns predictions and potential metrics.
Depending on the dataset and your use case, your test dataset may contain labels. In that case, this method
will also return metrics, like in `evaluate()`.


<Tip>
If your predictions or labels have different sequence lengths (for instance because you're doing dynamic
padding in a token classification task) the predictions will be padded (on the right) to allow for
concatenation into one array. The padding index is -100.
</Tip>
Returns: *NamedTuple* A namedtuple with the following keys:
- predictions (`np.ndarray`): The predictions on `test_dataset`.
- label_ids (`np.ndarray`, *optional*): The labels (if the dataset contained some).
- metrics (`Dict[str, float]`, *optional*): The potential dictionary of metrics (if the dataset contained
  labels).


</div></div>

## GaudiTrainingArguments[[optimum.habana.GaudiTrainingArguments]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class optimum.habana.GaudiTrainingArguments</name><anchor>optimum.habana.GaudiTrainingArguments</anchor><source>https://github.com/huggingface/optimum-habana/blob/v1.19.0/optimum/habana/transformers/training_args.py#L86</source><parameters>[{"name": "output_dir", "val": ": typing.Optional[str] = None"}, {"name": "overwrite_output_dir", "val": ": bool = False"}, {"name": "do_train", "val": ": bool = False"}, {"name": "do_eval", "val": ": bool = False"}, {"name": "do_predict", "val": ": bool = False"}, {"name": "eval_strategy", "val": ": typing.Union[transformers.trainer_utils.IntervalStrategy, str] = 'no'"}, {"name": "prediction_loss_only", "val": ": bool = False"}, {"name": "per_device_train_batch_size", "val": ": int = 8"}, {"name": "per_device_eval_batch_size", "val": ": int = 8"}, {"name": "per_gpu_train_batch_size", "val": ": typing.Optional[int] = None"}, {"name": "per_gpu_eval_batch_size", "val": ": typing.Optional[int] = None"}, {"name": "gradient_accumulation_steps", "val": ": int = 1"}, {"name": "eval_accumulation_steps", "val": ": typing.Optional[int] = None"}, {"name": "eval_delay", "val": ": typing.Optional[float] = 0"}, {"name": "torch_empty_cache_steps", "val": ": typing.Optional[int] = None"}, {"name": "learning_rate", "val": ": float = 5e-05"}, {"name": "weight_decay", "val": ": float = 0.0"}, {"name": "adam_beta1", "val": ": float = 0.9"}, {"name": "adam_beta2", "val": ": float = 0.999"}, {"name": "adam_epsilon", "val": ": typing.Optional[float] = 1e-06"}, {"name": "max_grad_norm", "val": ": float = 1.0"}, {"name": "num_train_epochs", "val": ": float = 3.0"}, {"name": "max_steps", "val": ": int = -1"}, {"name": "lr_scheduler_type", "val": ": typing.Union[transformers.trainer_utils.SchedulerType, str] = 'linear'"}, {"name": "lr_scheduler_kwargs", "val": ": typing.Union[dict[str, typing.Any], str, NoneType] = <factory>"}, {"name": "warmup_ratio", "val": ": float = 0.0"}, {"name": "warmup_steps", "val": ": int = 0"}, {"name": "log_level", "val": ": str = 'passive'"}, {"name": "log_level_replica", "val": ": str = 'warning'"}, {"name": "log_on_each_node", "val": ": bool = True"}, {"name": "logging_dir", "val": ": typing.Optional[str] = None"}, {"name": "logging_strategy", "val": ": typing.Union[transformers.trainer_utils.IntervalStrategy, str] = 'steps'"}, {"name": "logging_first_step", "val": ": bool = False"}, {"name": "logging_steps", "val": ": float = 500"}, {"name": "logging_nan_inf_filter", "val": ": typing.Optional[bool] = False"}, {"name": "save_strategy", "val": ": typing.Union[transformers.trainer_utils.SaveStrategy, str] = 'steps'"}, {"name": "save_steps", "val": ": float = 500"}, {"name": "save_total_limit", "val": ": typing.Optional[int] = None"}, {"name": "save_safetensors", "val": ": typing.Optional[bool] = True"}, {"name": "save_on_each_node", "val": ": bool = False"}, {"name": "save_only_model", "val": ": bool = False"}, {"name": "restore_callback_states_from_checkpoint", "val": ": bool = False"}, {"name": "no_cuda", "val": ": bool = False"}, {"name": "use_cpu", "val": ": bool = False"}, {"name": "use_mps_device", "val": ": bool = False"}, {"name": "seed", "val": ": int = 42"}, {"name": "data_seed", "val": ": typing.Optional[int] = None"}, {"name": "jit_mode_eval", "val": ": bool = False"}, {"name": "use_ipex", "val": ": bool = False"}, {"name": "bf16", "val": ": bool = False"}, {"name": "fp16", "val": ": bool = False"}, {"name": "fp16_opt_level", "val": ": str = 'O1'"}, {"name": "half_precision_backend", "val": ": str = 'hpu_amp'"}, {"name": "bf16_full_eval", "val": ": bool = False"}, {"name": "fp16_full_eval", "val": ": bool = False"}, {"name": "tf32", "val": ": typing.Optional[bool] = None"}, {"name": "local_rank", "val": ": int = -1"}, {"name": "ddp_backend", "val": ": typing.Optional[str] = None"}, {"name": "tpu_num_cores", "val": ": typing.Optional[int] = None"}, {"name": "tpu_metrics_debug", "val": ": bool = False"}, {"name": "debug", "val": ": typing.Union[str, list[transformers.debug_utils.DebugOption]] = ''"}, {"name": "dataloader_drop_last", "val": ": bool = False"}, {"name": "eval_steps", "val": ": typing.Optional[float] = None"}, {"name": "dataloader_num_workers", "val": ": int = 0"}, {"name": "dataloader_prefetch_factor", "val": ": typing.Optional[int] = None"}, {"name": "past_index", "val": ": int = -1"}, {"name": "run_name", "val": ": typing.Optional[str] = None"}, {"name": "disable_tqdm", "val": ": typing.Optional[bool] = None"}, {"name": "remove_unused_columns", "val": ": typing.Optional[bool] = True"}, {"name": "label_names", "val": ": typing.Optional[list[str]] = None"}, {"name": "load_best_model_at_end", "val": ": typing.Optional[bool] = False"}, {"name": "metric_for_best_model", "val": ": typing.Optional[str] = None"}, {"name": "greater_is_better", "val": ": typing.Optional[bool] = None"}, {"name": "ignore_data_skip", "val": ": bool = False"}, {"name": "fsdp", "val": ": typing.Union[list[transformers.trainer_utils.FSDPOption], str, NoneType] = ''"}, {"name": "fsdp_min_num_params", "val": ": int = 0"}, {"name": "fsdp_config", "val": ": typing.Union[dict[str, typing.Any], str, NoneType] = None"}, {"name": "fsdp_transformer_layer_cls_to_wrap", "val": ": typing.Optional[str] = None"}, {"name": "accelerator_config", "val": ": typing.Union[dict, str, NoneType] = None"}, {"name": "deepspeed", "val": ": typing.Union[dict, str, NoneType] = None"}, {"name": "label_smoothing_factor", "val": ": float = 0.0"}, {"name": "optim", "val": ": typing.Union[transformers.training_args.OptimizerNames, str, NoneType] = 'adamw_torch'"}, {"name": "optim_args", "val": ": typing.Optional[str] = None"}, {"name": "adafactor", "val": ": bool = False"}, {"name": "group_by_length", "val": ": bool = False"}, {"name": "length_column_name", "val": ": typing.Optional[str] = 'length'"}, {"name": "report_to", "val": ": typing.Union[NoneType, str, list[str]] = None"}, {"name": "ddp_find_unused_parameters", "val": ": typing.Optional[bool] = False"}, {"name": "ddp_bucket_cap_mb", "val": ": typing.Optional[int] = 230"}, {"name": "ddp_broadcast_buffers", "val": ": typing.Optional[bool] = None"}, {"name": "dataloader_pin_memory", "val": ": bool = True"}, {"name": "dataloader_persistent_workers", "val": ": bool = False"}, {"name": "skip_memory_metrics", "val": ": bool = True"}, {"name": "use_legacy_prediction_loop", "val": ": bool = False"}, {"name": "push_to_hub", "val": ": bool = False"}, {"name": "resume_from_checkpoint", "val": ": typing.Optional[str] = None"}, {"name": "hub_model_id", "val": ": typing.Optional[str] = None"}, {"name": "hub_strategy", "val": ": typing.Union[transformers.trainer_utils.HubStrategy, str] = 'every_save'"}, {"name": "hub_token", "val": ": typing.Optional[str] = None"}, {"name": "hub_private_repo", "val": ": typing.Optional[bool] = None"}, {"name": "hub_always_push", "val": ": bool = False"}, {"name": "hub_revision", "val": ": typing.Optional[str] = None"}, {"name": "gradient_checkpointing", "val": ": bool = False"}, {"name": "gradient_checkpointing_kwargs", "val": ": typing.Union[dict[str, typing.Any], str, NoneType] = None"}, {"name": "include_inputs_for_metrics", "val": ": bool = False"}, {"name": "include_for_metrics", "val": ": list = <factory>"}, {"name": "eval_do_concat_batches", "val": ": bool = True"}, {"name": "fp16_backend", "val": ": str = 'auto'"}, {"name": "push_to_hub_model_id", "val": ": typing.Optional[str] = None"}, {"name": "push_to_hub_organization", "val": ": typing.Optional[str] = None"}, {"name": "push_to_hub_token", "val": ": typing.Optional[str] = None"}, {"name": "mp_parameters", "val": ": str = ''"}, {"name": "auto_find_batch_size", "val": ": bool = False"}, {"name": "full_determinism", "val": ": bool = False"}, {"name": "torchdynamo", "val": ": typing.Optional[str] = None"}, {"name": "ray_scope", "val": ": typing.Optional[str] = 'last'"}, {"name": "ddp_timeout", "val": ": int = 1800"}, {"name": "torch_compile", "val": ": bool = False"}, {"name": "torch_compile_backend", "val": ": typing.Optional[str] = None"}, {"name": "torch_compile_mode", "val": ": typing.Optional[str] = None"}, {"name": "include_tokens_per_second", "val": ": typing.Optional[bool] = False"}, {"name": "include_num_input_tokens_seen", "val": ": typing.Optional[bool] = False"}, {"name": "neftune_noise_alpha", "val": ": typing.Optional[float] = None"}, {"name": "optim_target_modules", "val": ": typing.Union[NoneType, str, list[str]] = None"}, {"name": "batch_eval_metrics", "val": ": bool = False"}, {"name": "eval_on_start", "val": ": bool = False"}, {"name": "use_liger_kernel", "val": ": typing.Optional[bool] = False"}, {"name": "liger_kernel_config", "val": ": typing.Optional[dict[str, bool]] = None"}, {"name": "eval_use_gather_object", "val": ": typing.Optional[bool] = False"}, {"name": "average_tokens_across_devices", "val": ": typing.Optional[bool] = True"}, {"name": "use_habana", "val": ": typing.Optional[bool] = False"}, {"name": "gaudi_config_name", "val": ": typing.Optional[str] = None"}, {"name": "use_lazy_mode", "val": ": typing.Optional[bool] = True"}, {"name": "use_hpu_graphs", "val": ": typing.Optional[bool] = False"}, {"name": "use_hpu_graphs_for_inference", "val": ": typing.Optional[bool] = False"}, {"name": "use_hpu_graphs_for_training", "val": ": typing.Optional[bool] = False"}, {"name": "use_compiled_autograd", "val": ": typing.Optional[bool] = False"}, {"name": "compile_from_sec_iteration", "val": ": typing.Optional[bool] = False"}, {"name": "compile_dynamic", "val": ": typing.Optional[bool] = None"}, {"name": "use_zero3_leaf_promotion", "val": ": typing.Optional[bool] = False"}, {"name": "cache_size_limit", "val": ": typing.Optional[int] = None"}, {"name": "use_regional_compilation", "val": ": typing.Optional[bool] = False"}, {"name": "inline_inbuilt_nn_modules", "val": ": typing.Optional[bool] = None"}, {"name": "allow_unspec_int_on_nn_module", "val": ": typing.Optional[bool] = None"}, {"name": "disable_tensor_cache_hpu_graphs", "val": ": typing.Optional[bool] = False"}, {"name": "max_hpu_graphs", "val": ": typing.Optional[int] = None"}, {"name": "distribution_strategy", "val": ": typing.Optional[str] = 'ddp'"}, {"name": "context_parallel_size", "val": ": typing.Optional[int] = 1"}, {"name": "minimize_memory", "val": ": typing.Optional[bool] = False"}, {"name": "throughput_warmup_steps", "val": ": typing.Optional[int] = 0"}, {"name": "adjust_throughput", "val": ": bool = False"}, {"name": "pipelining_fwd_bwd", "val": ": typing.Optional[bool] = False"}, {"name": "ignore_eos", "val": ": typing.Optional[bool] = True"}, {"name": "non_blocking_data_copy", "val": ": typing.Optional[bool] = False"}, {"name": "profiling_warmup_steps", "val": ": typing.Optional[int] = 0"}, {"name": "profiling_steps", "val": ": typing.Optional[int] = 0"}, {"name": "profiling_warmup_steps_eval", "val": ": typing.Optional[int] = 0"}, {"name": "profiling_steps_eval", "val": ": typing.Optional[int] = 0"}, {"name": "profiling_record_shapes", "val": ": typing.Optional[bool] = True"}, {"name": "profiling_with_stack", "val": ": typing.Optional[bool] = False"}, {"name": "attn_implementation", "val": ": typing.Optional[str] = 'eager'"}, {"name": "flash_attention_recompute", "val": ": bool = False"}, {"name": "flash_attention_fast_softmax", "val": ": bool = False"}, {"name": "flash_attention_causal_mask", "val": ": bool = False"}, {"name": "flash_attention_fp8", "val": ": bool = False"}, {"name": "sdp_on_bf16", "val": ": bool = False"}, {"name": "fp8", "val": ": typing.Optional[bool] = False"}]</parameters><paramsdesc>- **use_habana** (`bool`, *optional*, defaults to `False`) --
  Whether to use Habana's HPU for running the model.
- **gaudi_config_name** (`str`, *optional*) --
  Pretrained Gaudi config name or path.
- **use_lazy_mode** (`bool`, *optional*, defaults to `True`) --
  Whether to use lazy mode for running the model.
- **use_hpu_graphs** (`bool`, *optional*, defaults to `False`) --
  Deprecated, use `use_hpu_graphs_for_inference` instead. Whether to use HPU graphs for performing inference.
- **use_hpu_graphs_for_inference** (`bool`, *optional*, defaults to `False`) --
  Whether to use HPU graphs for performing inference. It will speed up latency but may not be compatible with some operations.
- **use_hpu_graphs_for_training** (`bool`, *optional*, defaults to `False`) --
  Whether to use HPU graphs for performing inference. It will speed up training but may not be compatible with some operations.
- **use_compiled_autograd** (`bool`, *optional*, defaults to `False`) --
  Whether to use compiled autograd for training. Currently only for summarization models.
- **compile_from_sec_iteration** (`bool`, *optional*, defaults to `False`) --
  Whether to torch.compile from the second training iteration.
- **compile_dynamic** (`bool|None`, *optional*, defaults to `None`) --
  Set value of 'dynamic' parameter for torch.compile.
- **use_regional_compilation** (`bool`, *optional*, defaults to `False`) --
  Whether to use regional compile with deepspeed
- **inline_inbuilt_nn_modules** (`bool`, *optional*, defaults to `None`) --
  Set value of 'inline_inbuilt_nn_modules' parameter for torch._dynamo.config. Currently, disabling this parameter improves the performance of the ALBERT model.
- **cache_size_limit(`int`,** *optional*, defaults to 'None') --
  Set value of 'cache_size_limit' parameter for torch._dynamo.config
- **allow_unspec_int_on_nn_module** (`bool`, *optional*, defaults to `None`) --
  Set value of 'allow_unspec_int_on_nn_module' parameter for torch._dynamo.config.
- **disable_tensor_cache_hpu_graphs** (`bool`, *optional*, defaults to `False`) --
  Whether to disable tensor cache when using hpu graphs. If True, tensors won't be cached in hpu graph and memory can be saved.
- **max_hpu_graphs** (`int`, *optional*) --
  Maximum number of hpu graphs to be cached. Reduce to save device memory.
- **distribution_strategy** (`str`, *optional*, defaults to `ddp`) --
  Determines how data parallel distributed training is achieved. May be: `ddp` or `fast_ddp`.
- **throughput_warmup_steps** (`int`, *optional*, defaults to 0) --
  Number of steps to ignore for throughput calculation. For example, with `throughput_warmup_steps=N`,
  the first N steps will not be considered in the calculation of the throughput. This is especially
  useful in lazy mode where the first two or three iterations typically take longer.
- **adjust_throughput** ('bool', *optional*, defaults to `False`) --
  Whether to remove the time taken for logging, evaluating and saving from throughput calculation.
- **pipelining_fwd_bwd** (`bool`, *optional*, defaults to `False`) --
  Whether to add an additional `mark_step` between forward and backward for pipelining
  host backward building and HPU forward computing.
- **non_blocking_data_copy** (`bool`, *optional*, defaults to `False`) --
  Whether to enable async data copy when preparing inputs.
- **profiling_warmup_steps** (`int`, *optional*, defaults to 0) --
  Number of training steps to ignore for profiling.
- **profiling_steps** (`int`, *optional*, defaults to 0) --
  Number of training steps to be captured when enabling profiling.
- **profiling_warmup_steps_eval** (`int`, *optional*, defaults to 0) --
  Number of eval steps to ignore for profiling.
- **profiling_steps_eval** (`int`, *optional*, defaults to 0) --
  Number of eval steps to be captured when enabling profiling.</paramsdesc><paramgroups>0</paramgroups></docstring>

GaudiTrainingArguments is built on top of the Tranformers' [TrainingArguments](https://huggingface.co/docs/transformers/main_classes/trainer#transformers.TrainingArguments)
to enable deployment on Habana's Gaudi.




</div>

## GaudiSeq2SeqTrainingArguments[[optimum.habana.GaudiSeq2SeqTrainingArguments]]

<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>class optimum.habana.GaudiSeq2SeqTrainingArguments</name><anchor>optimum.habana.GaudiSeq2SeqTrainingArguments</anchor><source>https://github.com/huggingface/optimum-habana/blob/v1.19.0/optimum/habana/transformers/training_args_seq2seq.py#L30</source><parameters>[{"name": "output_dir", "val": ": typing.Optional[str] = None"}, {"name": "overwrite_output_dir", "val": ": bool = False"}, {"name": "do_train", "val": ": bool = False"}, {"name": "do_eval", "val": ": bool = False"}, {"name": "do_predict", "val": ": bool = False"}, {"name": "eval_strategy", "val": ": typing.Union[transformers.trainer_utils.IntervalStrategy, str] = 'no'"}, {"name": "prediction_loss_only", "val": ": bool = False"}, {"name": "per_device_train_batch_size", "val": ": int = 8"}, {"name": "per_device_eval_batch_size", "val": ": int = 8"}, {"name": "per_gpu_train_batch_size", "val": ": typing.Optional[int] = None"}, {"name": "per_gpu_eval_batch_size", "val": ": typing.Optional[int] = None"}, {"name": "gradient_accumulation_steps", "val": ": int = 1"}, {"name": "eval_accumulation_steps", "val": ": typing.Optional[int] = None"}, {"name": "eval_delay", "val": ": typing.Optional[float] = 0"}, {"name": "torch_empty_cache_steps", "val": ": typing.Optional[int] = None"}, {"name": "learning_rate", "val": ": float = 5e-05"}, {"name": "weight_decay", "val": ": float = 0.0"}, {"name": "adam_beta1", "val": ": float = 0.9"}, {"name": "adam_beta2", "val": ": float = 0.999"}, {"name": "adam_epsilon", "val": ": typing.Optional[float] = 1e-06"}, {"name": "max_grad_norm", "val": ": float = 1.0"}, {"name": "num_train_epochs", "val": ": float = 3.0"}, {"name": "max_steps", "val": ": int = -1"}, {"name": "lr_scheduler_type", "val": ": typing.Union[transformers.trainer_utils.SchedulerType, str] = 'linear'"}, {"name": "lr_scheduler_kwargs", "val": ": typing.Union[dict[str, typing.Any], str, NoneType] = <factory>"}, {"name": "warmup_ratio", "val": ": float = 0.0"}, {"name": "warmup_steps", "val": ": int = 0"}, {"name": "log_level", "val": ": str = 'passive'"}, {"name": "log_level_replica", "val": ": str = 'warning'"}, {"name": "log_on_each_node", "val": ": bool = True"}, {"name": "logging_dir", "val": ": typing.Optional[str] = None"}, {"name": "logging_strategy", "val": ": typing.Union[transformers.trainer_utils.IntervalStrategy, str] = 'steps'"}, {"name": "logging_first_step", "val": ": bool = False"}, {"name": "logging_steps", "val": ": float = 500"}, {"name": "logging_nan_inf_filter", "val": ": typing.Optional[bool] = False"}, {"name": "save_strategy", "val": ": typing.Union[transformers.trainer_utils.SaveStrategy, str] = 'steps'"}, {"name": "save_steps", "val": ": float = 500"}, {"name": "save_total_limit", "val": ": typing.Optional[int] = None"}, {"name": "save_safetensors", "val": ": typing.Optional[bool] = True"}, {"name": "save_on_each_node", "val": ": bool = False"}, {"name": "save_only_model", "val": ": bool = False"}, {"name": "restore_callback_states_from_checkpoint", "val": ": bool = False"}, {"name": "no_cuda", "val": ": bool = False"}, {"name": "use_cpu", "val": ": bool = False"}, {"name": "use_mps_device", "val": ": bool = False"}, {"name": "seed", "val": ": int = 42"}, {"name": "data_seed", "val": ": typing.Optional[int] = None"}, {"name": "jit_mode_eval", "val": ": bool = False"}, {"name": "use_ipex", "val": ": bool = False"}, {"name": "bf16", "val": ": bool = False"}, {"name": "fp16", "val": ": bool = False"}, {"name": "fp16_opt_level", "val": ": str = 'O1'"}, {"name": "half_precision_backend", "val": ": str = 'hpu_amp'"}, {"name": "bf16_full_eval", "val": ": bool = False"}, {"name": "fp16_full_eval", "val": ": bool = False"}, {"name": "tf32", "val": ": typing.Optional[bool] = None"}, {"name": "local_rank", "val": ": int = -1"}, {"name": "ddp_backend", "val": ": typing.Optional[str] = None"}, {"name": "tpu_num_cores", "val": ": typing.Optional[int] = None"}, {"name": "tpu_metrics_debug", "val": ": bool = False"}, {"name": "debug", "val": ": typing.Union[str, list[transformers.debug_utils.DebugOption]] = ''"}, {"name": "dataloader_drop_last", "val": ": bool = False"}, {"name": "eval_steps", "val": ": typing.Optional[float] = None"}, {"name": "dataloader_num_workers", "val": ": int = 0"}, {"name": "dataloader_prefetch_factor", "val": ": typing.Optional[int] = None"}, {"name": "past_index", "val": ": int = -1"}, {"name": "run_name", "val": ": typing.Optional[str] = None"}, {"name": "disable_tqdm", "val": ": typing.Optional[bool] = None"}, {"name": "remove_unused_columns", "val": ": typing.Optional[bool] = True"}, {"name": "label_names", "val": ": typing.Optional[list[str]] = None"}, {"name": "load_best_model_at_end", "val": ": typing.Optional[bool] = False"}, {"name": "metric_for_best_model", "val": ": typing.Optional[str] = None"}, {"name": "greater_is_better", "val": ": typing.Optional[bool] = None"}, {"name": "ignore_data_skip", "val": ": bool = False"}, {"name": "fsdp", "val": ": typing.Union[list[transformers.trainer_utils.FSDPOption], str, NoneType] = ''"}, {"name": "fsdp_min_num_params", "val": ": int = 0"}, {"name": "fsdp_config", "val": ": typing.Union[dict[str, typing.Any], str, NoneType] = None"}, {"name": "fsdp_transformer_layer_cls_to_wrap", "val": ": typing.Optional[str] = None"}, {"name": "accelerator_config", "val": ": typing.Union[dict, str, NoneType] = None"}, {"name": "deepspeed", "val": ": typing.Union[dict, str, NoneType] = None"}, {"name": "label_smoothing_factor", "val": ": float = 0.0"}, {"name": "optim", "val": ": typing.Union[transformers.training_args.OptimizerNames, str, NoneType] = 'adamw_torch'"}, {"name": "optim_args", "val": ": typing.Optional[str] = None"}, {"name": "adafactor", "val": ": bool = False"}, {"name": "group_by_length", "val": ": bool = False"}, {"name": "length_column_name", "val": ": typing.Optional[str] = 'length'"}, {"name": "report_to", "val": ": typing.Union[NoneType, str, list[str]] = None"}, {"name": "ddp_find_unused_parameters", "val": ": typing.Optional[bool] = False"}, {"name": "ddp_bucket_cap_mb", "val": ": typing.Optional[int] = 230"}, {"name": "ddp_broadcast_buffers", "val": ": typing.Optional[bool] = None"}, {"name": "dataloader_pin_memory", "val": ": bool = True"}, {"name": "dataloader_persistent_workers", "val": ": bool = False"}, {"name": "skip_memory_metrics", "val": ": bool = True"}, {"name": "use_legacy_prediction_loop", "val": ": bool = False"}, {"name": "push_to_hub", "val": ": bool = False"}, {"name": "resume_from_checkpoint", "val": ": typing.Optional[str] = None"}, {"name": "hub_model_id", "val": ": typing.Optional[str] = None"}, {"name": "hub_strategy", "val": ": typing.Union[transformers.trainer_utils.HubStrategy, str] = 'every_save'"}, {"name": "hub_token", "val": ": typing.Optional[str] = None"}, {"name": "hub_private_repo", "val": ": typing.Optional[bool] = None"}, {"name": "hub_always_push", "val": ": bool = False"}, {"name": "hub_revision", "val": ": typing.Optional[str] = None"}, {"name": "gradient_checkpointing", "val": ": bool = False"}, {"name": "gradient_checkpointing_kwargs", "val": ": typing.Union[dict[str, typing.Any], str, NoneType] = None"}, {"name": "include_inputs_for_metrics", "val": ": bool = False"}, {"name": "include_for_metrics", "val": ": list = <factory>"}, {"name": "eval_do_concat_batches", "val": ": bool = True"}, {"name": "fp16_backend", "val": ": str = 'auto'"}, {"name": "push_to_hub_model_id", "val": ": typing.Optional[str] = None"}, {"name": "push_to_hub_organization", "val": ": typing.Optional[str] = None"}, {"name": "push_to_hub_token", "val": ": typing.Optional[str] = None"}, {"name": "mp_parameters", "val": ": str = ''"}, {"name": "auto_find_batch_size", "val": ": bool = False"}, {"name": "full_determinism", "val": ": bool = False"}, {"name": "torchdynamo", "val": ": typing.Optional[str] = None"}, {"name": "ray_scope", "val": ": typing.Optional[str] = 'last'"}, {"name": "ddp_timeout", "val": ": int = 1800"}, {"name": "torch_compile", "val": ": bool = False"}, {"name": "torch_compile_backend", "val": ": typing.Optional[str] = None"}, {"name": "torch_compile_mode", "val": ": typing.Optional[str] = None"}, {"name": "include_tokens_per_second", "val": ": typing.Optional[bool] = False"}, {"name": "include_num_input_tokens_seen", "val": ": typing.Optional[bool] = False"}, {"name": "neftune_noise_alpha", "val": ": typing.Optional[float] = None"}, {"name": "optim_target_modules", "val": ": typing.Union[NoneType, str, list[str]] = None"}, {"name": "batch_eval_metrics", "val": ": bool = False"}, {"name": "eval_on_start", "val": ": bool = False"}, {"name": "use_liger_kernel", "val": ": typing.Optional[bool] = False"}, {"name": "liger_kernel_config", "val": ": typing.Optional[dict[str, bool]] = None"}, {"name": "eval_use_gather_object", "val": ": typing.Optional[bool] = False"}, {"name": "average_tokens_across_devices", "val": ": typing.Optional[bool] = True"}, {"name": "use_habana", "val": ": typing.Optional[bool] = False"}, {"name": "gaudi_config_name", "val": ": typing.Optional[str] = None"}, {"name": "use_lazy_mode", "val": ": typing.Optional[bool] = True"}, {"name": "use_hpu_graphs", "val": ": typing.Optional[bool] = False"}, {"name": "use_hpu_graphs_for_inference", "val": ": typing.Optional[bool] = False"}, {"name": "use_hpu_graphs_for_training", "val": ": typing.Optional[bool] = False"}, {"name": "use_compiled_autograd", "val": ": typing.Optional[bool] = False"}, {"name": "compile_from_sec_iteration", "val": ": typing.Optional[bool] = False"}, {"name": "compile_dynamic", "val": ": typing.Optional[bool] = None"}, {"name": "use_zero3_leaf_promotion", "val": ": typing.Optional[bool] = False"}, {"name": "cache_size_limit", "val": ": typing.Optional[int] = None"}, {"name": "use_regional_compilation", "val": ": typing.Optional[bool] = False"}, {"name": "inline_inbuilt_nn_modules", "val": ": typing.Optional[bool] = None"}, {"name": "allow_unspec_int_on_nn_module", "val": ": typing.Optional[bool] = None"}, {"name": "disable_tensor_cache_hpu_graphs", "val": ": typing.Optional[bool] = False"}, {"name": "max_hpu_graphs", "val": ": typing.Optional[int] = None"}, {"name": "distribution_strategy", "val": ": typing.Optional[str] = 'ddp'"}, {"name": "context_parallel_size", "val": ": typing.Optional[int] = 1"}, {"name": "minimize_memory", "val": ": typing.Optional[bool] = False"}, {"name": "throughput_warmup_steps", "val": ": typing.Optional[int] = 0"}, {"name": "adjust_throughput", "val": ": bool = False"}, {"name": "pipelining_fwd_bwd", "val": ": typing.Optional[bool] = False"}, {"name": "ignore_eos", "val": ": typing.Optional[bool] = True"}, {"name": "non_blocking_data_copy", "val": ": typing.Optional[bool] = False"}, {"name": "profiling_warmup_steps", "val": ": typing.Optional[int] = 0"}, {"name": "profiling_steps", "val": ": typing.Optional[int] = 0"}, {"name": "profiling_warmup_steps_eval", "val": ": typing.Optional[int] = 0"}, {"name": "profiling_steps_eval", "val": ": typing.Optional[int] = 0"}, {"name": "profiling_record_shapes", "val": ": typing.Optional[bool] = True"}, {"name": "profiling_with_stack", "val": ": typing.Optional[bool] = False"}, {"name": "attn_implementation", "val": ": typing.Optional[str] = 'eager'"}, {"name": "flash_attention_recompute", "val": ": bool = False"}, {"name": "flash_attention_fast_softmax", "val": ": bool = False"}, {"name": "flash_attention_causal_mask", "val": ": bool = False"}, {"name": "flash_attention_fp8", "val": ": bool = False"}, {"name": "sdp_on_bf16", "val": ": bool = False"}, {"name": "fp8", "val": ": typing.Optional[bool] = False"}, {"name": "sortish_sampler", "val": ": bool = False"}, {"name": "predict_with_generate", "val": ": bool = False"}, {"name": "generation_max_length", "val": ": typing.Optional[int] = None"}, {"name": "generation_num_beams", "val": ": typing.Optional[int] = None"}, {"name": "generation_config", "val": ": typing.Union[str, pathlib.Path, optimum.habana.transformers.generation.configuration_utils.GaudiGenerationConfig, NoneType] = None"}]</parameters><paramsdesc>- **predict_with_generate** (`bool`, *optional*, defaults to `False`) --
  Whether to use generate to calculate generative metrics (ROUGE, BLEU).
- **generation_max_length** (`int`, *optional*) --
  The `max_length` to use on each evaluation loop when `predict_with_generate=True`. Will default to the
  `max_length` value of the model configuration.
- **generation_num_beams** (`int`, *optional*) --
  The `num_beams` to use on each evaluation loop when `predict_with_generate=True`. Will default to the
  `num_beams` value of the model configuration.
- **generation_config** (`str` or `Path` or `transformers.generation.GenerationConfig`, *optional*) --
  Allows to load a `transformers.generation.GenerationConfig` from the `from_pretrained` method. This can be either:

  - a string, the *model id* of a pretrained model configuration hosted inside a model repo on
    huggingface.co.
  - a path to a *directory* containing a configuration file saved using the
    [transformers.GenerationConfig.save_pretrained](https://huggingface.co/docs/transformers/v4.57.1/en/main_classes/text_generation#transformers.GenerationConfig.save_pretrained) method, e.g., `./my_model_directory/`.
  - a `transformers.generation.GenerationConfig` object.</paramsdesc><paramgroups>0</paramgroups></docstring>

GaudiSeq2SeqTrainingArguments is built on top of the Tranformers' [Seq2SeqTrainingArguments](https://huggingface.co/docs/transformers/main_classes/trainer#transformers.Seq2SeqTrainingArguments)
to enable deployment on Habana's Gaudi.





<div class="docstring border-l-2 border-t-2 pl-4 pt-3.5 border-gray-100 rounded-tl-xl mb-6 mt-8">


<docstring><name>to_dict</name><anchor>optimum.habana.GaudiSeq2SeqTrainingArguments.to_dict</anchor><source>https://github.com/huggingface/optimum-habana/blob/v1.19.0/optimum/habana/transformers/training_args_seq2seq.py#L83</source><parameters>[]</parameters></docstring>

Serializes this instance while replace `Enum` by their values and `GaudiGenerationConfig` by dictionaries (for JSON
serialization support). It obfuscates the token values by removing their value.


</div></div>

<EditOnGithub source="https://github.com/huggingface/optimum-habana/blob/main/docs/source/package_reference/trainer.mdx" />